/**
 * @file
 * Behaviors for Path module's administration pages.
 */

(function ($) {

"use strict";

/**
 * Provides summary text for the various "URL alias patterns for specific ..."
 * fieldsets in the form generated by path_patterns_form() at
 * admin/config/urls/path/patterns.
 */
Backdrop.behaviors.pathSpecificPatternFieldsetSummaries = {
  attach: function (context) {
    $(context).find('fieldset.specific-path-patterns').backdropSetSummary(function (element) {
      var summary = '';
      // Assume that no customized patterns have been provided for specific
      // items (so only default patterns are being used for each path type).
      var empty = 0;
      var customized = 0;
      var total = 0;

      // Find all specific pattern fields within each "URL alias patterns for
      // specific ..." fieldset.
      var $specific_patterns = $(element).find('input.specific-pattern');
      // Iterate through each specific pattern field found within the fieldset.
      $specific_patterns.each(function() {
        // If at least one of them has a non-empty value, then flag the entire
        // fieldset as non-empty.
        if ($(this).val().length === 0) {
          empty++;
        }
        else {
          customized++;
        }
      });
      total = empty + customized;

      if (empty === total) {
        summary = Backdrop.t('default pattern used');
      }
      else {
        summary = Backdrop.t('@customized of @total customized', {'@customized': customized, '@total': total});
      }

      return summary;
    });
  }
};

/**
 * Manages checkboxes for Update URL Aliases (bulk-update) page.
 */
Backdrop.behaviors.pathGenerate = {
  attach: function (context) {
    var $form = $(context).find('#path-bulk-update-form').once('path-generate');
    if ($form.length === 0) {
      return;
    }

    // When you select an entity checkbox (like "content"), select the 
    // checkboxes for all the subtypes
    $form.find("input.path-base").on('change', function () {
      var type = $(this).attr('data-path-type');
      $("input.path-type[data-path-type='"+type+"']").prop('checked', this.checked);
    });

    $form.find("input.path-type").on('change', function () {
      var type = $(this).attr('data-path-type');
      var $base = $("input.path-base[data-path-type='"+type+"']");
      // If you uncheck a subtype checkbox, uncheck the entity checkbox.
      if($(this).is(":checked") == false) {
        $base.prop('checked', false);
        $("input.select-all-alias").prop('checked', false);
      }
      // If all subtype checkboxes checked, check the base checkbox.
      else {
        var unchecked = $("input.path-type[data-path-type='"+type+"']:checkbox:not(:checked)");
        if(unchecked.length < 1) {
          $base.prop('checked', true);
        }
      }
    });

    // Add check all checkboxes in the table head row.
    var strings = { 'selectAll': Backdrop.t('Select all rows in this table'), 'selectNone': Backdrop.t('Deselect all rows in this table') };
    $form.find('th.path-alias-generate').wrapInner('<label class="option" for="select-all-alias"></label>').prepend($('<input type="checkbox" id="select-all-alias" class="select-all-alias form-checkbox" />').attr('title', strings.selectAll)).on('click', function (event) {
      if ($(event.target).is('input[type="checkbox"]')) {
        $form.find("input.path-type, input.path-base").each(function () {
          $(this).prop('checked', event.target.checked);
        });
      }
    });

    // Uncheck the "select all" checkboxes if not all checkboxes checked.
    $form.find('input[type="checkbox"]').on('change', function () {
      var uncheckedAll = ($form.find("input.path-type:checkbox:not(:checked)").length)+($("input.path-base:checkbox:not(:checked)").length);
      $form.find("input.select-all-alias").prop('checked', uncheckedAll<1);
    });

  }
};
})(jQuery);
