<?php

/*
 * Copyright (C) 2023 Manuela v.d. Decken <manuela@isteam.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * ImageSizeCalculator
 * uses 3 Levels of limitation the size (i.e. global + modul + folder)
 * @category     Utils
 * @package      images
 * @copyright    Manuela v.d. Decken <manuela@isteam.de>
 * @author       Manuela v.d. Decken <manuela@isteam.de>
 * @license      General Public License 3.0
 * @version      0.0.1 $Rev: $
 * @revision     $Id: $
 * @since        File available since 19.04.2023
 * @deprecated   no / since 0000/00/00
 * @description  xxx
 */
declare(strict_types=1);

namespace App\Utils;

class ImageSizeCalc {

    private string $sSourceFile;     // FQ-Filename for the source image
    private float  $fRatio;          // Width/Height ratio of the original image
    private int    $iOrgWidth;       // size of the original source image
    private int    $iOrgHeight;

    private int    $iBoxWidth;
    private int    $iBoxHeight;

    private int    $iReducedWidth;
    private int    $iReducedHeight;
/**
 * initialise the class
 * width=0 AND height=0 disables the respective level
 * @param int  $iL1MaxWidth     // level_1 limit (system global) mandatory level
 * @param int  $iL1MaxHeight    //
 * @param int  $iL2MaxWidth     // level_2 limit (addon global)
 * @param int  $iL2MaxHeight    //
 * @param bool $bDoNotResize    // true if no resize after level 1 shall be done | default = false
 * @throws RuntimeException
 */
    public function __construct(
        private int  $iL1MaxWidth,
        private int  $iL1MaxHeight,
        private int  $iL2MaxWidth,
        private int  $iL2MaxHeight,
        private bool $bDoNotResize = false
    ) { }

/**
 * Can fit the picture in a box, similar to a passe-partout
 * The size of the box corresponds to the image size settings in Level_3 (the media folder).
 * Smaller images are not enlarged and must be centered.
 * @param string  $sSourceFile
 * @param int     $iWidth        // requested width from folder
 * @param int     $iHeight       // requested height from folder
 * @return bool                  // true if image needs resizing, otherwise show the original
 */
    public function resize(string $sSourceFile, int $iWidth, int $iHeight): bool
    {
        if (!\is_readable($sSourceFile)) {
            throw new RuntimeException('sourcefile not readable');
        }
        $this->sSourceFile = $sSourceFile;
        if (false === ($aSize = \getimagesize($sSourceFile))) {
            throw new RuntimeException('can not examine the sourcefile');
        }
        $this->iOrgWidth     = $aSize[0];
        $this->iOrgHeight    = $aSize[1];
        $this->fRatio        = $aSize[0] / $aSize[1];

        $aCalcSize = $this->calculateTargetsize($iWidth, $iHeight);
        $this->iReducedWidth = $aCalcSize[0];
        $this->iReducedHeight = $aCalcSize[1];
        $this->calcBox($iWidth, $iHeight);
        return ($this->iOrgWidth !== $aCalcSize[0] || $this->iOrgHeight !== $aCalcSize[1]);
    }
/**
 * Width of the display box
 * @return int
 */
    public function getBoxWidth():int
    {
        return $this->iBoxWidth;
    }
/**
 * Height of the display box
 * @return int
 */
    public function getBoxHeight():int
    {
        return $this->iBoxHeight;
    }
/**
 * Width of the resized image
 * @return int
 */
    public function getSrcWidth(): int
    {
        return $this->iReducedWidth;
    }
/**
 * HWidth of the resized image
 * @return int
 */
    public function getSrcHeight(): int
    {
        return $this->iReducedHeight;
    }
/**
 * X-position of the image inside the display box
 * @return int
 */
    public function getDestPosX(): int
    {
        return (int)\round(($this->iBoxWidth - $this->iReducedWidth) / 2);
    }
/**
 * Y-position of the image inside the display box
 * @return int
 */
    public function getDestPosY(): int
    {
        return (int)\round(($this->iBoxHeight - $this->iReducedHeight) / 2);
    }
/**
 * the aspect ratio of the image
 * @return float
 */
    public function getAspectRatio(): float
    {
        return $this->fRatio;
    }
/* ***********  internal functions ****************************** */
/**
 * select the smaller box from Level_1 and Level_2 restrictions
 * @return array  // (int)['width'], (int)['height']
 */
    protected function calculateTargetsize(int $iWidth, int $iHeight): array
    {
        $aLevel1Size  = $this->sanitizeSizeByRatio($this->iL1MaxWidth, $this->iL1MaxHeight);
        $aLevel2Size  = $this->sanitizeSizeByRatio($this->iL2MaxWidth, $this->iL2MaxHeight);
        $aLevel3Size  = $this->sanitizeSizeByRatio($iWidth, $iHeight);
        if ($aLevel1Size['width'] < $aLevel2Size['width']) {
            $iWidth   = $aLevel1Size['width'];
            $iHeight  = $aLevel1Size['height']; // priority 1
        }
        if ($this->bDoNotResize) { // level_1 limitation is mandatory!
            $aRetval = ['width' => $iWidth, 'heigth' => $iHeight];
        } else { // otherwise reduce it to Level_2 and Level_3 limits
            if ($aLevel2Size['width'] < $aLevel3Size['width']) {
                $iWidth   = $aLevel2Size['width'];
                $iHeight  = $aLevel2Size['height'];  // priority 2
            }
            if ($aLevel3Size['width'] < $this->iOrgWidth) {
                $iWidth   = $aLevel3Size['width'];
                $iHeight  = $aLevel3Size['height'];   // priority 3
            } else {
                $iWidth   = $this->iOrgWidth;
                $iHeight  = $this->iOrgHeight;   // priority 3
            }
            $aRetval = [$iWidth, $iHeight];
        }
        return $aRetval;
    }
/**
 * Calculate the virtual dimensions respecting the width/height ratio.
 * Missing dimensions will be added accordingly
 * If both dimensions missing, set to original image size
 * @param int $iWidth   // width
 * @param int $iHeight  // height
 * @return array        // (int)['width'], (int)['height']
 */
    protected function sanitizeSizeByRatio(int $iWidth, int $iHeight): array
    {
        if ($iWidth === 0 && $iHeight > 0) {
            $iWidth = (int)(\round($iHeight * $this->fRatio));
        }elseif ($iWidth > 0 && $iHeight === 0) {
            $iHeight = (int)(\round($iWidth / $this->fRatio));
        }elseif ($iWidth === 0 && $iHeight === 0) {
            $iWidth = $this->iOrgWidth;
            $iHeight = $this->iOrgHeight;
        } else {
            $iHeight = (int)(\round($iWidth / $this->fRatio));
        }
        return ['width' => $iWidth, 'height' => $iHeight];
    }
/**
 * Differences in the aspect ratios of the image and level_3 size
 * are compensated for using a display box. This display box can
 * have a transparent or plain colored background, for example.
 * @param int $iWidth
 * @param int $iHeight
 * @return void
 */
    protected function calcBox(int $iWidth, int $iHeight): void
    {
        $fFaktor = ($iWidth !== 0 && $iHeight !== 0)
                 ? $iWidth / $this->iReducedWidth
                 : 1;
        $this->iBoxWidth  = (int)\round($iWidth / $fFaktor);
        $this->iBoxHeight = (int)\round($iHeight / $fFaktor);
    }
}
