<?php

/*
 * This file is part of the Kipanga package.
 *
 * Copyright (C) 2018 Manuela v.d.Decken <manuela@isteam.de>
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT OR THIS HEADER
 *
 * This program is subject to proprietary license terms. It is not permitted to modify or
 * distribute the program or parts of it.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/**
 * Description of IpAddress
 *
 * @package      Kipanga\Helpers
 * @copyright    Manuela v.d.Decken <manuela@isteam.de>
 * @author       Manuela v.d.Decken <manuela@isteam.de>
 * @license      This program is subject to proprietary license terms.
 * @version      0.0.1
 * @revision     $Id: IpAddress.php 16 2020-04-13 00:15:09Z Manuela $
 * @since        File available since 21.01.2018
 * @deprecated   no / since 0000/00/00
 * @description  several methods to handle IP addresses. IPv4 and IPv6 as well.
 */
declare(strict_types=1);

namespace App\Utils;

// use source;

class IpAddress {
    public static function convertIp(string $sRawIp): string
    {
        $sRetval = '';
        $sV4Address  = self::sanitizeIp($sRawIp);
        $aOctets = \explode('.', $sV4Address);
        $sRetval = \sprintf('%02x', (int)$aOctets[3])
                 . \sprintf('%02x', (int)$aOctets[2]).''
                 . \sprintf('%02x', (int)$aOctets[1])
                 . \sprintf('%02x', (int)$aOctets[0]);
        return $sRetval;
    }

    public static function sanitizeIp(string $sRawIp): string
    {
/*
 *  allowed notations for IP addresses and their results
 *  ----------------------------------------------------
 *  192.168.0.222                             => 192.168.0.222
 *  ::192.168.0.222                           => 0000:0000:0000:0000:0000:0000:c0a8:00de
 *  1111:22:3333:4444:5555:6666:192.168.0.222 => 1111:0022:3333:4444:5555:6666:c0a8:00de
 *  1111:22::6666:192.168.0.222               => 1111:0022:0000:0000:0000:6666:c0a8:00de
 *  1111:22:3333:4444:5555:6666:0:8888        => 1111:0022:3333:4444:5555:6666:0000:8888
 *  1111:22::6666:7777:8888                   => 1111:0022:0000:0000:0000:6666:7777:8888
 *  ----------------------------------------------------
 */
        // clean address from CIDR, port and parts of URLs
        $sPattern = '/^[{\[]?([.:a-f0-9]*)(?:\/[0-1]*)?(?:[\]}]?.*)$/im';
        $sIpAddress = \preg_replace($sPattern, '$1', $sRawIp);
        if (\strpos($sIpAddress, ':') === false) {
        // sanitize IPv4 -------------------------------------------------------------- //
            if (false === \filter_var($sIpAddress, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
                throw new \InvalidArgumentException('illegal IPv4 address given! ['.$sIpAddress.']');
            }
        } else {
        // sanitize IPv6 -------------------------------------------------------------- //
            if (false === \filter_var($sIpAddress, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
                throw new \InvalidArgumentException('illegal IPv6 address given! ['.$sIpAddress.']');
            }
            $sPattern = '/^([:a-f0-9]*?)(?:(?:\:)([0-9]{1,3}(?:\.[0-9]{1,3}){3}))?$/is';
            $aMatches = [];
        // split possible V4 part and convert it into V6
            if (\preg_match($sPattern, $sIpAddress, $aMatches)) {
                $sIpAddress = $aMatches[1].(isset($aMatches[2]) ? ':'.self::convertIpV4ToIpV6($aMatches[2]) : '');
            }
        // expand all to full 128 bit (8*16 bit hex values)
            $sIpAddress = self::expandIpV6($sIpAddress);
        }
        return $sIpAddress;
    }

/**
 * check if IP is IPv4
 * @param string $sRawIp
 * @return bool
 */
    public static function isIpV4(string $sRawIp): bool
    {
        $bRetval = (false !== \filter_var($sRawIp, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4));
        return $bRetval;
    }

/**
 *
 * @param string $sV4Address
 * @return string
 * @description convert a IPv4 address into full size 32bit IPv6 format
 */
    public static function convertIpV4ToIpV6(string $sV4Address): string
    {
        $sRetval = '';
        $aOctets = \explode('.', $sV4Address);
        $sRetval = \sprintf('%02x', (int)$aOctets[0])
                 . \sprintf('%02x', (int)$aOctets[1]).':'
                 . \sprintf('%02x', (int)$aOctets[2])
                 . \sprintf('%02x', (int)$aOctets[3]);
        return $sRetval;
    }

/**
 *
 * @param string $sIpV6Address
 * @return string
 * @description expands IPv6 addresses, shortened by :: to all 8 16 bit words (full 128 bit)
 */
    public static function expandIpV6(string $sIpV6Address): string
    {
        $sIpV6Address = (\str_starts_with($sIpV6Address, '::')
                      ? \substr($sIpV6Address, 1) : $sIpV6Address);
        $aBlocks = \preg_split('/:/', $sIpV6Address);
        $iMissingBlocks = 8 - (\count($aBlocks)-1);
        \array_walk($aBlocks, function(&$sValue) use ($iMissingBlocks) {
            $sValue = ($sValue === '' && $iMissingBlocks > 0)
                    ? \rtrim(\str_repeat('0000:', $iMissingBlocks), ':')
                    : \str_pad($sValue, 4, '0', \STR_PAD_LEFT);
        });
        return \implode(':', $aBlocks);
    }

/**
 *
 * @param string $sAddress the RAW IP address
 * @param int|string $mNetmask ( /24 or 255.255.255.0 notation)
 * @return string    the masked IP address
 */
    public static function getMaskedIpV4(string $sAddress,  $mNetmask): string
    {
        if (\is_string($mNetmask)) {
            if (false === ($iIpMask = \ip2long($mNetmask))) {
                throw new \InvalidArgumentException('illegal IPv4 netmask given! ['.$mNetmask.']');
            }
        } else {
            if ($mNetmask < 0 || $mNetmask > 32) {
                throw new \InvalidArgumentException('illegal IPv4 prefix given! ['.$mNetmask.']');
            }
            $iIpMask = \bindec(\str_pad(\str_repeat('1', $mNetmask), 32, '0', \STR_PAD_RIGHT));
        }
        if (false === ($iIpAddress = \ip2long($sAddress))) {
            throw new \InvalidArgumentException('illegal IPv4 address given! ['.$sAddress.']');
        }
        return \long2ip($iIpAddress & $iIpMask);
    }

/**
 *
 * @param string $sAddress
 * @param int $iPrefixLength
 * @return string
 */
    public static function getMaskedIpV6(string $sAddress, int $iPrefixLength): string
    {
        // build binary netmask from iNetmaskLengthV6
        // and split all 8 parts into an array
        $iPreLen = $iPrefixLength < 0 ? 0 : ($iPrefixLength > 128 ? 128 : $iPrefixLength);
        $aMask = \str_split(
            \str_repeat('1', $iPreLen).
            \str_repeat('0', 128 - $iPreLen),
            16
        );
        $aIpV6 = \preg_split('/:/', $sAddress);
        // iterate all IP parts, apply its mask and reformat to IPv6 string notation.
        \array_walk(
            $aIpV6,
            function(&$sWord, $iIndex) use ($aMask) {
                $sWord = \sprintf('%04x', \hexdec($sWord) & \bindec($aMask[$iIndex]));
            }
        );
        // reformat to IPv6 string notation.
        return \implode(':', $aIpV6);
    }

} // end of class IpAddress
