<?php

/*
 * Copyright (C) 2023 Manuela v.d.Decken <manuela@isteam.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * SqlMode
 * Provides a temporary modification of sql_mode. On destruct the original settings
 * will be restored. Only options listed in ALLOWED_OPTIONS can be modified
 * This class is downgraded to WebsiteBaker 2 series less then 2.14
 *
 * @category     Core
 * @package      Database package
 * @copyright    Manuela v.d.Decken <manuela@isteam.de>
 * @author       Manuela v.d.Decken <manuela@isteam.de>
 * @license      GNU General Public License 3.0+
 * @version      1.0.0 $Rev: 145 $
 * @revision     $Id: SqlMode.php 145 2025-07-20 10:00:06Z Uwe $
 * @since        File available since 31.08.2023
 * @deprecated   no / since 0000/00/00
 */
declare(strict_types=1);

//namespace Db;

class SqlMode {

/**
 * list of changeable options
 * constant array of strings
 */
    private const    ALLOWED_OPTIONS = ['ONLY_FULL_GROUP_BY'];
/**
 * original settings before instantiating
 * @var array of strings
 */
    private array    $aOrigSettings  = [];
/**
 * current settings durung runtime
 * @var array of strings
 */
    private array    $aCurrSettings  = [];
/**
 * active database connection
 * @var DbConnectionInterface
 */
    private ?database $oDb            = null;

/**
 * constructor
 * @param DbConnectionInterface $oDb  active mysqli connection
 */
    public function __construct(database $oDb) {
        $this->oDb = $oDb;
        $sTmp = $this->getOptions();
        $this->aOrigSettings = $this->aCurrSettings = ($sTmp ? \explode(',', $sTmp) : []);
    }

/**
 * restore the original settings on destruct
 */
    public function __destruct() {
        $this->restore();
    }

/**
 * adds an option to the settings
 * @param string $sOption name of the option to add
 * @return bool
 */
    public function add(string $sOption): bool
    {
        $bRetval = false;
        $this->verifyOption($sOption);
        if ($this->aOrigSettings && !\in_array($sOption, $this->aCurrSettings)) {
            if ($this->setOptions(\implode(',', $this->aCurrSettings).','.$sOption)) {
                \array_push($this->aCurrSettings, $sOption);
                $bRetval = true;
            }
        }
        return $bRetval;
    }

/**
 * removes an option from settings
 * @param string $sOption
 * @return bool
 */
    public function remove(string $sOption): bool
    {
        $bRetval = false;
        $this->verifyOption($sOption);
        if ($this->aOrigSettings && \in_array($sOption, $this->aCurrSettings)) {
            $aTmp = \array_diff($this->aCurrSettings, [$sOption]);
            if ($this->setOptions(\implode(',', $aTmp))) {
                $this->aCurrSettings = $aTmp;
                $bRetval = true;
            }
        }
        return $bRetval;
    }

/**
 * restore the original settings
 * @return bool
 */
    public function restore(): bool
    {
        $bRetval = false;
        \sort($this->aOrigSettings, SORT_STRING);
        \sort($this->aCurrSettings, SORT_STRING);
        if ($this->aOrigSettings !== $this->aCurrSettings) {
            $bRetval = $this->setOptions(\implode(',', $this->aOrigSettings));
        }
        return $bRetval;
    }

/**
 * Check if option is allowed and case-sensitive
 * @param string $sOption
 * @return bool
 * @throws \InvalidArgumentException
 */
    protected function verifyOption(string $sOption): bool
    {
        if (!\in_array($sOption, self::ALLOWED_OPTIONS)) {
            throw new \InvalidArgumentException('modify option \''.$sOption.'\' is NOT allowed!!');
        }
        return true;
    }

/**
 * write options to database
 * @param string $sOptions
 * @return bool
 */
    protected function setOptions(string $sOptions): bool
    {
        $sSql = 'SET @@sql_mode = \''.$sOptions.'\'';
        return (bool)$this->oDb->query($sSql);
    }

/**
 * read options from database
 * @return string
 */
    protected function getOptions(): string
    {
        $sSql = 'SELECT @@sql_mode';
        $sRetval = $this->oDb->get_one($sSql);
        return ($sRetval ?? '');
    }
}
