<?php

declare(strict_types=1);

namespace Loupe\Loupe\Internal\Filter;

use Doctrine\Common\Lexer\Token;
use Loupe\Loupe\Exception\FilterFormatException;
use Loupe\Loupe\Internal\Engine;
use Loupe\Loupe\Internal\Filter\Ast\AbstractGroup;
use Loupe\Loupe\Internal\Filter\Ast\Ast;
use Loupe\Loupe\Internal\Filter\Ast\Concatenator;
use Loupe\Loupe\Internal\Filter\Ast\Filter;
use Loupe\Loupe\Internal\Filter\Ast\GeoBoundingBox;
use Loupe\Loupe\Internal\Filter\Ast\GeoDistance;
use Loupe\Loupe\Internal\Filter\Ast\Group;
use Loupe\Loupe\Internal\Filter\Ast\MultiAttributeFilter;
use Loupe\Loupe\Internal\Filter\Ast\Node;
use Loupe\Loupe\Internal\Filter\Ast\Operator;
use Loupe\Loupe\Internal\LoupeTypes;

class Parser
{
    private Ast $ast;

    private \SplStack $groups;

    private Lexer $lexer;

    public function __construct(?Lexer $lexer = null)
    {
        $this->lexer = $lexer ?? new Lexer();
        $this->groups = new \SplStack();
    }

    public function getAst(string $string, Engine $engine): Ast
    {
        $this->lexer->setInput($string);
        $this->ast = new Ast();

        $this->lexer->moveNext();

        $start = true;

        while (true) {
            if (!$this->lexer->lookahead) {
                break;
            }

            $this->lexer->moveNext();

            if ($start && !$this->lexer->token?->isA(
                Lexer::T_ATTRIBUTE_NAME,
                Lexer::T_GEO_RADIUS,
                Lexer::T_GEO_BOUNDING_BOX,
                Lexer::T_OPEN_PARENTHESIS
            )) {
                $this->syntaxError('an attribute name, _geoRadius() or \'(\'');
            }

            $start = false;

            if ($this->lexer->token?->type === Lexer::T_GEO_RADIUS) {
                $this->handleGeoRadius($engine);
                continue;
            }

            if ($this->lexer->token?->type === Lexer::T_GEO_BOUNDING_BOX) {
                $this->handleGeoBoundingBox($engine);
                continue;
            }

            if ($this->lexer->token?->type === Lexer::T_ATTRIBUTE_NAME) {
                $this->handleAttribute($engine);
                continue;
            }

            if ($this->lexer->token?->isA(Lexer::T_AND, Lexer::T_OR)) {
                $this->addNode(Concatenator::fromString((string) $this->lexer->token->value));
            }

            if ($this->lexer->token?->isA(Lexer::T_OPEN_PARENTHESIS)) {
                $this->groups->push(new Group());
            }

            if ($this->lexer->token?->isA(Lexer::T_CLOSE_PARENTHESIS)) {
                $activeGroup = $this->groups->isEmpty() ? null : $this->groups->pop();

                // Close a potentially previously opened multi attribute group too
                if ($activeGroup instanceof MultiAttributeFilter) {
                    $this->addNode($activeGroup);
                    $activeGroup = $this->groups->isEmpty() ? null : $this->groups->pop();
                }

                if ($activeGroup instanceof Group) {
                    $this->addNode($activeGroup);
                } else {
                    $this->syntaxError('an opened group statement');
                }
            }
        }

        $activeGroup = $this->getActiveGroup();

        if ($activeGroup instanceof MultiAttributeFilter) {
            $activeGroup = $this->groups->pop();
            $this->addNode($activeGroup);
        }

        if ($activeGroup !== null && !$activeGroup instanceof MultiAttributeFilter) {
            $this->syntaxError('a closing parenthesis');
        }

        return $this->ast;
    }

    private function addNode(Node $node): self
    {
        // Ignore empty groups
        if ($node instanceof AbstractGroup && $node->isEmpty()) {
            return $this;
        }

        $activeGroup = $this->getActiveGroup();

        if ($activeGroup instanceof AbstractGroup) {
            $activeGroup->addChild($node);
            return $this;
        }

        $this->ast->addNode($node);

        return $this;
    }

    private function assertAndExtractFloat(?Token $token, bool $allowNegative = false): float
    {
        $multipler = 1;
        if ($allowNegative && $token !== null && $token->type === Lexer::T_MINUS) {
            $multipler = -1;
            $this->lexer->moveNext();
            $token = $this->lexer->token;
        }

        $this->assertFloat($token);
        return (float) $this->lexer->token?->value * $multipler;
    }

    private function assertClosingParenthesis(?Token $token): void
    {
        $this->assertTokenTypes($token, [Lexer::T_CLOSE_PARENTHESIS], "')'");
    }

    private function assertComma(?Token $token): void
    {
        $this->assertTokenTypes($token, [Lexer::T_COMMA], "','");
    }

    private function assertFloat(?Token $token): void
    {
        $this->assertTokenTypes($token, [Lexer::T_FLOAT], 'valid float value');
    }

    private function assertOpeningParenthesis(?Token $token): void
    {
        $this->assertTokenTypes($token, [Lexer::T_OPEN_PARENTHESIS], "'('");
    }

    private function assertOperator(?Token $token): void
    {
        $type = $token->type ?? null;

        if (!\is_int($type) || $type < 10 || $type > 30) {
            $this->syntaxError('valid operator', $token);
        }
    }

    private function assertStringOrFloatOrBoolean(?Token $token): void
    {
        $this->assertTokenTypes($token, [
            Lexer::T_FLOAT,
            Lexer::T_STRING,
            Lexer::T_TRUE,
            Lexer::T_FALSE,
        ], 'valid string, float or boolean value');
    }

    /**
     * @param array<int> $types
     */
    private function assertTokenTypes(?Token $token, array $types, string $error): void
    {
        $type = $token->type ?? null;

        if ($type === null || !\in_array($type, $types, true)) {
            $this->syntaxError($error, $token);
        }
    }

    private function getActiveGroup(): null|AbstractGroup
    {
        return $this->groups->isEmpty() ? null : $this->groups->top();
    }

    private function getTokenValueBasedOnType(): float|string|bool
    {
        $value = $this->lexer->token?->value;

        if ($value === null) {
            $this->syntaxError('NULL is not supported, use IS NULL or IS NOT NULL');
        }

        return match ($this->lexer->token?->type) {
            Lexer::T_FLOAT => LoupeTypes::convertToFloat($value),
            Lexer::T_STRING => LoupeTypes::convertToString($value),
            Lexer::T_FALSE => false,
            Lexer::T_TRUE => true,
            default => throw new FilterFormatException('This should never happen, please file a bug report.')
        };
    }

    private function handleAttribute(Engine $engine): void
    {
        $attributeName = (string) $this->lexer->token?->value;

        $this->validateFilterableAttribute($engine, $attributeName);

        if (\in_array($attributeName, $engine->getIndexInfo()->getMultiFilterableAttributes(), true)) {
            $activeGroup = $this->getActiveGroup();

            // Start a new multi attribute filter group if not already opened. Validated it is already here.
            if (!$activeGroup instanceof MultiAttributeFilter) {
                $this->groups->push(new MultiAttributeFilter($attributeName));
            }
        }

        $this->assertOperator($this->lexer->lookahead);
        $this->lexer->moveNext();
        $operator = (string) $this->lexer->token?->value;

        if ($this->lexer->token?->type === Lexer::T_IS) {
            $this->handleIs($attributeName, $engine);
            return;
        }

        // Greater than or smaller than operators
        if ($this->lexer->lookahead?->type === Lexer::T_EQUALS) {
            $this->lexer->moveNext();
            $operator .= $this->lexer->token?->value;
        }

        if ($this->lexer->token?->type === Lexer::T_NOT) {
            if ($this->lexer->lookahead?->type !== Lexer::T_IN) {
                $this->syntaxError('must be followed by IN ()', $this->lexer->lookahead);
            }

            $this->lexer->moveNext();
            $operator .= ' ' . $this->lexer->token?->value;
        }

        if ($this->lexer->token?->type === Lexer::T_IN) {
            $this->handleIn($attributeName, $operator);
            return;
        }

        $this->assertStringOrFloatOrBoolean($this->lexer->lookahead);

        $this->lexer->moveNext();

        $this->addNode(new Filter($attributeName, Operator::fromString($operator), $this->getTokenValueBasedOnType()));
    }

    private function handleGeoBoundingBox(Engine $engine): void
    {
        $startPosition = ($this->lexer->lookahead?->position ?? 0) + 1;

        $this->assertOpeningParenthesis($this->lexer->lookahead);
        $this->lexer->moveNext();
        $this->lexer->moveNext();

        $attributeName = (string) $this->lexer->token?->value;

        $this->validateFilterableAttribute($engine, $attributeName);

        $this->lexer->moveNext();
        $this->lexer->moveNext();
        $north = $this->assertAndExtractFloat($this->lexer->token, true);
        $this->assertComma($this->lexer->lookahead);

        $this->lexer->moveNext();
        $this->lexer->moveNext();
        $east = $this->assertAndExtractFloat($this->lexer->token, true);
        $this->assertComma($this->lexer->lookahead);

        $this->lexer->moveNext();
        $this->lexer->moveNext();
        $south = $this->assertAndExtractFloat($this->lexer->token, true);
        $this->assertComma($this->lexer->lookahead);

        $this->lexer->moveNext();
        $this->lexer->moveNext();
        $west = $this->assertAndExtractFloat($this->lexer->token, true);
        $this->assertClosingParenthesis($this->lexer->lookahead);

        try {
            $this->addNode(new GeoBoundingBox($attributeName, $north, $east, $south, $west));
        } catch (\InvalidArgumentException $e) {
            $this->syntaxError(
                $e->getMessage(),
                // create a fake token to show the user the whole value for better developer experience as we don't know
                // which latitude or longitude value caused the exception
                new Token(implode(', ', [$attributeName, $north, $east, $south, $west]), Lexer::T_FLOAT, $startPosition),
            );
        }

        $this->lexer->moveNext();
    }

    private function handleGeoRadius(Engine $engine): void
    {
        $this->assertOpeningParenthesis($this->lexer->lookahead);
        $this->lexer->moveNext();
        $this->lexer->moveNext();

        $attributeName = (string) $this->lexer->token?->value;

        $this->validateFilterableAttribute($engine, $attributeName);

        $this->lexer->moveNext();
        $this->lexer->moveNext();
        $lat = $this->assertAndExtractFloat($this->lexer->token, true);
        $this->assertComma($this->lexer->lookahead);
        $this->lexer->moveNext();
        $this->lexer->moveNext();
        $lng = $this->assertAndExtractFloat($this->lexer->token, true);
        $this->assertComma($this->lexer->lookahead);
        $this->lexer->moveNext();
        $this->assertFloat($this->lexer->lookahead);
        $this->lexer->moveNext();
        $distance = (float) $this->lexer->token?->value;
        $this->assertClosingParenthesis($this->lexer->lookahead);

        $this->addNode(new GeoDistance($attributeName, $lat, $lng, $distance));

        $this->lexer->moveNext();
    }

    private function handleIn(string $attributeName, string $operator): void
    {
        $this->assertOpeningParenthesis($this->lexer->lookahead);
        $this->lexer->moveNext();
        $this->lexer->moveNext();

        $values = [];

        while (true) {
            $this->assertStringOrFloatOrBoolean($this->lexer->token);
            $values[] = $this->getTokenValueBasedOnType();

            if ($this->lexer->lookahead === null) {
                $this->assertClosingParenthesis($this->lexer->token);
            }

            if ($this->lexer->lookahead?->type === Lexer::T_CLOSE_PARENTHESIS) {
                $this->lexer->moveNext();
                break;
            }

            $this->assertComma($this->lexer->lookahead);
            $this->lexer->moveNext();
            $this->lexer->moveNext();
        }

        $this->addNode(new Filter($attributeName, Operator::fromString($operator), $values));
    }

    private function handleIs(mixed $attributeName, Engine $engine): void
    {
        if ($this->lexer->lookahead?->type === Lexer::T_NULL) {
            $this->addNode(new Filter($attributeName, Operator::Equals, LoupeTypes::VALUE_NULL));
            return;
        }

        if ($this->lexer->lookahead?->type === Lexer::T_EMPTY) {
            $this->addNode(new Filter($attributeName, Operator::Equals, LoupeTypes::VALUE_EMPTY));
            return;
        }

        if ($this->lexer->lookahead?->type === Lexer::T_NOT && $this->lexer->glimpse()?->type === Lexer::T_NULL) {
            $this->addNode(new Filter($attributeName, Operator::NotEquals, LoupeTypes::VALUE_NULL));
            return;
        }

        if ($this->lexer->lookahead?->type === Lexer::T_NOT && $this->lexer->glimpse()?->type === Lexer::T_EMPTY) {
            $this->addNode(new Filter($attributeName, Operator::NotEquals, LoupeTypes::VALUE_EMPTY));
            return;
        }

        $this->syntaxError('"NULL", "NOT NULL", "EMPTY" or "NOT EMPTY" after is', $this->lexer->lookahead);
    }

    private function syntaxError(string $expected = '', ?Token $token = null): void
    {
        if ($token === null) {
            $token = $this->lexer->token;
        }

        $tokenPos = $token->position ?? '-1';

        $message = sprintf('Col %d: Error: ', $tokenPos);
        $message .= $expected !== '' ? sprintf('Expected %s, got ', $expected) : 'Unexpected ';
        $message .= $this->lexer->lookahead === null ? 'end of string.' : sprintf("'%s'", $token?->value);

        throw new FilterFormatException($message);
    }

    private function validateFilterableAttribute(Engine $engine, string $attributeName): void
    {
        $allowedAttributeNames = $engine->getConfiguration()->getFilterableAttributes();
        if (!\in_array($attributeName, $allowedAttributeNames, true)) {
            $this->syntaxError('filterable attribute');
        }

        $activeGroup = $this->getActiveGroup();

        if ($activeGroup instanceof MultiAttributeFilter && $activeGroup->attribute !== $attributeName) {
            $this->syntaxError(sprintf('identical multi attributes within same group,"%s" and "%s" given.', $activeGroup->attribute, $attributeName));
        }
    }
}
