<?php

declare(strict_types=1);

/*
 * This file is part of Contao.
 *
 * (c) Leo Feyer
 *
 * @license LGPL-3.0-or-later
 */

namespace Contao\CoreBundle\HttpKernel;

use Contao\CoreBundle\Framework\Adapter;
use Contao\CoreBundle\Framework\ContaoFramework;
use Contao\CoreBundle\Routing\ScopeMatcher;
use Contao\Model;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Controller\ValueResolverInterface;
use Symfony\Component\HttpKernel\ControllerMetadata\ArgumentMetadata;

class ModelArgumentResolver implements ValueResolverInterface
{
    /**
     * @internal
     */
    public function __construct(
        private readonly ContaoFramework $framework,
        private readonly ScopeMatcher $scopeMatcher,
    ) {
    }

    public function resolve(Request $request, ArgumentMetadata $argument): array
    {
        if (!$this->scopeMatcher->isContaoRequest($request)) {
            return [];
        }

        $this->framework->initialize();

        if (!is_a($argument->getType(), Model::class, true)) {
            return [];
        }

        if (!$argument->isNullable() && !$this->fetchModel($request, $argument)) {
            return [];
        }

        if (!$model = $this->fetchModel($request, $argument)) {
            return [];
        }

        return [$model];
    }

    private function fetchModel(Request $request, ArgumentMetadata $argument): Model|null
    {
        $name = $this->getArgumentName($request, $argument);

        if (null === $name) {
            return null;
        }

        /** @var class-string<Model> $type */
        $type = $argument->getType();

        /** @var Model|int $value */
        $value = $request->attributes->get($name);

        if ($type && $value instanceof $type) {
            return $value;
        }

        /** @var Adapter<Model> $model */
        $model = $this->framework->getAdapter($type);

        return $model->findById((int) $value);
    }

    /**
     * Returns the argument name from the model class.
     */
    private function getArgumentName(Request $request, ArgumentMetadata $argument): string|null
    {
        if ($request->attributes->has($argument->getName())) {
            return $argument->getName();
        }

        $className = lcfirst($this->stripNamespace($argument->getType()));

        if ($request->attributes->has($className)) {
            return $className;
        }

        return null;
    }

    /**
     * Strips the namespace from a class name.
     */
    private function stripNamespace(string $fqcn): string
    {
        if (false !== ($pos = strrpos($fqcn, '\\'))) {
            return substr($fqcn, $pos + 1);
        }

        return $fqcn;
    }
}
