<?php

declare(strict_types=1);

namespace Drupal\RecipeKit\Installer;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\PluralTranslatableMarkup;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Decorates the messenger to suppress or alter certain install-time messages.
 *
 * @internal
 *    Everything in this class is internal, which means it could be changed in
 *    any way, or removed outright, at any time without warning. It is only
 *   meant to be used by profiles that were generated by this kit. You should
 *   not use it in your own code in any way.
 */
final class Messenger implements MessengerInterface {

  private static array $reject = [
    'Congratulations, you installed @drupal!',
  ];

  public function __construct(
    private readonly MessengerInterface $decorated,
  ) {}

  /**
   * Adds a message to the reject list.
   *
   * @param string $message
   *   The untranslated, unformatted (i.e., containing raw placeholders) message
   *   that should be suppressed.
   */
  public static function reject(string $message): void {
    static::$reject[] = $message;
  }

  /**
   * Adds a pluralized message to the reject list.
   *
   * @param string $singular
   *   The singular, untranslated form of the message to suppress.
   * @param string $plural
   *    The plural, untranslated form of the message to suppress.
   */
  public static function rejectPlural(string $singular, string $plural): void {
    $string = (new PluralTranslatableMarkup(0, $singular, $plural))
      ->getUntranslatedString();

    static::reject($string);
  }

  /**
   * {@inheritdoc}
   */
  public function addMessage($message, $type = self::TYPE_STATUS, $repeat = FALSE): static {
    $raw = $message instanceof TranslatableMarkup
      ? $message->getUntranslatedString()
      : strval($message);

    if (!in_array($raw, static::$reject, TRUE)) {
      $this->decorated->addMessage($message, $type, $repeat);
    }
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function addStatus($message, $repeat = FALSE): static {
    return $this->addMessage($message, self::TYPE_STATUS, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function addError($message, $repeat = FALSE): static {
    return $this->addMessage($message, self::TYPE_ERROR, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function addWarning($message, $repeat = FALSE): static {
    return $this->addMessage($message, self::TYPE_WARNING, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function all(): array {
    return $this->decorated->all();
  }

  /**
   * {@inheritdoc}
   */
  public function messagesByType($type): array {
    return $this->decorated->messagesByType($type);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAll(): array {
    return $this->decorated->deleteAll();
  }

  /**
   * {@inheritdoc}
   */
  public function deleteByType($type): array {
    return $this->decorated->deleteByType($type);
  }

}
