import '../../../vendor/bower-asset/jquery/dist/jquery.min.js';
import { aG as sprintf } from '../../../chunks/egw_json-39123901.js';
import '../../../chunks/egw_inheritance-a27f268b.js';

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 * @version $Id$
 */

/**
 * Log debug messages to browser console and persistent html5 localStorage
 *
 * localStorage is limited by a clientside quota, so we need to deal with
 * situation that storing something in localStorage will throw an exception!
 *
 * @param {string} _app
 * @param {object} _wnd
 */
egw.extend('debug', egw.MODULE_GLOBAL, function(_app, _wnd)
{
	"use strict";

	/**
	 * DEBUGLEVEL specifies which messages are printed to the console.
	 * Decrease the value of EGW_DEBUGLEVEL to get less messages.
	 *
	 * @type Number
	 * 0 = off, no logging
	 * 1 = only "error"
	 * 2 = -- " -- plus "warning"
	 * 3 = -- " -- plus "info"
	 * 4 = -- " -- plus "log"
	 * 5 = -- " -- plus a stacktrace
	 */
	var DEBUGLEVEL = 3;

	/**
	 * Log-level for local storage and error-display in GUI
	 *
	 * @type Number
	 * 0 = off, no logging AND no global error-handler bound
	 * 1 = ... see above
	 */
	var LOCAL_LOG_LEVEL = 0;
	/**
	 * Number of log-entries stored on client, new errors overwrite old ones
	 *
	 * @type Number
	 */
	var MAX_LOGS = 200;
	/**
	 * Number of last old log entry = next one to overwrite
	 *
	 * @type String
	 */
	var LASTLOG = 'lastLog';
	/**
	 * Prefix for key of log-message, message number gets appended to it
	 *
	 * @type String
	 */
	var LOG_PREFIX = 'log_';

	/**
	 * Log to clientside html5 localStorage
	 *
	 * @param {String} _level "navigation", "log", "info", "warn", "error"
	 * @param {Array} _args arguments to egw.debug
	 * @param {string} _stack
	 * @returns {Boolean} false if localStorage is NOT supported, null if level requires no logging, true if logged
	 */
	function log_on_client(_level, _args, _stack)
	{
		if (!window.localStorage) return false;

		switch(_level)
		{
			case 'warn':
				if (LOCAL_LOG_LEVEL < 2) return null;
			case 'info':
				if (LOCAL_LOG_LEVEL < 3) return null;
			case 'log':
				if (LOCAL_LOG_LEVEL < 4) return null;
			default:
				if (!LOCAL_LOG_LEVEL) return null;
		}
		var data = {
			time: (new Date()).getTime(),
			level: _level,
			args: _args
		};
		// Add in a trace, if no navigation _level
		if (_level != 'navigation')
		{
			if (_stack)
			{
				data.stack = _stack;
			}
			else
			{
				// IE needs to throw the error to get a stack trace!
				try {
					throw new Error;
				}
				catch(error) {
					data.stack = error.stack;
				}
			}
		}
		if (typeof window.localStorage[LASTLOG] == 'undefined')
		{
			window.localStorage[LASTLOG] = 0;
		}
		// check if MAX_LOGS changed in code --> clear whole log
		if (window.localStorage[LASTLOG] > MAX_LOGS)
		{
			clear_client_log();
		}
		try {
			window.localStorage[LOG_PREFIX+window.localStorage[LASTLOG]] = JSON.stringify(data);
			window.localStorage[LASTLOG] = (1 + parseInt(window.localStorage[LASTLOG])) % MAX_LOGS;
		}
		catch(e) {
			switch (e.name)
			{
				case 'QuotaExceededError':	// storage quota is exceeded --> delete whole log
				case 'NS_ERROR_DOM_QUOTA_REACHED':	// FF-name
					clear_client_log();
					break;

				default:
					// one of the args is not JSON.stringify, because it contains circular references eg. an et2 widget
					for(var i=0; i < data.args.length; ++i)
					{
						try {
							JSON.stringify(data.args[i]);
						}
						catch(e) {
							// for Class we try removing _parent and _children attributes and try again to stringify
							if (data.args[i] instanceof Class)
							{
								data.args[i] = jQuery.extend({}, data.args[i]);
								delete data.args[i]._parent;
								delete data.args[i]._children;
								try {
									JSON.stringify(data.args[i]);
									continue;	// stringify worked --> check other arguments
								}
								catch(e) {
									// ignore error and remove whole argument
								}
							}
							// if above doesnt work, we remove the attribute
							data.args[i] = '** removed, circular reference **';
						}
					}
			}
			try {
				window.localStorage[LOG_PREFIX+window.localStorage[LASTLOG]] = JSON.stringify(data);
				window.localStorage[LASTLOG] = (1 + parseInt(window.localStorage[LASTLOG])) % MAX_LOGS;
			}
			catch(e) {
				// ignore error, if eg. localStorage exceeds quota on client
			}
		}
	}

	/**
	 * Get log from localStorage with oldest message first
	 *
	 * @returns {Array} of Object with values for attributes level, message, trace
	 */
	function get_client_log()
	{
		var logs = [];

		if (window.localStorage && typeof window.localStorage[LASTLOG] != 'undefined')
		{
			var lastlog = parseInt(window.localStorage[LASTLOG]);
			for(var i=lastlog; i < lastlog+MAX_LOGS; ++i)
			{
				var log = window.localStorage[LOG_PREFIX+(i%MAX_LOGS)];
				if (typeof log != 'undefined')
				{
					try {
						logs.push(JSON.parse(log));
					}
					catch(e) {
						// ignore not existing log entries
					}
				}
			}
		}
		return logs;
	}

	/**
	 * Clears whole client log
	 */
	function clear_client_log()
	{
		// Remove indicator icon
		jQuery('#topmenu_info_error').remove();

		if (!window.localStorage) return false;

		var max = MAX_LOGS;
		// check if we have more log entries then allowed, happens if MAX_LOGS get changed in code
		if (window.localStorage[LASTLOG] > MAX_LOGS)
		{
			max = 1000;	// this should NOT be changed, if MAX_LOGS get's smaller!
		}
		for(var i=0; i < max; ++i)
		{
			if (typeof window.localStorage[LOG_PREFIX+i] != 'undefined')
			{
				delete window.localStorage[LOG_PREFIX+i];
			}
		}
		delete window.localStorage[LASTLOG];

		return true;
	}

	/**
	 * Format one log message for display
	 *
	 * @param {Object} log {{level: string, time: number, stack: string, args: array[]}} Log information
	 *	Actual message is in args[0]
	 * @returns {DOMNode}
	 */
	function format_message(log)
	{
		var row = document.createElement('tr');
		row.setAttribute('class', log.level);
		var timestamp = row.insertCell(-1);
		timestamp.appendChild(document.createTextNode(new Date(log.time)));
		timestamp.setAttribute('class', 'timestamp');

		var level = row.insertCell(-1);
		level.appendChild(document.createTextNode(log.level));
		level.setAttribute('class', 'level');

		var message = row.insertCell(-1);
		for(var i = 0; i < log.args.length; i++)
		{

			var arg = document.createElement('p');
			arg.appendChild(
				document.createTextNode(typeof log.args[i] == 'string' ? log.args[i] : JSON.stringify( log.args[i]))
			);
			message.appendChild(arg);
		}

		var stack = row.insertCell(-1);
		stack.appendChild(document.createTextNode(log.stack||''));
		stack.setAttribute('class','stack');

		return row;
	}

	/**
	 * Show user an error happend by displaying a clickable icon with tooltip of current error
	 */
	function raise_error()
	{
		var icon = jQuery('#topmenu_info_error');
		if (!icon.length)
		{
			var icon = jQuery(egw(_wnd).image_element(egw.image('dialog_error')));
			icon.addClass('topmenu_info_item').attr('id', 'topmenu_info_error');
			// ToDo: tooltip
			icon.on('click', egw(_wnd).show_log);
			jQuery('#egw_fw_topmenu_info_items,#topmenu_info').append(icon);
		}
	}

	// bind to global error handler, only if LOCAL_LOG_LEVEL > 0
	if (LOCAL_LOG_LEVEL)
	{
		jQuery(_wnd).on('error', function(e)
		{
			// originalEvent does NOT always exist in IE
			var event = typeof e.originalEvent == 'object' ? e.originalEvent : e;
			// IE(11) gives a syntaxerror on each pageload pointing to first line of html page (doctype).
			// As I cant figure out what's wrong there, we are ignoring it for now.
			if (navigator.userAgent.match(/Trident/i) && typeof event.name == 'undefined' &&
				Object.prototype.toString.call(event) == '[object ErrorEvent]' &&
				event.lineno == 1 && event.filename.indexOf('/index.php') != -1)
			{
				return false;
			}
			log_on_client('error', [event.message], typeof event.stack != 'undefined' ? event.stack : null);
			raise_error();
			// rethrow error to let browser log and show it in usual way too
			if (typeof event.error == 'object')
			{
				throw event.error;
			}
			throw event.message;
		});
	}

	/**
	 * The debug function can be used to send a debug message to the
	 * java script console. The first parameter specifies the debug
	 * level, all other parameters are passed to the corresponding
	 * console function.
	 */
	return {
		debug_level: function() {
			return DEBUGLEVEL;
		},
		debug: function(_level) {
			if (typeof _wnd.console != "undefined")
			{
				// Get the passed parameters and remove the first entry
				var args = [];
				for (var i = 1; i < arguments.length; i++)
				{
					args.push(arguments[i]);
				}

				// Add in a trace
				if (DEBUGLEVEL >= 5 && typeof (new Error).stack != "undefined")
				{
					var stack = (new Error).stack;
					args.push(stack);
				}

				if (_level == "log" && DEBUGLEVEL >= 4 &&
					typeof _wnd.console.log == "function")
				{
					_wnd.console.log.apply(_wnd.console, args);
				}

				if (_level == "info" && DEBUGLEVEL >= 3 &&
					typeof _wnd.console.info == "function")
				{
					_wnd.console.info.apply(_wnd.console, args);
				}

				if (_level == "warn" && DEBUGLEVEL >= 2 &&
					typeof _wnd.console.warn == "function")
				{
					_wnd.console.warn.apply(_wnd.console, args);
				}

				if (_level == "error" && DEBUGLEVEL >= 1 &&
					typeof _wnd.console.error == "function")
				{
					_wnd.console.error.apply(_wnd.console, args);
				}
			}
			// raise errors to user, if LOCAL_LOG_LEVEL > 0
			if (LOCAL_LOG_LEVEL && _level == "error") raise_error(args);

			// log to html5 localStorage
			if (typeof stack != 'undefined') args.pop();	// remove stacktrace again
			log_on_client(_level, args);
		},

		/**
		 * Display log to user because he clicked on icon showed by raise_error
		 *
		 * @returns {undefined}
		 */
		show_log: function()
		{
			var table = document.createElement('table');
			var body = document.createElement('tbody');
			var client_log = get_client_log();
			for(var i = 0; i < client_log.length; i++)
			{
				body.appendChild(format_message(client_log[i]));
			}
			table.appendChild(body);

			// Use a wrapper div for ease of styling
			var wrapper = document.createElement('div');
			wrapper.setAttribute('class', 'client_error_log');
			wrapper.appendChild(table);

			if(window.jQuery && window.jQuery.ui.dialog)
			{
				var $wrapper = jQuery(wrapper);
				// Start hidden
				jQuery('tr',$wrapper).addClass('hidden')
					.on('click', function() {
						jQuery(this).toggleClass('hidden',{});
						jQuery(this).find('.stack').children().toggleClass('ui-icon ui-icon-circle-plus');
					});
				// Wrap in div so we can control height
				jQuery('td',$wrapper).wrapInner('<div/>')
					.filter('.stack').children().addClass('ui-icon ui-icon-circle-plus');

				$wrapper.dialog({
					title: egw.lang('Error log'),
					buttons: [
						{text: egw.lang('OK'), click: function() {jQuery(this).dialog( "close" ); }},
						{text: egw.lang('clear'), click: function() {clear_client_log(); jQuery(this).empty();}}
					],
					width: 800,
					height: 400
				});
				$wrapper[0].scrollTop = $wrapper[0].scrollHeight;
			}
			if (_wnd.console) _wnd.console.log(get_client_log());
		}
	};
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 * @version $Id$
 */

egw.extend('preferences', egw.MODULE_GLOBAL, function()
{
	"use strict";

	/**
	 * Object holding the prefences as 2-dim. associative array, use
	 * egw.preference(name[,app]) to access it.
	 *
	 * @access: private, use egw.preferences() or egw.set_perferences()
	 */
	var prefs = {
		common:{textsize:12}
	};
	var grants = {};

	/**
	 * App-names in egw_preference table are limited to 16 chars, so we can not store anything longer
	 *
	 * Also modify tab-names used in CRM-view ("addressbook-*") to "infolog".
	 *
	 * @param _app
	 * @returns {string}
	 */
	function sanitizeApp(_app)
	{
		if (typeof _app === 'undefined') _app = 'common';

		if (_app.length > 16)
		{
			_app = _app.substring(0, 16);
		}
		if (_app.match(/^addressbook-/))
		{
			_app = 'infolog';
		}
		return _app;
	}

	// Return the actual extension
	return {
		/**
		 * Setting prefs for an app or 'common'
		 *
		 * @param {object} _data object with name: value pairs to set
		 * @param {string} _app application name, 'common' or undefined to prefes of all apps at once
		 * @param {boolean} _need_clone _data need to be cloned, as it is from different window context
		 *	and therefore will be inaccessible in IE, after that window is closed
		 */
		set_preferences: function(_data, _app, _need_clone)
		{
			if (typeof _app == 'undefined')
			{
				prefs = _need_clone ? jQuery.extend(true, {}, _data) : _data;
			}
			else
			{
				prefs[sanitizeApp(_app)] = jQuery.extend(true, {}, _data);	// we always clone here, as call can come from this.preferences!
			}
		},

		/**
		 * Query an EGroupware user preference
		 *
		 * If a preference is not already loaded (only done for "common" by default),
		 * it is synchronously queried from the server, if no _callback parameter is given!
		 *
		 * @param {string} _name name of the preference, eg. 'dateformat', or '*' to get all the application's preferences
		 * @param {string} _app default 'common'
		 * @param {function|boolean|undefined} _callback optional callback, if preference needs loading first
		 *  - default/undefined: preference is synchronously queried, if not loaded, and returned
		 *  - function: if loaded, preference is returned, if not false and callback is called once it's loaded
		 * 	- true:  a promise for the preference is returned
		 *	- false: if preference is not loaded, undefined is return and no (synchronous) request is send to server
		 * @param {object} _context context for callback
		 * @return Promise|object|string|bool (Promise for) preference value or false, if callback given and preference not yet loaded
		 */
		preference: function(_name, _app, _callback, _context)
		{
			_app = sanitizeApp(_app);

			if (typeof prefs[_app] === 'undefined')
			{
				if (_callback === false) return undefined;
				const request = this.json('EGroupware\\Api\\Framework::ajax_get_preference', [_app], _callback, _context);
				const promise = request.sendRequest(typeof _callback !== 'undefined', 'GET');
				if (typeof prefs[_app] === 'undefined') prefs[_app] = promise;
				if (_callback === true) return promise.then(() => this.preference(_name, _app));
				if (typeof _callback === 'function') return false;
			}
			else if (typeof prefs[_app] === 'object' && typeof prefs[_app].then === 'function')
			{
				if (_callback === false) return undefined;
				if (_callback === true) return prefs[_app].then(() => this.preference(_name, _app));
				if (typeof _callback === 'function') return false;
			}
			let ret;
			if (_name === "*")
			{
				ret = typeof prefs[_app] === 'object' ? jQuery.extend({}, prefs[_app]) : prefs[_app];
			}
			else
			{
				ret = typeof prefs[_app][_name] === 'object' && prefs[_app][_name] !== null ?
					jQuery.extend({}, prefs[_app][_name]) : prefs[_app][_name];
			}
			if (_callback === true)
			{
				return Promise.resolve(ret);
			}
			return ret;
		},

		/**
		 * Set a preference and sends it to the server
		 *
		 * Server will silently ignore setting preferences, if user has no right to do so!
		 *
		 * Preferences are only send to server, if they are changed!
		 *
		 * @param {string} _app application name or "common"
		 * @param {string} _name name of the pref
		 * @param {string} _val value of the pref, null, undefined or "" to unset it
		 * @param {function} _callback Function passed along to the queue, called after preference is set server-side,
		 *	IF the preference is changed / has a value different from the current one
		 */
		set_preference: function(_app, _name, _val, _callback)
		{
			_app = sanitizeApp(_app);

			// if there is no change, no need to submit it to server
			if (typeof prefs[_app] != 'undefined')
			{
				var current = prefs[_app][_name];
				var setting = _val;
				// to compare objects we serialize them
				if (typeof current == 'object') current = JSON.stringify(current);
				if (typeof setting == 'object') setting = JSON.stringify(setting);
				if (setting === current) return;
			}

			this.jsonq('EGroupware\\Api\\Framework::ajax_set_preference',[_app, _name, _val], _callback);

			// update own preference cache, if _app prefs are loaded (don't update otherwise, as it would block loading of other _app prefs!)
			if (typeof prefs[_app] != 'undefined')
			{
				if (_val === undefined || _val === "" || _val === null)
				{
					delete prefs[_app][_name];
				}
				else
				{
					prefs[_app][_name] = _val;
				}
			}
		},

		/**
		 * Endpoint for push to request reload of preference, if loaded and affected
		 *
		 * @param _app app-name of prefs to reload
		 * @param _account_id _account_id 0: allways reload (default or forced prefs), <0: reload if member of group
		 */
		reload_preferences: function(_app, _account_id)
		{
			if (typeof _account_id !== 'number') _account_id = parseInt(_account_id);
			if (typeof prefs[_app] === 'undefined' ||	// prefs not loaded
				_account_id < 0 && this.user('memberships').indexOf(_account_id) < 0)	// no member of this group
			{
				return;
			}
			var request = this.json('EGroupware\\Api\\Framework::ajax_get_preference', [_app]);
			request.sendRequest();
		},

		/**
		 * Call context / open app specific preferences function
		 *
		 * @param {string} name type 'acl', 'prefs', or 'cats'
		 * @param {(array|object)} apps array with apps allowing to call that type, or object/hash with app and boolean or hash with url-params
		 */
		show_preferences: function (name, apps)
		{
			var current_app = this.app_name();
			var query = {menuaction:'',current_app: current_app};
			// give warning, if app does not support given type, but all apps link to common prefs, if they dont support prefs themselfs
			if (jQuery.isArray(apps) && jQuery.inArray(current_app, apps) == -1 && (name != 'prefs' && name != 'acl') ||
				!jQuery.isArray(apps) && (typeof apps[current_app] == 'undefined' || !apps[current_app]))
			{
				egw_message(egw.lang('Not supported by current application!'), 'warning');
			}
			else
			{
				var url = '/index.php';
				switch(name)
				{
					case 'prefs':
						query.menuaction ='preferences.preferences_settings.index';
						if (jQuery.inArray(current_app, apps) != -1) query.appname=current_app;
						egw.open_link(egw.link(url, query), '_blank', '1200x600');
						break;

					case 'acl':
						query.menuaction='preferences.preferences_acl.index';
						if (jQuery.inArray(current_app, apps) != -1) query.acl_app=current_app;
						egw.open_link(egw.link(url, query), '_blank', '1200x600');
						break;

					case 'cats':
						if (typeof apps[current_app] == 'object')
						{
							for(var key in apps[current_app])
							{
								query[key] = encodeURIComponent(apps[current_app][key]);
							}
						}
						else
						{
							query.menuaction='preferences.preferences_categories_ui.index';
							query.cats_app=current_app;
						}
						query.ajax = true;
						egw.link_handler(egw.link(url, query), current_app);
						break;
				}
			}
		},

		/**
		 * Setting prefs for an app or 'common'
		 *
		 * @param {object} _data
		 * @param {string} _app application name or undefined to set grants of all apps at once
		 *	and therefore will be inaccessible in IE, after that window is closed
		 */
		set_grants: function(_data, _app)
		{
			if (_app)
			{
				grants[_app] = jQuery.extend(true, {}, _data);
			}
			else
			{
				grants = jQuery.extend(true, {}, _data);
			}
		},

		/**
		 * Query an EGroupware user preference
		 *
		 * We currently load grants from all apps in egw.js, so no need for a callback or promise.
		 *
		 * @param {string} _app app-name
		 * @param {function|false|undefined} _callback optional callback, if preference needs loading first
		 * if false given and preference is not loaded, undefined is return and no (synchronious) request is send to server
		 * @param {object} _context context for callback
		 * @return {object|undefined|false} grant object, false if not (yet) loaded and no callback or undefined
		 */
		grants: function( _app) //, _callback, _context)
		{
			/* we currently load grants from all apps in egw.js, so no need for a callback or promise
			if (typeof grants[_app] == 'undefined')
			{
				if (_callback === false) return undefined;
				var request = this.json('EGroupware\\Api\\Framework::ajax_get_preference', [_app], _callback, _context);
				request.sendRequest(typeof _callback == 'function', 'GET');	// use synchronous (cachable) GET request
				if (typeof grants[_app] == 'undefined') grants[_app] = {};
				if (typeof _callback == 'function') return false;
			}*/
			return typeof grants[_app] === 'object' ? jQuery.extend({}, grants[_app]) : grants[_app];
		},

		/**
		 * Get mime types supported by file editor AND not excluded by user
		 *
		 * @param {string} _mime current mime type
		 * @returns {object|null} returns object of filemanager editor hook
		 */
		file_editor_prefered_mimes: function(_mime)
		{
			const fe = jQuery.extend(true, {}, this.link_get_registry('filemanager-editor'));
			let ex_mimes = this.preference('collab_excluded_mimes', 'filemanager');
			const dblclick_action = this.preference('document_doubleclick_action', 'filemanager');
			if (dblclick_action === 'download' && typeof _mime === 'string')
			{
				ex_mimes = !ex_mimes ? _mime : ex_mimes+','+_mime;
			}
			if (fe && fe.mime && ex_mimes && typeof ex_mimes === 'string')
			{
				ex_mimes = ex_mimes.split(',');
				for (let mime in fe.mime)
				{
					for (let i in ex_mimes)
					{
						if (ex_mimes[i] === mime) delete(fe.mime[mime]);
					}
				}
			}
			return fe && fe.mime ? fe : null;
		}
	};
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 * @version $Id$
 */

/**
 * @augments Class
 */
egw.extend('lang', egw.MODULE_GLOBAL, function()
{
	"use strict";

	/**
	 * Translations
	 *
	 * @access: private, use egw.lang() or egw.set_lang_arr()
	 */
	const lang_arr = {};

	// Return the actual extension
	return {
		/**
		 * Set translation for a given application
		 *
		 * @param {string} _app
		 * @param {object} _messages message => translation pairs
		 * @param {boolean} _need_clone _messages need to be cloned, as it is from different window context
		 *	and therefore will be inaccessible in IE, after that window is closed
		 * @memberOf egw
		 */
		set_lang_arr: function(_app, _messages, _need_clone)
		{
			if(!jQuery.isArray(_messages))
			{
				// no deep clone jQuery.extend(true,...) neccessary, as _messages contains only string values
				lang_arr[_app] = _need_clone ? jQuery.extend({}, _messages) : _messages;
			}
		},

		/**
		 * Translate a given phrase replacing optional placeholders
		 *
		 * @param {string} _msg message to translate
		 * @param {...string} _arg1 ... _argN
		 * @return {string}
		 */
		lang: function(_msg, _arg1)
		{
			if(_msg === null)
			{
				return '';
			}
			if(typeof _msg !== "string" && _msg)
			{
				egw().debug("warn", "Cannot translate an object", _msg);
				return _msg;
			}
			var translation = _msg;
			_msg = _msg.toLowerCase();

			// search apps in given order for a replacement
			var apps = this.lang_order || ['custom', this.getAppName(), 'etemplate', 'common', 'notifications'];
			for(var i = 0; i < apps.length; ++i)
			{
				if (typeof lang_arr[apps[i]] != "undefined" &&
					typeof lang_arr[apps[i]][_msg] != 'undefined')
				{
					translation = lang_arr[apps[i]][_msg];
					break;
				}
			}
			if (arguments.length == 1) return translation;

			if (arguments.length == 2) return translation.replace('%1', arguments[1]);

			// to cope with arguments containing '%2' (eg. an urlencoded path like a referer),
			// we first replace all placeholders '%N' with '|%N|' and then we replace all '|%N|' with arguments[N]
			translation = translation.replace(/%([0-9]+)/g, '|%$1|');
			for(var i = 1; i < arguments.length; ++i)
			{
				translation = translation.replace('|%'+i+'|', arguments[i]);
			}
			return translation;
		},

		/**
		 * Load default langfiles for an application: common, _appname, custom
		 *
		 * @param {DOMElement} _window
		 * @param {string} _appname name of application to load translations for
		 * @param {function} _callback
		 * @param {object} _context
		 */
		langRequireApp: function(_window, _appname, _callback, _context)
		{
			var lang = egw.preference('lang');
			var langs = [{app: 'common', lang: lang}];

			if (_appname && _appname != 'eGroupWare')
			{
				langs.push({app: _appname, lang: lang});
			}
			langs.push({app: 'custom', lang: 'en'});

			this.langRequire(_window, langs, _callback, _context);
		},

		/**
		 * Includes the language files for the given applications -- if those
		 * do not already exist, include them.
		 *
		 * @param {DOMElement} _window is the window which needs the language -- this is
		 * 	needed as the "ready" event has to be postponed in that window until
		 * 	all lang files are included.
		 * @param {array} _apps is an array containing the applications for which the
		 * 	data is needed as objects of the following form:
		 * 		{
		 * 			app: <APPLICATION NAME>,
		 * 			lang: <LANGUAGE CODE>
		 * 		}
		 * @param {function} _callback called after loading, if not given ready event will be postponed instead
		 * @param {object} _context for callback
		 * @return Promise
		 */
		langRequire: function(_window, _apps, _callback, _context) {
			// Get the ready and the files module for the given window
			var ready = this.module("ready", _window);
			var files = this.module("files", this.window);

			// Build the file names which should be included
			var jss = [];
			var apps = [];
			for (var i = 0; i < _apps.length; i++)
			{
				if (!_apps[i].app) continue;
				if (typeof lang_arr[_apps[i].app] === "undefined")
				{
					jss.push(this.webserverUrl +
						'/api/lang.php?app=' + _apps[i].app +
						'&lang=' + _apps[i].lang +
						'&etag=' + (_apps[i].etag || this.config('max_lang_time')));
				}
				apps.push(_apps[i].app);
			}
			if (this !== egw && apps.length > 0)
			{
				this.lang_order = apps.reverse();
			}

			const promise = Promise.all(jss.map((src) => import(src)));
			return typeof _callback === 'function' ? promise.then(() => _callback.call(_context)) : promise;
		}
	};

});

/**
 * EGroupware clientside API: link-registry, link-titles, generation links
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 */

/**
 * @augments Class
 */
egw.extend('links', egw.MODULE_GLOBAL, function()
{
	"use strict";

	/**
	 * Link registry
	 *
	 * @access: private, use egw.open() or egw.set_link_registry()
	 */
	let link_registry = undefined;

	/**
	 * Local cache for link-titles
	 *
	 * @access private, use egw.link_title(_app, _id[, _callback, _context])
	 */
	let title_cache = {};

	/**
	 * Queue for link_title requests
	 *
	 * @access private, use egw.link_title(_app, _id[, _callback, _context])
	 * @var object _app._id.[{callback: _callback, context: _context}[, ...]]
	 */
	let title_queue = {};

	/**
	 * Uid of active jsonq request, to not start another one, as we get notified
	 * before it's actually send to the server via our link_title_before_send callback.
	 * @access private
	 */
	let title_uid = null;

	/**
	 * Encode query parameters
	 *
	 * @param object|array|string _values
	 * @param string? _prefix
	 * @param array? _query
	 * @return array
	 */
	function urlencode(_values, _prefix, _query)
	{
		if (typeof _query === 'undefined') _query = [];
		if (Array.isArray(_values))
		{
			if (!_prefix) throw "array of value needs a prefix";
			for(const value of _values)
			{
				_query.push(_prefix+'[]='+encodeURIComponent(value));
			}
		}
		else if (_values && typeof _values === 'object')
		{
			for(const name in _values)
			{
				urlencode(_values[name], _prefix ? _prefix+'['+name+']' : name, _query);
			}
		}
		else
		{
			_query.push(_prefix+'='+encodeURIComponent(_values || ''));
		}
		return _query;
	}

	return {
		/**
		 * Check if $app is in the registry and has an entry for $name
		 *
		 * @param {string} _app app-name
		 * @param {string} _name name / key in the registry, eg. 'view'
		 * @return {boolean|string} false if $app is not registered, otherwise string with the value for $name
		 * @memberOf egw
		 */
		link_get_registry: function(_app, _name)
		{
			if (typeof link_registry !== 'object')
			{
				alert('egw.open() link registry is NOT defined!');
				return false;
			}
			if (typeof link_registry[_app] === 'undefined')
			{
				return false;
			}
			const reg = link_registry[_app];

			// some defaults (we set them directly in the registry, to do this only once)
			if (typeof reg[_name] === 'undefined')
			{
				switch(_name)
				{
					case 'name':
						reg.name = _app;
						break;
					case 'icon':
						const app_data = this.app(_app);
						if (typeof app_data !== 'undefined' &&
							typeof app_data.icon !== 'undefined' && app_data.icon !== null)
						{
							reg.icon = (typeof app_data.icon_app != 'undefined' ? app_data.icon_app : _app)+'/'+app_data.icon;
						}
						else
						{
							reg.icon = _app+'/navbar';
						}
						break;
				}
			}
			if (reg && typeof _name === 'undefined')
			{
				// No key requested, return the whole thing
				return reg;
			}
			return typeof reg[_name] === 'undefined' ? false : reg[_name];
		},

		/**
		 * Get mime-type information from app-registry
		 *
		 * We prefer a full match over a wildcard like 'text/*' (written as regular expr. "/^text\\//"
		 *
		 * @param {string} _type
		 * @param {number|string} _app_or_num default 1, return 1st, 2nd, n-th match, or match from application _app_or_num only
		 * @return {object} with values for keys 'menuaction', 'mime_id' (path) or 'mime_url' and options 'mime_popup' and other values to pass one
		 */
		get_mime_info: function(_type, _app_or_num)
		{
			if (!_app_or_num) _app_or_num = 1;
			let wildcard_mime;
			for(const app of isNaN(_app_or_num) ? [_app_or_num] : Object.keys(link_registry))
			{
				const reg = link_registry[app];
				if (typeof reg?.mime !== 'undefined')
				{
					for(let mime in reg.mime)
					{
						if (mime === _type)
						{
							if (isNaN(_app_or_num) || !--_app_or_num)
							{
								return reg.mime[_type];
							}
							continue;
						}
						if (mime[0] === '/' && _type.match(new RegExp(mime.substring(1, mime.length-1), 'i')))
						{
							wildcard_mime = reg.mime[mime];
						}
					}
				}
			}
			return wildcard_mime ? wildcard_mime : null;
		},

		/**
		 * Get handler (link-data) for given path and mime-type
		 *
		 * @param {string|object} _path vfs path, egw_link::set_data() id or
		 *	object with attr path, optional download_url or id, app2 and id2 (path=/apps/app2/id2/id)
		 * @param {string} _type mime-type, if not given in _path object
		 * @param {number|string} _app_or_num default 1, use 1st, 2nd, n-th match, or match from application _app_or_num only
		 * @return {string|object} string with EGw relative link, array with get-parameters for '/index.php' or null (directory and not filemanager access)
		 */
		mime_open: function(_path, _type, _app_or_num)
		{
			let path;
			if (typeof _path === 'object')
			{
				if (typeof _path.path === 'undefined')
				{
					path = '/apps/'+_path.app2+'/'+_path.id2+'/'+_path.id;
				}
				else
				{
					path = _path.path;
				}
				if (typeof _path.type === 'string')
				{
					_type = _path.type;
				}
			}
			else if(_path[0] !== '/')
			{

			}
			else
			{
				path = _path;
			}
			let mime_info = this.get_mime_info(_type, _app_or_num);
			let data = {};
			if (mime_info)
			{
				if ((typeof _app_or_num === 'undefined' || _app_or_num === 'collabora') && this.isCollaborable(_type))
				{
					data = {
						'menuaction': 'collabora.EGroupware\\collabora\\Ui.editor',
						'path': path,
						'cd': 'no'	// needed to not reload framework in sharing
					};
					return data;
				}
				for(let attr in mime_info)
				{
					switch(attr)
					{
						case 'mime_url':
							if (path)
							{
								data[mime_info.mime_url] = 'vfs://default' + path;
							}
							break;
						case 'mime_data':
							if (!path && _path && typeof _path === 'string')
							{
								data[mime_info.mime_data] = _path;
							}
							break;
						case 'mime_type':
							data[mime_info.mime_type] = _type;
							break;
						case 'mime_id':
							data[mime_info.mime_id] = path;
							break;
						default:
							data[attr] = mime_info[attr];
					}
				}
				// if mime_info did NOT define mime_url attribute, we use a WebDAV url drived from path
				if (typeof mime_info.mime_url === 'undefined')
				{
					data.url = typeof _path === 'object' && _path.download_url ? _path.download_url : '/webdav.php' + path;
				}
			}
			else
			{
				data = typeof _path === 'object' && _path.download_url ? _path.download_url : '/webdav.php' + path;
			}
			return data;
		},

		/**
		 * Get list of link-aware apps the user has rights to use
		 *
		 * @param {string} _must_support capability the apps need to support, eg. 'add', default ''=list all apps
		 * @return {object} with app => title pairs
		 */
		link_app_list: function(_must_support)
		{
			let apps = [];
			for (let type in link_registry)
			{
				const reg = link_registry[type];

				if (typeof _must_support !== 'undefined' && _must_support && typeof reg[_must_support] === 'undefined') continue;

				const app_sub = type.split('-');
				if (this.app(app_sub[0]))
				{
					apps.push({"type": type, "label": this.lang(this.link_get_registry(type,'name'))});
				}
			}
			// sort labels (case-insensitive) alphabetic
			apps = apps.sort((_a, _b) =>
			{
				var al = _a.label.toUpperCase();
				var bl = _b.label.toUpperCase();
				return al === bl ? 0 : (al > bl ? 1 : -1);
			});
			// create sorted associative array / object
			const sorted = {};
			for(let i = 0; i < apps.length; ++i)
			{
				sorted[apps[i].type] = apps[i].label;
			}
			return sorted;
		},

		/**
		 * Set link registry
		 *
		 * @param {object} _registry whole registry or entries for just one app
		 * @param {string} _app
		 * @param {boolean} _need_clone _images need to be cloned, as it is from different window context
		 *	and therefore will be inaccessible in IE, after that window is closed
		 */
		set_link_registry: function (_registry, _app, _need_clone)
		{
			if (typeof _app === 'undefined')
			{
				link_registry = _need_clone ? jQuery.extend(true, {}, _registry) : _registry;
			}
			else
			{
				link_registry[_app] = _need_clone ? jQuery.extend(true, {}, _registry) : _registry;
			}
		},

		/**
		 * Generate a url which supports url or cookies based sessions
		 *
		 * Please note, the values of the query get url encoded!
		 *
		 * @param {string} _url a url relative to the egroupware install root, it can contain a query too or
		 *	full url containing a schema and "://"
		 * @param {object|string} _extravars query string arguements as string or array (prefered)
		 * 	if string is used ambersands in vars have to be already urlencoded as '%26', function ensures they get NOT double encoded
		 * @return {string} generated url
		 */
		link: function(_url, _extravars)
		{
			if (_url.substr(0,4) === 'http' && _url.indexOf('://') <= 5)
			{
				// already a full url (eg. download_url of vfs), nothing to do
			}
			else
			{
				if (_url[0] != '/')
				{
					alert("egw.link('"+_url+"') called with url starting NOT with a slash!");
					const app = window.egw_appName;
					if (app != 'login' && app != 'logout') _url = app+'/'+_url;
				}
				// append the url to the webserver url, if not already contained or empty
				if (this.webserverUrl && this.webserverUrl != '/' && _url.indexOf(this.webserverUrl+'/') != 0)
				{
					_url = this.webserverUrl + _url;
				}
			}
			const vars = {};

			// check if the url already contains a query and ensure that vars is an array and all strings are in extravars
			const url_othervars = _url.split('?',2);
			_url = url_othervars[0];
			const othervars = url_othervars[1];
			if (_extravars && typeof _extravars == 'object')
			{
				jQuery.extend(vars, _extravars);
				_extravars = othervars;
			}
			else
			{
				if (!_extravars) _extravars = '';
				if (othervars) _extravars += (_extravars?'&':'')+othervars;
			}

			// parse extravars string into the vars array
			if (_extravars)
			{
				_extravars = _extravars.split('&');
				for(let i=0; i < _extravars.length; ++i)
				{
					const name_val = _extravars[i].split('=', 2);
					let name = name_val[0];
					let val = name_val[1] || '';
					if (val.indexOf('%26') !== -1) val = val.replace(/%26/g,'&');	// make sure to not double encode &
					if (name.lastIndexOf('[]') != -1 && name.lastIndexOf('[]') == name.length-2)
					{
						name = name.substr(0,name.length-2);
						if (typeof vars[name] === 'undefined') vars[name] = [];
						vars[name].push(val);
					}
					else
					{
						vars[name] = val;
					}
				}
			}

			// if there are vars, we add them urlencoded to the url
			return Object.keys(vars).length ? _url+'?'+urlencode(vars).join('&') : _url;
		},

		/**
		 * Query a title of _app/_id
		 *
		 * Deprecated default of returning string or null for no callback, will change in future to always return a Promise!
		 *
		 * @param {string} _app
		 * @param {string|number} _id
		 * @param {boolean|function|undefined} _callback true to always return a promise, false: just lookup title-cache or optional callback
		 * 	NOT giving either a boolean value or a callback is deprecated!
		 * @param {object|undefined} _context context for the callback
		 * @param {boolean} _force_reload true load again from server, even if already cached
		 * @return {Promise|string|null} Promise for _callback given (function or true), string with title if it exists in local cache or null if not
		 */
		link_title: function(_app, _id, _callback, _context, _force_reload)
		{
			// check if we have a cached title --> return it direct
			if (typeof title_cache[_app] !== 'undefined' && typeof title_cache[_app][_id] !== 'undefined' && _force_reload !== true)
			{
				if (typeof _callback === 'function')
				{
					_callback.call(_context, title_cache[_app][_id]);
				}
				if (_callback)
				{
					return Promise.resolve(title_cache[_app][_id]);
				}
				return title_cache[_app][_id];
			}
			// no callback --> return null
			if (!_callback)
			{
				if (_callback !== false)
				{
					console.trace('Deprecated use of egw.link() without 3rd parameter callback!');
				}
				return null;	// not found in local cache and can't do a synchronous request
			}
			// queue the request
			if (typeof title_queue[_app] === 'undefined')
			{
				title_queue[_app] = {};
			}
			if (typeof title_queue[_app][_id] === 'undefined')
			{
				title_queue[_app][_id] = [];
			}
			let promise = new Promise(_resolve => {
				title_queue[_app][_id].push({callback: _resolve, context: _context});
			});
			if (typeof _callback === 'function')
			{
				promise = promise.then(_data => {
					_callback.bind(_context)(_data);
					return _data;
				});
			}
			// if there's no active jsonq request, start a new one
			if (title_uid === null)
			{
				title_uid = this.jsonq('EGroupware\\Api\\Etemplate\\Widget\\Link::ajax_link_titles',[{}], undefined, this, this.link_title_before_send)
					.then(_response => this.link_title_callback(_response));
			}
			return promise;
		},

		/**
		 * Callback to add all current title requests
		 *
		 * @param {object} _params of parameters, only first parameter is used
		 */
		link_title_before_send: function(_params)
		{
			// add all current title-requests
			for(let app in title_queue)
			{
				for(let id in title_queue[app])
				{
					if (typeof _params[0][app] === 'undefined')
					{
						_params[0][app] = [];
					}
					_params[0][app].push(id);
				}
			}
			title_uid = null;	// allow next request to jsonq
		},

		/**
		 * Callback for server response
		 *
		 * @param {object} _response _app => _id => title
		 */
		link_title_callback: function(_response)
		{
			if (typeof _response !== 'object')
			{
				throw "Wrong parameter for egw.link_title_callback!";
			}
			for(let app in _response)
			{
				if (typeof title_cache[app] !== 'object')
				{
					title_cache[app] = {};
				}
				for (let id in _response[app])
				{
					const title = _response[app][id];
					// cache locally
					title_cache[app][id] = title;
					// call callbacks waiting for title of app/id
					if (typeof title_queue[app] !== 'undefined' && typeof title_queue[app][id] !== "undefined")
					{
						for(let i=0; i < title_queue[app][id].length; ++i)
						{
							const callback = title_queue[app][id][i];
							callback.callback.call(callback.context, title);
						}
						delete title_queue[app][id];
					}
				}
			}
		},

		/**
		 * Create quick add selectbox
		 *
		 * @param {DOMnode} _parent parent to create selectbox in
		 */
		link_quick_add: function(_parent)
		{
			// check if quick-add selectbox is already there, only create it again if not
			if (document.getElementById('quick_add_selectbox') || egwIsMobile())
			{
				return;
			}

			// Use node as the trigger
			const parent = document.getElementById(_parent);
			const select = document.createElement('et2-select');
			select.setAttribute('id', 'quick_add_selectbox');
			// Empty label is required to clear value, but we hide it
			select.emptyLabel = "Select";
			select.placement = "bottom";
			parent.append(select);
			const plus = parent.querySelector("span");
			plus.addEventListener("click", () => {
				select.show();
			});

			// bind change handler
			select.addEventListener('change', () =>
			{
				if (select.value)
				{
					this.open('', select.value, 'add', {}, undefined, select.value, true);
				}
				select.value = '';
			});
			// need to load common translations for app-names
			this.langRequire(window, [{app: 'common', lang: this.preference('lang')}], () =>
			{
				let options = [];
				const apps = this.link_app_list('add');
				for(let app in apps)
				{
					if (this.link_get_registry(app, 'no_quick_add'))
					{
						continue;
					}
					options.push({
						value: app,
						label: this.lang(this.link_get_registry(app, 'entry') || apps[app]),
						icon: this.image('navbar', app)
					});
				}
				select.select_options = options;

				select.updateComplete.then(() =>
				{
					// Adjust popup positioning to account for hidden select parts
					select.select.popup.position = "top end";
					select.select.popup.sync = "";
					select.select.popup.distance = -32;
				});
			});
		},

		/**
		 * Check if a mimetype is editable
		 *
		 * Check mimetype & user preference
		 */
		isEditable: function (mime)
		{
			if (!mime)
			{
				return false;
			}
			let fe = this.file_editor_prefered_mimes(mime);
			if (!fe || !fe.mime || fe && fe.mime && !fe.mime[mime])
			{
				return false;
			}
			return ['edit'].indexOf(fe.mime[mime].name) !== -1;
		},

		/**
		 * Check if a mimetype is openable in Collabora
		 * (without needing to have Collabora JS loaded)
		 *
		 * @param mime
		 *
		 * @return string|false
		 */
		isCollaborable: function (mime)
		{
			if (typeof this.user('apps')['collabora'] == "undefined")
			{
				return false;
			}

			// Additional check to see if Collabora can open the file at all, not just edit it
			let fe = this.file_editor_prefered_mimes(mime);
			if (fe && fe.mime && fe.mime[mime] && fe.mime[mime].name || this.isEditable(mime))
			{
				return fe.mime[mime].name;
			}
		}
	}
});

/**
 * EGroupware clientside API: opening of windows, popups or application entries
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 * @version $Id$
 */

/**
 * @augments Class
 * @param {object} _egw
 * @param {DOMwindow} _wnd
 */
egw.extend('open', egw.MODULE_WND_LOCAL, function(_egw, _wnd)
{
	"use strict";

	/**
	 * Magic handling for mailto: uris using mail application.
	 *
	 * We check for open compose windows and add the address in as specified in
	 * the URL.  If there are no open compose windows, a new one is opened.  If
	 * there are more than one open compose window, we prompt for which one to
	 * use.
	 *
	 * The user must have set the 'Open EMail addresses in external mail program' preference
	 * to No, otherwise the browser will handle it.
	 *
	 * @param {String} uri
	 */
	function mailto(uri)
	{
		// Parse uri into a map
		var match = [], index;
		var mailto = uri.match(/^mailto:([^?]+)/) || [];
		var hashes = uri.slice(uri.indexOf('?') + 1).split('&');
		for(var i = 0; i < hashes.length; i++)
		{
			index = hashes[i].replace(/__AMPERSAND__/g, '&').split('=');
			match.push(index[0]);
			match[index[0]] = index[1];
		}
		if (mailto[1]) mailto[1] = mailto[1].replace(/__AMPERSAND__/g, '&');
		var content = {
			to: mailto[1] || [],
			cc: match['cc']	|| [],
			bcc: match['bcc'] || []
		};

		// No CSV, split them here and pay attention to quoted commas
		const split_regex = /("[^"]*" <[^>]*>)|("[^"]*")|('[^']*')|([^,]+)/g;
		for (let index in content)
		{
			if (typeof content[index] == "string")
			{
				const matches = content[index].match(split_regex);
				content[index] = matches ?? content[index];
			}
		}

		// Encode html entities in the URI, otherwise server XSS protection won't
		// allow it to pass, because it may get mistaken for some forbidden tags,
		// e.g., "Mathias <mathias@example.com>" the first part of email "<mathias"
		// including "<" would get mistaken for <math> tag, and server will cut it off.
		uri = uri.replace(/</g,'&lt;').replace(/>/g,'&gt;');

		egw.openWithinWindow ("mail", "setCompose", content, {'preset[mailto]':uri}, /mail_compose.compose/);

		for (var index in content)
		{
			if (content[index].length > 0)
			{
				var cLen = content[index];
				egw.message(egw.lang('%1 email(s) added into %2', cLen.length, egw.lang(index)));
				return;
			}
		}

	}
	return {
		/**
		 * View an EGroupware entry: opens a popup of correct size or redirects window.location to requested url
		 *
		 * Examples:
		 * - egw.open(123,'infolog') or egw.open('infolog:123') opens popup to edit or view (if no edit rights) infolog entry 123
		 * - egw.open('infolog:123','timesheet','add') opens popup to add new timesheet linked to infolog entry 123
		 * - egw.open(123,'addressbook','view') opens addressbook view for entry 123 (showing linked infologs)
		 * - egw.open('','addressbook','list',{ search: 'Becker' }) opens list of addresses containing 'Becker'
		 *
		 * @param {string}|int|object id_data either just the id or if app=="" "app:id" or object with all data
		 * 	to be able to open files you need to give: (mine-)type, path or id, app2 and id2 (path=/apps/app2/id2/id"
		 * @param {string} app app-name or empty (app is part of id)
		 * @param {string} type default "edit", possible "view", "view_list", "edit" (falls back to "view") and "add"
		 * @param {object|string} extra extra url parameters to append as object or string
		 * @param {string} target target of window to open
		 * @param {string} target_app target application to open in that tab
		 * @param {boolean} _check_popup_blocker TRUE check if browser pop-up blocker is on/off, FALSE no check
		 * - This option only makes sense to be enabled when the open_link requested without user interaction
		 * @memberOf egw
		 *
		 * @return {object|void} returns object for given specific target like '_tab'
		 */
		open: function(id_data, app, type, extra, target, target_app, _check_popup_blocker)
		{
			// Log for debugging purposes - special log tag 'navigation' always
			// goes in user log, if user log is enabled
			egw.debug("navigation",
				"egw.open(id_data=%o, app=%s, type=%s, extra=%o, target=%s, target_app=%s)",
				id_data,app,type,extra,target,target_app
			);

			var id;
			if(typeof target === 'undefined')
			{
				target = '_blank';
			}
			if (!app)
			{
				if (typeof id_data != 'object')
				{
					var app_id = id_data.split(':',2);
					app = app_id[0];
					id = app_id[1];
				}
				else
				{
					app = id_data.app;
					id = id_data.id;
					if(typeof id_data.type != 'undefined')
					{
						type = id_data.type;
					}
				}
			}
			else if (app != 'file')
			{
				id = id_data;
				id_data = { 'id': id, 'app': app, 'extra': extra };
			}
			var url;
			var popup;
			var params;
			if (app == 'file')
			{
				url = this.mime_open(id_data);
				if (typeof url == 'object')
				{
			 		if(typeof url.mime_popup != 'undefined')
					{
						popup = url.mime_popup;
						delete url.mime_popup;
					}
			 		if(typeof url.mime_target != 'undefined')
					{
						target = url.mime_target;
						delete url.mime_target;
					}
					if (typeof url.url == 'string')
					{
						url = url.url;
					}
					else
					{
						params = url;
						url = '/index.php';
					}
				}
			}
			else
			{
				var app_registry = this.link_get_registry(app);

				if (!app || !app_registry)
				{
					alert('egw.open() app "'+app+'" NOT defined in link registry!');
					return;
				}
				if (typeof type == 'undefined') type = 'edit';
				if (type == 'edit' && typeof app_registry.edit == 'undefined') type = 'view';
				if (typeof app_registry[type] == 'undefined')
				{
					alert('egw.open() type "'+type+'" is NOT defined in link registry for app "'+app+'"!');
					return;
				}
				url = '/index.php';
				if(typeof app_registry[type] === 'object')
				{
					// Copy, not get a reference, or we'll change the registry
					params = jQuery.extend({},app_registry[type]);
				}
				else if (typeof app_registry[type] === 'string' &&
					(app_registry[type].substr(0, 11) === 'javascript:' || app_registry[type].substr(0, 4) === 'app.'))
				{
					// JavaScript, just pass it on
					url = app_registry[type];
					params = {};
				}
				if (type == 'view' || type == 'edit')	// add id parameter for type view or edit
				{
					params[app_registry[type+'_id']] = id;
				}
				else if (type == 'add' && id)	// add add_app and app_id parameters, if given for add
				{
					var app_id = id.split(':',2);
					params[app_registry.add_app] = app_id[0];
					params[app_registry.add_id] = app_id[1];
				}

				if (typeof extra == 'string')
				{
					url += '?'+extra;
				}
				else if (typeof extra == 'object')
				{
					jQuery.extend(params, extra);
				}
				popup = app_registry[type+'_popup'];
			}
			if (url.substr(0, 11) === 'javascript:')
			{
				// Add parameters into javascript
				url = 'javascript:var params = '+ JSON.stringify(params) + '; '+ url.substr(11);
			}
			// app.<appname>.<method>: call app method direct with parameter object as first parameter
			else if (url.substr(0, 4) === 'app.')
			{
				return this.callFunc(url, params);
			}
			else
			{
				url = this.link(url, params);
			}
			if (target == '_tab') return {url: url};
			if (type == 'view'  && params && params.target == 'tab') {
				return this.openTab(params[app_registry['view_id']], app, type, params, {
					id: params[app_registry['view_id']] + '-' + this.appName,
					icon: params['icon'],
					displayName: id_data['title'] + " (" + egw.lang(this.appName) + ")",
				});
			}
			return this.open_link(url, target, popup, target_app, _check_popup_blocker);
		},

		/**
		 * View an EGroupware entry: opens a framework tab for the given app entry
		 *
		 * @param {string}|int|object _id either just the id or if app=="" "app:id" or object with all data
		 * @param {string} _app app-name or empty (app is part of id)
		 * @param {string} _type default "edit", possible "view", "view_list", "edit" (falls back to "view") and "add"
		 * @param {object|string} _extra extra url parameters to append as object or string
		 * @param {object} _framework_app framework app attributes e.g. title or displayName
		 * @return {string} appname of new tab
		  */
		openTab: function(_id, _app, _type, _extra, _framework_app)
		{
			if (_wnd.framework && _wnd.framework.tabLinkHandler)
			{
				var data = this.open(_id, _app, _type, _extra, "_tab", false);
				// Use framework's link handler
				return _wnd.framework.tabLinkHandler(data.url, _framework_app);
			}
			else
			{
				this.open(_id, _app, _type, _extra);
			}
		},

		/**
		 * Open a link, which can be either a menuaction, a EGroupware relative url or a full url
		 *
		 * @param {string} _link menuaction, EGroupware relative url or a full url (incl. "mailto:" or "javascript:")
		 * @param {string} _target optional target / window name
		 * @param {string} _popup widthxheight, if a popup should be used
		 * @param {string} _target_app app-name for opener
		 * @param {boolean} _check_popup_blocker TRUE check if browser pop-up blocker is on/off, FALSE no check
		 * - This option only makes sense to be enabled when the open_link requested without user interaction
		 * @param {string} _mime_type if given, we check if any app has registered a mime-handler for that type and use it
		 */
		open_link: function(_link, _target, _popup, _target_app, _check_popup_blocker, _mime_type)
		{
			// Log for debugging purposes - don't use navigation here to avoid
			// flooding log with details already captured by egw.open()
			egw.debug("log",
				"egw.open_link(_link=%s, _target=%s, _popup=%s, _target_app=%s)",
				_link,_target,_popup,_target_app
			);
			//Check browser pop-up blocker
			if (_check_popup_blocker)
			{
				if (this._check_popupBlocker(_link, _target, _popup, _target_app)) return;
			}
			var url = _link;
			if (url.indexOf('javascript:') == 0)
			{
				(new Function(url.substr(11)))();
				return;
			}
			if (url.indexOf('mailto:') == 0)
			{
				return mailto(url);
			}
			// link is not necessary an url, it can also be a menuaction!
			if (url.indexOf('/') == -1 && url.split('.').length >= 3 &&
				!(url.indexOf('mailto:') == 0 || url.indexOf('/index.php') == 0 || url.indexOf('://') != -1))
			{
				url = "/index.php?menuaction="+url;
			}
			// append the url to the webserver url, if not already contained or empty
			if (url[0] == '/' && this.webserverUrl && this.webserverUrl != '/' && url.indexOf(this.webserverUrl+'/') != 0)
			{
				url = this.webserverUrl + url;
			}
			var mime_info = _mime_type ? this.get_mime_info(_mime_type, _target_app) : undefined;
			if (mime_info && (mime_info.mime_url || mime_info.mime_data))
			{
				var data = {};
				for(var attr in mime_info)
				{
					switch(attr)
					{
						case 'mime_popup':
							_popup = mime_info.mime_popup;
							break;
						case 'mime_target':
							_target = mime_info.mime_target;
							break;
						case 'mime_type':
							data[mime_info.mime_type] = _mime_type;
							break;
						case 'mime_data':
							data[mime_info[attr]] = _link;
							break;
						case 'mime_url':
							data[mime_info[attr]] = url;
							break;
						default:
							data[attr] = mime_info[attr];
							break;
					}
				}
				url = egw.link('/index.php', data);
			}
			else if (mime_info)
			{
				if (mime_info.mime_popup) _popup = mime_info.mime_popup;
				if (mime_info.mime_target) _target = mime_info.mime_target;
			}

			if (_popup && _popup.indexOf('x') > 0)
			{
				var w_h = _popup.split('x');
				var popup_window = this.openPopup(url, w_h[0], w_h[1], _target && _target != _target_app ? _target : '_blank', _target_app, true);

				// Remember which windows are open
				egw().storeWindow(_target_app, popup_window);

				return popup_window;
			}
			else if ((typeof _target == 'undefined' || _target == '_self' || typeof this.link_app_list()[_target] != "undefined"))
			{
				if(_target == '_self')
				{
					// '_self' isn't allowed, but we can handle it
					_target = undefined;
				}
				// Use framework's link handler, if present
				return this.link_handler(url,_target);
			}
			else
			{
				// No mime type registered, set target properly based on browsing environment
				if (_target == '_browser')
				{
					_target = egwIsMobile()?'_self':'_blank';
				}
				_target = _target == '_phonecall' && _popup && _popup.indexOf('x') < 0 ? _popup:_target;
				return _wnd.open(url, _target);
			}
		},

		/**
		 * Opens a menuaction in an Et2Dialog instead of a popup
		 *
		 * Please note:
		 * This method does NOT (yet) work in popups, only in the main EGroupware window!
		 * For popups you have to use the app.ts method openDialog(), which creates the dialog in the correct window / popup.
		 *
		 * @param string _menuaction
		 * @return Promise<Et2Dialog>
		 */
		openDialog: function(_menuaction)
		{
			let resolver;
			let rejector;
			const dialog_promise = new Promise((resolve, reject) =>
			{
				resolver = value => resolve(value);
				rejector = reason => reject(reason);
			});
			let request = egw.json(_menuaction.match(/^([^.:]+)/)[0] + '.jdots_framework.ajax_exec.template.' + _menuaction,
				['index.php?menuaction=' + _menuaction, true], _response =>
				{
					if (Array.isArray(_response) && typeof _response[0] === 'string')
					{
						let dialog = jQuery(_response[0]).appendTo(_wnd.document.body);
						if (dialog.length > 0 && dialog.get(0))
						{
							resolver(dialog.get(0));
						}
						else
						{
							console.log("Unable to add dialog with dialogExec('" + _menuaction + "')", _response);
							rejector(new Error("Unable to add dialog"));
						}
					}
					else
					{
						console.log("Invalid response to dialogExec('" + _menuaction + "')", _response);
						rejector(new Error("Invalid response to dialogExec('" + _menuaction + "')"));
					}
				}).sendRequest();
			return dialog_promise;
		},

		/**
		 * Open a (centered) popup window with given size and url
		 *
		 * @param {string} _url
		 * @param {number} _width
		 * @param {number} _height
		 * @param {string} _windowName or "_blank"
		 * @param {string|boolean} _app app-name for framework to set correct opener or false for current app
		 * @param {boolean} _returnID true: return window, false: return undefined
		 * @param {string} _status "yes" or "no" to display status bar of popup
		 * @param {boolean} _skip_framework
		 * @returns {DOMWindow|undefined}
		 */
		openPopup: function(_url, _width, _height, _windowName, _app, _returnID, _status, _skip_framework)
		{
			// Log for debugging purposes
			egw.debug("navigation", "openPopup(%s, %s, %s, %o, %s, %s)",_url,_windowName,_width,_height,_status,_app);

			if (_height == 'availHeight') _height = this.availHeight();

			// if we have a framework and we use mobile template --> let framework deal with opening popups
			if (!_skip_framework && _wnd.framework)
			{
				return _wnd.framework.openPopup(_url, _width, _height, _windowName, _app, _returnID, _status, _wnd);
			}

			if (typeof(_app) == 'undefined') _app = false;
			if (typeof(_returnID) == 'undefined') _returnID = false;

			var $wnd = jQuery(egw.top);
			var positionLeft = ($wnd.outerWidth()/2)-(_width/2)+_wnd.screenX;
			var positionTop  = ($wnd.outerHeight()/2)-(_height/2)+_wnd.screenY;

			// IE fails, if name contains eg. a dash (-)
			if (navigator.userAgent.match(/msie/i)) _windowName = !_windowName ? '_blank' : _windowName.replace(/[^a-zA-Z0-9_]+/,'');

			var windowID = _wnd.open(_url, _windowName || '_blank', "width=" + _width + ",height=" + _height +
				",screenX=" + positionLeft + ",left=" + positionLeft + ",screenY=" + positionTop + ",top=" + positionTop +
				",location=no,menubar=no,directories=no,toolbar=no,scrollbars=yes,resizable=yes,status="+_status);

			// inject egw object
			if (windowID) windowID.egw = _wnd.egw;

			// returning something, replaces whole window in FF, if used in link as "javascript:egw_openWindowCentered2()"
			if (_returnID !== false) return windowID;
		},

		/**
		 * Get available height of screen
		 */
		availHeight: function()
		{
			return screen.availHeight < screen.height ?
				(navigator.userAgent.match(/windows/ig)? screen.availHeight -100:screen.availHeight) // Seems chrome not counting taskbar in available height
				: screen.height - 100;
		},

		/**
		 * Use frameworks (framed template) link handler to open a url
		 *
		 * @param {string} _url
		 * @param {string} _target
		 */
		link_handler: function(_url, _target)
		{
			// if url is supposed to open in admin, use admins loader to open it in it's own iframe
			// (otherwise there's no tree and sidebox!)
			if (_target === 'admin' && !_url.match(/menuaction=admin\.admin_ui\.index/))
			{
				_url = _url.replace(/menuaction=([^&]+)/, 'menuaction=admin.admin_ui.index&load=$1');
			}
			if (_wnd.framework)
			{
				_wnd.framework.linkHandler(_url, _target);
			}
			else
			{
				_wnd.location.href = _url;
			}
		},

		/**
		 * Close current window / popup
		 */
		close: function()
		{
			if (_wnd.framework && typeof _wnd.framework.popup_close == "function")
			{
				_wnd.framework.popup_close(_wnd);
			}
			else
			{
				_wnd.close();
			}
		},

		/**
		 * Check if browser pop-up blocker is on/off
		 *
		 * @param {string} _link menuaction, EGroupware relative url or a full url (incl. "mailto:" or "javascript:")
		 * @param {string} _target optional target / window name
		 * @param {string} _popup widthxheight, if a popup should be used
		 * @param {string} _target_app app-name for opener
		 *
		 * @return boolean returns false if pop-up blocker is off
		 * - returns true if pop-up blocker is on,
		 * - and re-call the open_link with provided parameters, after user interaction.
		 */
		_check_popupBlocker: function(_link, _target, _popup, _target_app)
		{
			var popup = window.open("","",'top='+(screen.height/2)+',left='+(screen.width/2)+',width=1,height=1,menubar=no,resizable=yes,scrollbars=yes,status=no,toolbar=no,dependent=yes');

			if (!popup||popup == 'undefined'||popup == null)
			{
				Et2Dialog.show_dialog(function ()
					{
						window.egw.open_link(_link, _target, _popup, _target_app);
					}, egw.lang("The browser popup blocker is on. Please click on OK button to see the pop-up.\n\nIf you would like to not see this message for the next time, allow your browser pop-up blocker to open popups from %1", window.location.hostname),
					"Popup Blocker Warning", {}, Et2Dialog.BUTTONS_OK, Et2Dialog.WARNING_MESSAGE);
				return true;
			}
			else
			{
				popup.close();
				return false;
			}
		},

		/**
		 * This function helps to append content/ run commands into an already
		 * opened popup window. Popup windows now are getting stored in framework
		 * object and can be retrieved/closed from framework.
		 *
		 * @param {string} _app name of application to be requested its popups
		 * @param {string} _method application method implemented in app.js
		 * @param {object} _content content to be passed to method
		 * @param {string|object} _extra url or object of extra
		 * @param {regex} _regexp regular expression to get specific popup with matched url
		 * @param {boolean} _check_popup_blocker TRUE check if browser pop-up blocker is on/off, FALSE no check
		 */
		openWithinWindow: function (_app, _method, _content, _extra, _regexp, _check_popup_blocker)
		{
			var popups = window.framework.popups_get(_app, _regexp);

			var openUp = function (_app, _extra) {

				var len = 0;
				if (typeof _extra == "string")
				{
					len = _extra.length;
				}
				else if (typeof _extra == "object")
				{
					for (var i in _extra)
					{
						if (jQuery.isArray(_extra[i]))
						{
							var tmp = '';
							for (var j in _extra[i])
							{
								tmp += i+'[]='+_extra[i][j]+'&';

							}
							len += tmp.length;
						}
						else if(_extra[i])
						{
							len += _extra[i].length;
						}
					}
				}

				// According to microsoft, IE 10/11 can only accept a url with 2083 characters
				// therefore we need to send request to compose window with POST method
				// instead of GET. We create a temporary <Form> and will post emails.
				// ** WebServers and other browsers also have url length limit:
				// Firefox:~ 65k, Safari:80k, Chrome: 2MB, Apache: 4k, Nginx: 4k
				if (len > 2083)
				{
					var popup = egw.open('','mail','add','','compose__','mail', _check_popup_blocker);
					var $tmpForm = jQuery(document.createElement('form'));
					var $tmpSubmitInput = jQuery(document.createElement('input')).attr({type:"submit"});
					for (var i in _extra)
					{
						if (jQuery.isArray(_extra[i]))
						{
							$tmpForm.append(jQuery(document.createElement('input')).attr({name:i, type:"text", value: JSON.stringify(_extra[i])}));
						}
						else
						{
							$tmpForm.append(jQuery(document.createElement('input')).attr({name:i, type:"text", value: _extra[i]}));
						}
					}

					// Set the temporary form's attributes
					$tmpForm.attr({target:popup.name, action:"index.php?menuaction=mail.mail_compose.compose", method:"post"})
							.append($tmpSubmitInput).appendTo('body');
					$tmpForm.submit();
					// Remove the form after submit
					$tmpForm.remove();
				}
				else
				{
					egw.open('', _app, 'add', _extra, _app, _app, _check_popup_blocker);
				}
			};
			for(var i = 0; i < popups.length; i++)
			{
				if(popups[i].closed)
				{
					window.framework.popups_grabage_collector();
				}
			}

			const pref_name = "mail_add_address_new_popup";
			const new_dialog_pref = egw.preference(pref_name, "common");

			if (popups.length == 0 || new_dialog_pref == "new")
			{
				return openUp(_app, _extra);
			}
			if (new_dialog_pref == "add" && popups.length == 1)
			{
				try
				{
					popups[0].app[_app][_method](popups[0], _content);
				}
				catch (e)
				{
					window.setTimeout(function ()
					{
						openUp(_app, _extra);
					});
				}
				return;
			}

			var buttons = [
				{label: this.lang("Add"), id: "add", "class": "ui-priority-primary", "default": true},
				{label: this.lang("Cancel"), id: "cancel"}
			];

			// Fill dialog options
			var c = [];
			for (var i = 0; i < popups.length; i++)
			{
				c.push({label: popups[i].document.title || this.lang(_app), index: i});
			}
			c.push({label: this.lang("New %1", egw.link_get_registry(_app, "entry")), index: "new"});

			// Set initial value
			switch (new_dialog_pref)
			{
				case "new":
					c.index = "new";
					break;
				default:
				case "add":
					c.index = 0;
					break;
			}
			let dialog = new Et2Dialog(this.app_name());
			dialog.transformAttributes({
				callback: function (_button_id, _value)
				{
					if (_value.remember)
					{
						// Remember and do not ask again (if they chose new)
						egw.set_preference("common", pref_name, _value.remember && _value.grid.index == "new" ? "new" : "add");
					}
					if (_value && _value.grid)
					{
						switch (_button_id)
						{
							case "add":
								if (_value.grid.index == "new")
								{
									return openUp(_app, _extra);
								}
								popups[_value.grid.index].app[_app][_method](popups[_value.grid.index], _content);
								return;
							case "cancel":
						}
					}
				},
				title: this.lang("Select an opened dialog"),
				buttons: buttons,
				value: {content: {grid: c}},
				template: this.webserverUrl + '/api/templates/default/promptOpenedDialog.xet?1',
				resizable: false
			});
			document.body.appendChild(dialog);
		}
	};
});


// Add protocol handler as an option if mail handling is not forced so mail can handle mailto:
/* Not working consistantly yet
jQuery(function() {
try {
	if(egw.user('apps').mail && (egw.preference('force_mailto','addressbook')||true) != '0')
	{
		var params = egw.link_get_registry('mail','add');
		if(params)
		{
			params['preset[mailto]'] = ''; // %s, but egw.link will encode it
			navigator.registerProtocolHandler("mailto",egw.link('/index.php', params)+'%s', egw.lang('mail'));
		}
	}
} catch (e) {}
});
*/

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 */

egw.extend('user', egw.MODULE_GLOBAL, function()
{
	"use strict";

	/**
	 * Data about current user
	 *
	 * @access: private, use egw.user(_field) or egw.app(_app)
	 */
	let userData = {apps: {}};

	/**
	 * Client side cache of accounts user has access to
	 * Used by account select widgets
	 */
	let accountStore = {
		// Filled by AJAX when needed
		//accounts: {},
		//groups: {},
		//owngroups: {}
	};

	/**
	 * Clientside cache for accountData calls
	 */
	let accountData = {};
	let resolveGroup = {};

	// Hold in-progress request to avoid making more
	let request = null;

	return {
		/**
		 * Set data of current user
		 *
		 * @param {object} _data
		 * @param {boolean} _need_clone _data need to be cloned, as it is from different window context
		 *	and therefore will be inaccessible in IE, after that window is closed
		 */
		set_user: function (_data, _need_clone)
		{
			userData = _need_clone ? jQuery.extend(true, {}, _data) : _data;
		},

		/**
		 * Get data about current user
		 *
		 * @param {string} _field
		 * - 'account_id','account_lid','person_id','account_status','memberships'
		 * - 'account_firstname','account_lastname','account_email','account_fullname','account_phone'
		 * - 'apps': object with app => data pairs the user has run-rights for
		 * @return {string|array|null}
		 */
		user: function (_field)
		{
			return userData[_field];
		},

		/**
		 * Return data of apps the user has rights to run
		 *
		 * Can be used the check of run rights like: if (egw.app('addressbook')) { do something if user has addressbook rights }
		 *
		 * @param {string} _app
		 * @param {string} _name attribute to return, default return whole app-data-object
		 * @return object|string|null null if not found
		 */
		app: function(_app, _name)
		{
			return typeof _name == 'undefined' || typeof userData.apps[_app] == 'undefined' ?
				userData.apps[_app] : userData.apps[_app][_name];
		},

		/**
		 * Same as app(), but use the translated app-name / title
		 *
		 * @param {string} _title
		 * @param {string} _name attribute to return, default return whole app-data-object
		 */
		appByTitle: function(_title, _name)
		{
			for(const app in userData.apps)
			{
				if (userData.apps[app].title === _title)
				{
					return typeof _name == 'undefined' || typeof userData.apps[app] == 'undefined' ?
						userData.apps[app] : userData.apps[app][_name];
				}
			}
		},

		/**
		 * Get a list of accounts the user has access to
		 * The list is filtered by type, one of 'accounts','groups','both', 'owngroups'
		 *
		 * @param {string} type
		 * @returns {Promise<{value:string,label:string,icon?:string}[]>}
		 */
		accounts: function (type)
		{
			if (typeof type === 'undefined')
			{
				type = 'accounts';
			}

			if (request !== null)
			{
				return request.then(() =>
				{
					return this.accounts(type)
				});
			}
			if (jQuery.isEmptyObject(accountStore))
			{
				const cache_it = data =>
				{
					let types = ["accounts", "groups", "owngroups"];
					for (let t of types)
					{
						if (typeof data[t] === "object")
						{
							accountStore[t] = (Array.isArray(data[t]) ? data[t]:Object.values(data[t]) ?? []).map(a => {a.value = ""+a.value; return a});
						}
					}
				};
				request = egw.request("EGroupware\\Api\\Framework::ajax_user_list", []).then(_data =>
				{
					cache_it(_data);
					request = null;
					return this.accounts(type);
				});
				return request;
			}
			let result = [];
			if (type === 'both')
			{
				result = [...Object.values(accountStore.accounts), ...Object.values(accountStore.groups)];
			}
			else
			{
				result = [...Object.values(accountStore[type])];
			}
			return Promise.resolve(result);
		},

		/**
		 * Get account-infos for given numerical _account_ids
		 *
		 * @param {int|int[]} _account_ids
		 * @param {string} _field default 'account_email'
		 * @param {boolean} _resolve_groups true: return attribute for all members, false: return attribute of group
		 * @param {function|undefined} _callback deprecated, use egw.accountDate(...).then(data => _callback.bind(_context)(data))
		 * @param {object|undefined} _context deprecated, see _context
		 * @return {Promise} resolving to object { account_id => value, ... }
		 */
		accountData: function(_account_ids, _field, _resolve_groups, _callback, _context)
		{
			if (!_field) _field = 'account_email';
			if (!Array.isArray(_account_ids)) _account_ids = [_account_ids];

			// check our cache or current user first
			const data = {};
			let pending = false;
			for(let i=0; i < _account_ids.length; ++i)
			{
				const account_id = _account_ids[i];

				if (account_id == userData.account_id)
				{
					data[account_id] = userData[_field];
				}
				else if ((!_resolve_groups || account_id > 0) && typeof accountData[account_id] !== 'undefined' &&
					typeof accountData[account_id][_field] !== 'undefined')
				{
					data[account_id] = accountData[account_id][_field];
					pending = pending || data[account_id] instanceof Promise;
				}
				else if (_resolve_groups && account_id < 0 && typeof resolveGroup[account_id] !== 'undefined' &&
					typeof resolveGroup[account_id][_field] != 'undefined')
				{
					// Groups are resolved on the server, but then the response
					// is cached, so we can re-resolve it locally
					for(let id in resolveGroup[account_id][_field])
					{
						data[id] = resolveGroup[account_id][_field][id];
						pending = pending || data[id] instanceof Promise;
					}
				}
				else
				{
					continue;
				}
				_account_ids.splice(i--, 1);
			}

			let promise;
			// something not found in cache --> ask server
			if (_account_ids.length)
			{
				promise = egw.request('EGroupware\\Api\\Framework::ajax_account_data',[_account_ids, _field, _resolve_groups]).then(_data =>
				{
					for(let account_id in _data)
					{
						if (typeof accountData[account_id] === 'undefined')
						{
							accountData[account_id] = {};
						}
						data[account_id] = accountData[account_id][_field] = _data[account_id];
					}
					// If resolving for 1 group, cache the whole answer too
					// (More than 1 group, we can't split to each group)
					if(_resolve_groups && _account_ids.length === 1 && _account_ids[0] < 0)
					{
						const group_id = _account_ids[0];
						if (typeof resolveGroup[group_id] === 'undefined')
						{
							resolveGroup[group_id] = {};
						}
						resolveGroup[group_id][_field] = _data;
					}
					return data;
				});

				// store promise, in case someone asks while the request is pending, to not query the server again
				_account_ids.forEach(account_id =>
				{
					if (_resolve_groups && account_id < 0) return;	// we must NOT cache the promise for account_id!

					if (typeof accountData[account_id] === 'undefined')
					{
						accountData[account_id] = {};
					}
					accountData[account_id][_field] = promise.then(function(_data)
					{
						const result = {};
						result[this.account_id] = _data[this.account_id];
						return result;
					}.bind({ account_id: account_id }));
				});
				if (_resolve_groups && _account_ids.length === 1 && _account_ids[0] < 0)
				{
					resolveGroup[_account_ids[0]] = promise;
				}
			}
			else
			{
				promise = Promise.resolve(data);
			}

			// if we have any pending promises, we need to resolve and merge them
			if (pending)
			{
				promise = promise.then(_data =>
				{
					const promises = [];
					for (let account_id in _data)
					{
						if (_data[account_id] instanceof Promise)
						{
							promises.push(_data[account_id]);
						}
					}
					return Promise.all(promises).then(_results =>
					{
						_results.forEach(result =>
						{
							for (let account_id in result)
							{
								_data[account_id] = result[account_id];
							}
						});
						return _data;
					});
				});
			}

			// if deprecated callback is given, call it with then
			if (typeof _callback === 'function')
			{
				promise = promise.then(_data =>
				{
					_callback.bind(_context)(_data);
					return _data;
				});
			}
			return promise;
		},

		/**
		 * Set account data.  This one can be called from the server to pre-fill the cache.
		 *
		 * @param {Array} _data
		 * @param {String} _field
		 */
		set_account_cache: function(_data, _field)
		{
			for(let account_id in _data)
			{
				if (typeof accountData[account_id] === 'undefined')
				{
					accountData[account_id] = {};
				}
				accountData[account_id][_field] = _data[account_id];
			}
		},

		/**
		 * Set specified account-data of selected user in an other widget
		 *
		 * Used eg. in template as: onchange="egw.set_account_data(widget, 'target', 'account_email')"
		 *
		 * @param {et2_widget} _src_widget widget to select the user
		 * @param {string} _target_name name of widget to set the data
		 * @param {string} _field name of data to set eg. "account_email" or "{account_fullname} <{account_email}>"
		 */
		set_account_data: function(_src_widget, _target_name, _field)
		{
			const user = _src_widget.get_value();
			const target = _src_widget.getRoot().getWidgetById(_target_name);
			const field = _field;

			if (user && target)
			{
				egw.accountData(user, _field, false, function(_data)
				{
					let data;
					if (field.indexOf('{') == -1)
					{
						data = _data[user];
						target.set_value(data);
					}
					else
					{
						data = field;

						/**
						 * resolve given data whilst the condition met
						 */
						const resolveData = function(_d, condition, action) {
							const whilst = function (_d) {
								return condition(_d) ? action(condition(_d)).then(whilst) : Promise.resolve(_d);
							};
							return whilst(_d);
						};

						/**
						 * get data promise
						 */
						const getData = function(_match)
						{
							const match = _match;
							return new Promise(function(resolve)
							{
							  egw.accountData(user, match, false, function(_d)
								{
									data = data.replace(/{([^}]+)}/, _d[user]);
									resolve(data);
								});
							});
						};

						// run resolve data
						resolveData(data, function(_d) {
							const r = _d.match(/{([^}]+)}/);
							return r && r.length > 0 ? r[1] : r;
						},
						getData).then(function(data){
							target.set_value(data);
						});
					}
				});
			}
		},

		/**
		 * Invalidate client-side account cache
		 *
		 * For _type == "add" we invalidate the whole cache currently.
		 *
		 * @param {number} _id nummeric account_id, !_id will invalidate whole cache
		 * @param {string} _type "add", "delete", "update" or "edit"
		 */
		invalidate_account: function(_id, _type)
		{
			if (_id)
			{
				delete accountData[_id];
				delete resolveGroup[_id];
			}
			else
			{
				accountData = {};
				resolveGroup = {};
			}
			if (jQuery.isEmptyObject(accountStore)) return;

			switch(_type)
			{
				case 'delete':
				case 'edit':
				case 'update':
					if (_id)
					{
						const store = _id < 0 ? accountStore.groups : accountStore.accounts;
						for(let i=0; i < store.length; ++i)
						{
							if (store && typeof store[i] != 'undefined' && _id == store[i].value)
							{
								if (_type === 'delete')
								{
									delete(store[i]);
								}
								else
								{
									this.link_title('api-accounts', _id, function(_label)
									{
										store[i].label = _label;
										if (_id < 0)
										{
											for(let j=0; j < accountStore.owngroups.length; ++j)
											{
												if (_id == accountStore.owngroups[j].value)
												{
													accountStore.owngroups[j].label = _label;
													break;
												}
											}
										}
									}, this, true);	// true = force reload
								}
								break;
							}
						}
						break;
					}
					// fall through
				default:
					accountStore = {};
					break;
			}
		}
	};
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 * @version $Id$
 */

egw.extend('config', egw.MODULE_GLOBAL, function()
{
	"use strict";

	/**
	 * Clientside config
	 *
	 * @access: private, use egw.config(_name, _app="phpgwapi")
	 */
	var configs = {};


	/**
	 * register our mail as mailto protocol handler (only for main-page, FF seems to pop it up constantly, if we do so in an iframe)
 	 */
	function install_mailto_handler()
	{
		if (document.location.href.match(/(\?|&)cd=yes(&|$)/) &&
			!window.sessionStorage.getItem('asked-mailto-handler') &&
			typeof navigator.registerProtocolHandler === 'function')	// eg. Safari 15.5 does NOT implement it
		{
			const _ask_mailto_handler = () => {
				let url = egw_webserverUrl;
				if (url[0] === '/') url = document.location.protocol+'//'+document.location.hostname+(url !== '/' ? url : '');
				navigator.registerProtocolHandler('mailto', url+'/index.php?menuaction=mail.mail_compose.compose&preset[mailto]=%s', 'Mail');
				// remember not to ask again for this "session"
				window.sessionStorage.setItem('asked-mailto-handler', 'yes');
			};
			// FF does not support user to opt out of the mailto-handler / have a "Don't ask me again" option,
			// so we add that ourselves here for Firefox only:
			if (navigator.userAgent.match(/firefox/i) && !navigator.userAgent.match(/chrome/i))
			{
				if (window.localStorage.getItem('asked-mailto-handler'))
				{
					return;
				}
				const dialog = window.Et2Dialog;
				if (typeof dialog === 'undefined')
				{
					window.setTimeout(install_mailto_handler.bind(this), 1000);
					return;
				}
				dialog.show_dialog((_button) =>
				{
					switch(_button)
					{
						case dialog.YES_BUTTON:
							_ask_mailto_handler();
							// fall through
						case dialog.NO_BUTTON:
							window.localStorage.setItem('asked-mailto-handler', _button == dialog.YES_BUTTON ? 'answer-was-yes' : 'answer-was-no');
							break;
						case dialog.CANCEL_BUTTON:
							// ask again next session ...
							window.sessionStorage.setItem('asked-mailto-handler', 'yes');
					}
				}, egw.lang('Answering no will not ask you again for this browser.'), egw.lang('Install EGroupware as mail-handler?'),
					undefined, dialog.BUTTONS_YES_NO_CANCEL);
			}
			else
			{
				_ask_mailto_handler();
			}
		}
	}

	return {
		/**
		 * Query clientside config
		 *
		 * @param {string} _name name of config variable
		 * @param {string} _app default "phpgwapi"
		 * @return mixed
		 */
		config: function (_name, _app)
		{
			if (typeof _app == 'undefined') _app = 'phpgwapi';

			if (typeof configs[_app] == 'undefined') return null;

			return configs[_app][_name];
		},

		/**
		 * Set clientside configuration for all apps
		 *
		 * @param {object} _configs
		 * @param {boolean} _need_clone _configs need to be cloned, as it is from different window context
		 *	and therefore will be inaccessible in IE, after that window is closed
		 */
		set_configs: function(_configs, _need_clone)
		{
			configs = _need_clone ? jQuery.extend(true, {}, _configs) : _configs;

			if (this.config('install_mailto_handler') !== 'disabled')
			{
				install_mailto_handler();
			}
		}
	};
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 * @version $Id$
 */

egw.extend('images', egw.MODULE_GLOBAL, function()
{
	"use strict";

	/**
	 * Map to serverside available images for users template-set
	 *
	 * @access: private, use egw.image(_name, _app)
	 */
	let images;

	/**
	 * Mapping some old formats to the newer form, or any other aliasing for mime-types
	 *
	 * Should be in sync with ../inc/class.mime_magic.inc.php
	 */
	const mime_alias_map = {
		'text/vcard': 'text/x-vcard',
		'text/comma-separated-values': 'text/csv',
		'text/rtf': 'application/rtf',
		'text/xml': 'application/xml',
		'text/x-diff': 'text/diff',
		'application/x-jar': 'application/java-archive',
		'application/x-javascript': 'application/javascript',
		'application/x-troff': 'text/troff',
		'application/x-egroupware-etemplate': 'application/xml'
	};

	return {
		/**
		 * Set imagemap, called from /api/images.php
		 *
		 * @param {array|object} _images
		 * @param {boolean} _need_clone _images need to be cloned, as it is from different window context
		 *	and therefore will be inaccessible in IE, after that window is closed
		 */
		set_images: function (_images, _need_clone)
		{
			images = _need_clone ? jQuery.extend(true, {}, _images) : _images;
		},

		/**
		 * Get image URL for a given image-name and application
		 *
		 * @param {string} _name image-name without extension
		 * @param {string} _app application name, default current app of window
		 * @return string with URL of image
		 */
		image: function (_name, _app)
		{
			// For logging all paths tried
			var tries = {};

			if (!images)
			{
				console.log("calling egw.image('"+_name+"', '"+_app+"') before egw.set_images() returning null");
				return null;
			}

			if (typeof _app === 'undefined')
			{
				// If the application name is not given, set it to the name of
				// current application
				_app = this.getAppName();
			}

			// Handle images in appname/imagename format
			if(_name.indexOf('/') > 0)
			{
				var split = _name.match(/^([^/]+)\/(.*)$/);
				// e.g. dhtmlxtree and egw_action are subdirs in image dir, not applications
				if (typeof images[split[1]] !== 'undefined')
				{
					_app = split[1];
					_name = split[2];
				}
			}

			// own instance specific images in vfs have the highest precedence
			tries.vfs = _name;
			if (typeof images.vfs !== 'undefined' && typeof images.vfs[_name] === 'string')
			{
				return this.webserverUrl+images.vfs[_name];
			}
			if (typeof images.global !== 'undefined' && (_name !== 'navbar' || _app === 'api'))
			{
				tries.global = '('+_app+'/)'+_name;
				let replace = images.global[_app+'/'+_name] || images.global[_name];
				if (replace)
				{
					if (typeof images.bootstrap[replace] === 'string')
					{
						return this.webserverUrl+images.bootstrap[replace];
					}
					const parts = replace.split('/');
					if (parts.length > 1)
					{
						_app = parts.shift();
						_name = parts.join('/');
					}
					else
					{
						_name = replace;
					}
				}
			}
			tries[_app + (_app == 'phpgwapi' ? " (current app)" : "")] = _name;
			if (typeof images[_app] !== 'undefined' && typeof images[_app][_name] === 'string')
			{
				return this.webserverUrl+images[_app][_name];
			}
			tries.bootstrap = _name;
			if (typeof images.bootstrap !== 'undefined' && typeof images.bootstrap[_name] === 'string')
			{
				return this.webserverUrl+images.bootstrap[_name];
			}
			tries.api = _name;
			if (typeof images.api !== 'undefined' && typeof images.api[_name] === 'string')
			{
				return this.webserverUrl+images.api[_name];
			}
			// if no match, check if it might contain an extension
			var matches = _name.match(/\.(png|gif|jpg)$/i);
			if (matches)
			{
				return this.image(_name.replace(/.(png|gif|jpg)$/i,''), _app);
			}
			if(matches != null) tries[_app + " (matched)"]= matches;
			egw.debug("log",'egw.image("'+_name+'", "'+_app+'") image NOT found!  Tried ', tries);
			return null;
		},

		/**
		 * Get image url for a given mime-type and option file
		 *
		 * @param {string} _mime
		 * @param {string} _path vfs path to generate thumbnails for images
		 * @param {number} _size defaults to 128 (only supported size currently)
		 * @param {number} _mtime current modification time of file to allow infinit caching as url changes
		 * @returns url of image
		 */
		mime_icon: function(_mime, _path, _size, _mtime)
		{
			if (typeof _size == 'undefined') _size = 128;
			if (!_mime) _mime = 'unknown';
			if (_mime == 'httpd/unix-directory') _mime = 'directory';

			if (typeof _path == 'string' && _mime === 'directory')
			{
				const path_parts = _path.split('/');
				if (path_parts.length === 3 && (path_parts[1] === 'apps' || path_parts[1] === 'templates'))
				{
					_mime = 'egw/'+path_parts[2];
				}
			}

			var type  = _mime.toLowerCase().split('/');
			var image = type[0] == 'egw' ? this.image('navbar',type[1]) : undefined;

			if (image)
			{

			}
			else if (typeof _path == 'string' && (type[0] == 'image' && type[1].match(/^(png|jpe?g|gif|bmp)$/) ||
				type[0] == 'application' && (
					// Open Document
					type[1].indexOf('vnd.oasis.opendocument.') === 0 ||
					// PDF
					type[1] == 'pdf' ||
					// Microsoft
					type[1].indexOf('vnd.openxmlformats-officedocument.') === 0
				)
			))
			{
				var params = { 'path': _path, 'thsize': this.config('link_list_thumbnail') || 64};
				if (_mtime) params.mtime = _mtime;
				image = this.link('/api/thumbnail.php', params);
			}
			// for svg return image itself
			else if (type[0] == 'image' && type[1] == 'svg+xml' && typeof _path == "string")
			{
				image = this.webserverUrl+'/webdav.php'+_path;
			}
			else
			{
				if ((typeof type[1] == 'undefined' || !(image = this.image('mime'+_size+'_'+type[0]+'_'+type[1], 'etemplate')) &&
					!(typeof mime_alias_map[_mime] != 'undefined' && (image=this.mime_icon(mime_alias_map[_mime], _path, _size, _mtime)))) &&
					!(image = this.image('mime'+_size+'_'+type[0], 'etemplate')))
				{
					image = this.image('mime'+_size+'_unknown', 'etemplate');
				}
			}
			return image;
		},

		/**
		 * Create DOM img or svn element depending on url
		 *
		 * @param {string} _url source url
		 * @param {string} _alt alt attribute for img tag
		 * @returns DOM node
		 */
		image_element: function(_url, _alt)
		{
			var icon;
			icon = document.createElement('img');
			if (_url) icon.src = _url;
			if (_alt) icon.alt = _alt;
			return icon;
		}
	};
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 */

egw.extend('jsonq', egw.MODULE_GLOBAL, function()
{
	"use strict";

	/**
	 * Explicit registered push callbacks
	 *
	 * @type {Function[]}
	 */
	let push_callbacks = [];

	/**
	 * Queued json requests (objects with attributes menuaction, parameters, context, callback, sender and callbeforesend)
	 *
	 * @access private, use jsonq method to queue requests
	 */
	const jsonq_queue = {};

	/**
	 * Next uid (index) in queue
	 */
	let jsonq_uid = 0;

	/**
	 * Running timer for next send of queued items
	 */
	let jsonq_timer = null;

	/**
	 * Send the whole job-queue to the server in a single json request with menuaction=queue
	 */
	function jsonq_send()
	{
		if (jsonq_uid > 0 && typeof jsonq_queue['u'+(jsonq_uid-1)] == 'object')
		{
			const jobs_to_send = {};
			let something_to_send = false;
			for(let uid in jsonq_queue)
			{
				const job = jsonq_queue[uid];

				if (job.menuaction === 'send') continue;	// already send to server

				// if job has a callbeforesend callback, call it to allow it to modify parameters
				if (typeof job.callbeforesend === 'function')
				{
					job.callbeforesend.call(job.sender, job.parameters);
				}
				jobs_to_send[uid] = {
					menuaction: job.menuaction,
					parameters: job.parameters
				};
				job.menuaction = 'send';
				job.parameters = null;
				something_to_send = true;
			}
			if (something_to_send)
			{
				egw.request('api.queue', jobs_to_send).then(_data =>
				{
					if (typeof _data != 'object') throw "jsonq_callback called with NO object as parameter!";

					const json = egw.json('none');
					for(let uid in _data)
					{
						if (typeof jsonq_queue[uid] == 'undefined')
						{
							console.log("jsonq_callback received response for not existing queue uid="+uid+"!");
							console.log(_data[uid]);
							continue;
						}
						const job = jsonq_queue[uid];
						const response = _data[uid];

						// The ajax request has completed, get just the data & pass it on
						if(response)
						{
							for(let value of response)
							{
								if(value.type && value.type === "data" && typeof value.data !== "undefined")
								{
									// Data was packed in response
									job.resolve(value.data);
								}
								else if (value && typeof value.type === "undefined" && typeof value.data === "undefined")
								{
									// Just raw data
									job.resolve(value);
								}
								else
								{
									// fake egw.json_request object, to call it with the current response
									json.handleResponse({response: response});
								}
							}
							// Response is there, but empty.  Make sure to resolve it or the callback doesn't get called.
							if (typeof response.length !== "undefined" && response.length == 0)
							{
								job.resolve();
							}
						}

						delete jsonq_queue[uid];
					}
					// if nothing left in queue, stop interval-timer to give browser a rest
					if (jsonq_timer && typeof jsonq_queue['u'+(jsonq_uid-1)] != 'object')
					{
						window.clearInterval(jsonq_timer);
						jsonq_timer = null;
					}
				});
			}
		}
	}

	return {
		/**
		 * Send a queued JSON call to the server
		 *
		 * @param {string} _menuaction the menuaction function which should be called and
		 *   which handles the actual request. If the menuaction is a full featured
		 *   url, this one will be used instead.
		 * @param {array} _parameters which should be passed to the menuaction function.
		 * @param {function|undefined} _callback callback function which should be called upon a "data" response is received
		 * @param {object|undefined} _sender is the reference object the callback function should get
		 * @param {function|undefined} _callbeforesend optional callback function which can modify the parameters, eg. to do some own queuing
		 * @return Promise
		 */
		jsonq: function(_menuaction, _parameters, _callback, _sender, _callbeforesend)
		{
			const uid = 'u'+(jsonq_uid++);
			jsonq_queue[uid] = {
				menuaction: _menuaction,
				// IE JSON-serializes arrays passed in from different window contextx (eg. popups)
				// as objects (it looses object-type of array), causing them to be JSON serialized
				// as objects and loosing parameters which are undefined
				// JSON.stringify([123,undefined]) --> '{"0":123}' instead of '[123,null]'
				parameters: _parameters ? [].concat(_parameters) : [],
				callbeforesend: _callbeforesend && _sender ? _callbeforesend.bind(_sender) : _callbeforesend,
			};
			let promise = new Promise(resolve => {
				jsonq_queue[uid].resolve = resolve;
			});
			if (typeof _callback === 'function')
			{
				const callback = _callback.bind(_sender);
				promise = promise.then(_data => {
					callback(_data);
					return _data;
				});
			}

			if (jsonq_timer == null)
			{
				// check / send queue every N ms
				jsonq_timer = window.setInterval(() => jsonq_send(), 100);
			}
			return promise;
		},

		/**
		 * Register a callback to receive push broadcasts eg. in a popup or iframe
		 *
		 * It's also used internally by egw_message's push method to dispatch to the registered callbacks.
		 *
		 * @param {Function|PushData} data callback (with bound context) or PushData to dispatch to callbacks
		 */
		registerPush: function(data)
		{
			if (typeof data === "function")
			{
				push_callbacks.push(data);
			}
			else
			{
				for (let n in push_callbacks)
				{
					try {
						push_callbacks[n].call(this, data);
					}
					// if we get an exception, we assume the callback is no longer available and remove it
					catch (ex) {
						push_callbacks.splice(n, 1);
					}
				}
			}
		}

	};
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 */

/**
 * @augments Class
 * @param {string} _app application name object is instanciated for
 * @param {object} _wnd window object is instanciated for
 */
egw.extend('files', egw.MODULE_WND_LOCAL, function(_app, _wnd)
{
	"use strict";

	var egw = this;

	/**
	 * Remove optional timestamp attached as query parameter, eg. /path/name.js?12345678[&other=val]
	 *
	 * Examples:
	 *  /path/file.js --> /path/file.js
	 *  /path/file.js?123456 --> /path/file.js
	 *  /path/file.php?123456&param=value --> /path/file.php?param=value
	 *  /path/file.php?param=value&123456 --> /path/file.php?param=value
	 *
	 * @param _src url
	 * @return url with timestamp stripped off
	 */
	function removeTS(_src)
	{
		return _src.replace(/[?&][0-9]+&?/, '?').replace(/\?$/, '');
	}

	/**
	 * RegExp to extract string with comma-separated files from a bundle-url
	 *
	 * @type RegExp
	 */
	var bundle2files_regexp = /phpgwapi\/inc\/min\/\?b=[^&]+&f=([^&]+)/;

	/**
	 * Regexp to detect and remove .min.js extension
	 *
	 * @type RegExp
	 */
	var min_js_regexp = /\.min\.js$/;

	/**
	 * Return array of files-sources from bundle(s) incl. bundle-src itself
	 *
	 * @param {string|Array} _srcs all url's have to be egw releativ!
	 * @returns {Array}
	 */
	function files_from_bundles(_srcs)
	{
		var files = [];

		if (typeof _srcs == 'string') _srcs = [_srcs];

		for(var n=0; n < _srcs.length; ++n)
		{
			var file = _srcs[n];
			files.push(file.replace(min_js_regexp, '.js'));
			var contains = file.match(bundle2files_regexp);

			if (contains && contains.length > 1)
			{
				var bundle = contains[1].split(',');
				for(var i=0; i < bundle.length; ++i)
				{
					files.push(bundle[i].replace(min_js_regexp, '.js'));
				}
			}
		}
		return files;
	}

	/**
	 * Strip of egw_url from given urls (if containing it)
	 *
	 * @param {array} _urls absolute urls
	 * @returns {array} relativ urls
	 */
	function strip_egw_url(_urls)
	{
		var egw_url = egw.webserverUrl;
		if (egw_url.charAt(egw_url.length-1) != '/') egw_url += '/';

		for(var i=0; i < _urls.length; ++i)
		{
			var file = _urls[i];
			// check if egw_url is only path and urls contains full url incl. protocol
			// --> prefix it with our protocol and host, as eg. splitting by just '/' will fail!
			var need_full_url = egw_url[0] == '/' && file.substr(0,4) == 'http' ? window.location.protocol+'//'+window.location.host : '';
			var parts = file.split(need_full_url+egw_url);
			if (parts.length > 1)
			{
				// discard protocol and host
				parts.shift();
				_urls[i] = parts.join(need_full_url+egw_url);
			}
		}
		return _urls;
	}

	/**
	 * Array which contains all currently bound in javascript and css files.
	 */
	var files = [];
	// add already included scripts
	var tags = jQuery('script', _wnd.document);
	for(var i=0; i < tags.length; ++i)
	{
		files.push(removeTS(tags[i].src));
	}
	// add already included css
	tags = jQuery('link[type="text/css"]', _wnd.document);
	for(var i=0; i < tags.length; ++i)
	{
		files.push(removeTS(tags[i].href));
	}
	// make urls egw-relative
	files = strip_egw_url(files);
	// resolve bundles and replace .min.js with .js
	files = files_from_bundles(files);

	return {
		/**
		 * Load and execute javascript file(s) in order
		 *
		 * Deprecated because with egw composition happening in main window the used import statement happens in that context
		 * and NOT in the window (eg. popup or iframe) this module is instantiated for!
		 *
		 * @memberOf egw
		 * @param {string|array} _jsFiles (array of) urls to include
		 * @param {function} _callback called after JS files are loaded and executed
		 * @param {object} _context
		 * @param {string} _prefix prefix for _jsFiles
		 * @deprecated use es6 import statement: Promise.all([].concat(_jsFiles).map((src)=>import(_prefix+src))).then(...)
		 * @return Promise
		 */
		includeJS: function(_jsFiles, _callback, _context, _prefix)
		{
			// Also allow including a single javascript file
			if (typeof _jsFiles === 'string')
			{
				_jsFiles = [_jsFiles];
			}
			// filter out files included by script-tag via egw.js
			_jsFiles = _jsFiles.filter((src) => src.match(egw.legacy_js_regexp) === null);
			let promise;
			if (_jsFiles.length === 1)	// running this in below case fails when loading app.js from etemplate.load()
			{
				const src = _jsFiles[0];
				promise = import(_prefix ? _prefix+src : src)
					.catch((err) => {
						console.error(src+": "+err.message);
						return Promise.reject(err.message);
					});
			}
			else
			{
				promise = Promise.all(_jsFiles.map((src) => {
					import(_prefix ? _prefix+src : src)
						.catch((err) => {
							console.error(src+": "+err.message);
							return Promise.reject(err.message);
						});
				}));
			}
			return typeof _callback === 'undefined' ? promise : promise.then(_callback.call(_context));
		},

		/**
		 * Check if file is already included and optional mark it as included if not yet included
		 *
		 * Check does NOT differenciate between file.min.js and file.js.
		 * Only .js get's recored in files for further checking, if _add_if_not set.
		 *
		 * @param {string} _file
		 * @param {boolean} _add_if_not if true mark file as included
		 * @return boolean true if file already included, false if not
		 */
		included: function(_file, _add_if_not)
		{
			var file = removeTS(_file).replace(min_js_regexp, '.js');
			var not_inc = files.indexOf(file) == -1;

			if (not_inc && _add_if_not)
			{
				files = files.concat(files_from_bundles(file));
			}
			return !not_inc;
		},

		/**
		 * Include a CSS file
		 *
		 * @param {string|array} _cssFiles full url of file to include
		 */
		includeCSS: function(_cssFiles)
		{
			if (typeof _cssFiles == 'string') _cssFiles = [_cssFiles];
			_cssFiles = strip_egw_url(_cssFiles);

			for(var n=0; n < _cssFiles.length; ++n)
			{
				var file = _cssFiles[n];
				if (!this.included(file, true))	// check if included and marks as such if not
				{
					// Create the node which is used to include the css file
					var cssnode = _wnd.document.createElement('link');
					cssnode.type = "text/css";
					cssnode.rel = "stylesheet";
					cssnode.href = egw.webserverUrl+'/'+file;

					// Get the head node and append the newly created "link" nod to it
					var head = _wnd.document.getElementsByTagName('head')[0];
					head.appendChild(cssnode);
				}
			}
		}
	};
});

/**
 * EGroupware clientside API for persistant storage
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Nathan Gray
 * @version $Id$
 */

/**
 * Store is a wrapper around browser based, persistant storage.
 *
 *
 * @see http://www.w3.org/TR/webstorage/#storage
 *
 * @param {string} _app
 * @param {DOMWindow} _wnd
 */
egw.extend('store', egw.MODULE_GLOBAL, function(_app, _wnd)
{
	"use strict";

	var egw = this;

	/**
	 * Since the storage is shared across at least all applications, make
	 * the key include some extra info.
	 *
	 * @param {string} application
	 * @param {string} key
	 * @returns {undefined}
	 */
	function mapKey(application, key)
	{
		return application + '-' + key;
	}

	return {
		/**
		 * Retrieve a value from session storage
		 *
		 * @param {string} application Name of application, or common
		 * @param {string} key
		 * @returns {string}
		 */
		getSessionItem: function(application, key) {
			key = mapKey(application, key);
			return _wnd.sessionStorage.getItem(key);
		},

		/**
		 * Set a value in session storage
		 *
		 * @param {string} application Name of application, or common
		 * @param {string} key
		 * @param {string} value
		 * @returns {@exp;window@pro;sessionStorage@call;setItem}
		 */
		setSessionItem: function(application, key, value) {
			key = mapKey(application, key);
			return _wnd.sessionStorage.setItem(key, value);
		},

		/**
		 * Remove a value from session storage
		 * @param {string} application
		 * @param {string} key
		 * @returns {@exp;window@pro;sessionStorage@call;removeItem}
		 */
		removeSessionItem: function(application, key) {
			key = mapKey(application, key);
			return _wnd.sessionStorage.removeItem(key);
		},

		/**
		 * Set an item to localStorage
		 *
		 * @param {string} application an application name or a prefix
		 * @param {string} item
		 * @param {any} value
		 * @returns {undefined} returns undefined
		 */
		setLocalStorageItem: function(application, item, value){
			item = mapKey (application, item);
			return localStorage.setItem(item,value);
		},

		/**
		 * Get an item from localStorage
		 *
		 * @param {string} application an application name or prefix
		 * @param {stirng} item an item name stored in localStorage
		 * @return {string|null} reutrns requested item value otherwise null
		 */
		getLocalStorageItem: function(application, item){
			item = mapKey(application, item);
			return localStorage.getItem(item);
		},

		/**
		 * Remove an item from localStorage
		 *
		 * @param {string} application application name or prefix
		 * @param {string} item an item name to remove
		 * @return {undefined} returns undefined
		 */
		removeLocalStorageItem: function (application, item){
			item = mapKey(application, item);
			return localStorage.removeItem(item);
		}
	};
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 * @version $Id$
 */

/**
 *
 * @param {string} _app application name object is instanciated for
 * @param {object} _wnd window object is instanciated for
 */
egw.extend('tooltip', egw.MODULE_WND_LOCAL, function(_app, _wnd)
{
	"use strict";

	const tooltipped = [];
	let tooltip_div = null;
	let current_elem = null;
	_wnd.addEventListener("pagehide", () =>
	{
		tooltipped.forEach(node =>
		{
			egw.tooltipUnbind(node);
		});
		tooltipped.splice(0, tooltipped.length);
		if (tooltip_div && tooltip_div.off)
		{
			tooltip_div.off();
			tooltip_div = null;
		}
		return null;
	});


	const time_delta = 100;
	let show_delta = 0;
	const show_delay = 200;

	let x = 0;
	let y = 0;

	const optionsDefault = {
		hideonhover: true,
		position:'right',
		open: function(){},
		close: function(){}
	};

	/**
	 * Removes the tooltip_div from the DOM if it does exist.
	 */
	function hide()
	{
		if (tooltip_div != null)
		{
			tooltip_div = null;
		}
		_wnd.document.querySelectorAll("body > .egw_tooltip").forEach(t => t.remove());
	}

	/**
	 * Shows the tooltip at the current cursor position.
	 */
	function show(node, event, options)
	{
		if (tooltip_div && typeof x !== 'undefined' && typeof y !== 'undefined')
		{
			options.open.call(node, event, tooltip_div);
			//Calculate the cursor_rectangle - this is a space the tooltip might
			//not overlap with
			var cursor_rect = {
				left: (x - 8),
				top: (y - 8),
				right: (x + (options.position == "center" ? -1 * tooltip_div.width()/2 : 8)),
				bottom: (y + 8)
			};

			//Calculate how much space is left on each side of the rectangle
			var window_width = jQuery(_wnd.document).width();
			var window_height = jQuery(_wnd.document).height();
			var space_left = {
				left: (cursor_rect.left),
				top: (cursor_rect.top),
				right: (window_width - cursor_rect.right),
				bottom: (window_height - cursor_rect.bottom)
			};

			//Get the width and the height of the tooltip
			var tooltip_width = tooltip_div.width();
			if (tooltip_width > 300) tooltip_width = 300;
			var tooltip_height = tooltip_div.height();

			if (space_left.right < tooltip_width) {
				tooltip_div.css('left', Math.max(0,cursor_rect.left - tooltip_width));
			} else if (space_left.left >= tooltip_width) {
				tooltip_div.css('left', cursor_rect.right);
			} else	{
				tooltip_div.css('left', cursor_rect.right);
				tooltip_div.css('max-width', space_left.right);
			}

			// tooltip does fit neither above nor below: put him vertical centered left or right of cursor
			if (space_left.bottom < tooltip_height && space_left.top < tooltip_height) {
				if (tooltip_height > window_height-20) {
					tooltip_div.css('max-height', tooltip_height=window_height-20);
				}
				tooltip_div.css('top', (window_height-tooltip_height)/2);
			} else if (space_left.bottom < tooltip_height) {
				tooltip_div.css('top', cursor_rect.top - tooltip_height);
			} else {
				tooltip_div.css('top', cursor_rect.bottom);
			}
			tooltip_div.fadeIn(100);

		}
	}

	/**
	 * Creates the tooltip_div with the given text.
	 *
	 * @param {string} _html
	 * @param {boolean} _isHtml if set to true content gets appended as html
	 */
	function prepare(_html, _isHtml, _options)
	{
		// Free and null the old tooltip_div
		hide();

		//Generate the tooltip div, set it's text and append it to the body tag
		tooltip_div = jQuery(_wnd.document.createElement('div'));
		tooltip_div.hide();
		if (_isHtml)
		{
			tooltip_div.append(_html);
		}
		else
		{
			tooltip_div.text(_html);
		}
		tooltip_div.addClass("egw_tooltip");
		jQuery(_wnd.document.body).append(tooltip_div);

		//The tooltip should automatically hide when the mouse comes over it
		tooltip_div.get(0).addEventListener("mouseover", () =>
		{
				if (_options.hideonhover) hide();
		});
	}

	/**
	 * showTooltipTimeout is used to prepare showing the tooltip.
	 */
	function showTooltipTimeout(node, event, options)
	{
		if (current_elem != null)
		{
			show_delta += time_delta;
			if (show_delta < show_delay)
			{
				//Repeat the call of timeout
				_wnd.setTimeout(function(){showTooltipTimeout(this, event, options);}.bind(node), time_delta);
			}
			else
			{
				show_delta = 0;
				show(node, event, options);
			}
		}
	}

	return {
		/**
		 * Binds a tooltip to the given DOM-Node with the given html.
		 * It is important to remove all tooltips from all elements which are
		 * no longer needed, in order to prevent memory leaks.
		 *
		 * @param _elem is the element to which the tooltip should get bound.
		 * @param _html is the html code which should be shown as tooltip.
		 * @param _options
		 */
		tooltipBind: function(_elem, _html, _isHtml, _options) {

			var options = {...optionsDefault, ...(_options||{})};
			const elem = _elem instanceof Node ? _elem : (typeof _elem.get == "function" ? _elem.get(0) : _elem);
			tooltipped.push(elem);
			_elem = jQuery(_elem);
			if (_html && !egwIsMobile())
			{
				_elem.bind('mouseenter.tooltip', function(e) {
					if (_elem != current_elem)
					{
						//Prepare the tooltip
						prepare(_html, _isHtml, options);

						// Set the current element the mouse is over and
						// initialize the position variables
						current_elem = _elem;
						show_delta = 0;
						x = e.clientX;
						y = e.clientY;
						let self = this;
						// Create the timeout for showing the timeout
						_wnd.setTimeout(function(){showTooltipTimeout(self, e, options);}, time_delta);
					}

					return false;
				});

				_elem.bind('mouseleave.tooltip', function(e) {
					current_elem = null;
					show_delta = 0;
					if (options.close.call(this, e, tooltip_div)) return;
					if (tooltip_div)
					{
						tooltip_div.fadeOut(100);
					}
				});

				_elem.bind('mousemove.tooltip', function(e) {
					//Calculate the distance the mouse took since the last call of mousemove
					var dx = x - e.clientX;
					var dy = y - e.clientY;
					var movedist = Math.sqrt(dx * dx + dy * dy);

					//Block appereance of the tooltip on fast movements (with small movedistances)
					if (movedist > 2)
					{
						show_delta = 0;
					}

					x = e.clientX;
					y = e.clientY;
				});
			}
		},

		/**
		 * Unbinds the tooltip from the given DOM-Node.
		 *
		 * @param _elem is the element from which the tooltip should get
		 * removed. _elem has to be a jQuery node.
		 */
		tooltipUnbind: function(_elem) {
			_elem = jQuery(_elem);
			if (current_elem == _elem) {
				hide();
				current_elem = null;
			}

			// Unbind all "tooltip" events from the given element
			_elem.unbind('.tooltip');
			tooltipped.splice(tooltipped.indexOf(_elem), 1);
		},

		tooltipDestroy: function () {
			if (tooltip_div)
			{
				tooltip_div.fadeOut(100);
				tooltip_div.remove();
			}
		},

		/**
		 * Hide tooltip, cancel the timer
		 */
		tooltipCancel: function ()
		{
			hide();
			current_elem = null;
		}
	};

});

/**
 * eGroupWare eTemplate2 - Stylesheet class
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel
 * @copyright Stylite 2011
 * @version $Id$
 */

/**
 * Module which allows to add stylesheet rules at runtime. Exports the following
 * functions:
 * - css
 * @param {string} _app application name object is instanciated for
 * @param {object} _wnd window object is instanciated for
 */
egw.extend('css', egw.MODULE_WND_LOCAL, function(_app, _wnd)
{
	"use strict";

	/**
	 * Assoziative array which stores the current css rule for a given selector.
	 */
	var selectors = {};

	/**
	 * Variable used to calculate unique id for the selectors.
	 */
	var selectorCount = 0;
	var sheet;

	return {
		/**
		 * The css function can be used to introduce a rule for the given css
		 * selector. So you're capable of adding new custom css selector while
		 * runtime and also update them.
		 *
		 * @param _selector is the css select which can be used to apply the
		 * 	stlyes to the html elements.
		 * @param _rule is the rule which should be connected to the selector.
		 * 	if empty or omitted, the given selector gets removed.
		 */
		css: function(_selector, _rule) {
			// Set the current index to the maximum index
			var index = sheet ? Math.min(selectorCount, sheet.cssRules.length) : 0;

			if (!sheet || !sheet.ownerNode || sheet.ownerNode.ownerDocument !== _wnd.document)
			{
				// Generate a style tag, which will be used to hold the newly generated css
				// rules.
				var style = _wnd.document.createElement('style');
				_wnd.document.getElementsByTagName('head')[0].appendChild(style);

				// Obtain the reference to the styleSheet object of the generated style tag
				sheet = style.sheet ? style.sheet : style.styleSheet;

				selectorCount = 0;
				selectors = {};
			}

			// Remove any existing rule first, of no rule exists for the
			if (typeof selectors[_selector] !== "undefined")
			{
				// Store the old index
				index = selectors[_selector];
				if(index < sheet.cssRules.length)
				{
					if (typeof sheet.removeRule !== "undefined")
					{
						sheet.removeRule(index);
					}
					else
					{
						sheet.deleteRule(index);
					}
				}

				delete (selectors[_selector]);
				if(!_rule)
				{
					selectorCount--;
				}
			}
			else
			{
				selectorCount++;
			}

			if (_rule)
			{
				// Add the rule to the stylesheet
				if (typeof sheet.addRule !== "undefined")
				{
					sheet.addRule(_selector, _rule, index);
				}
				else
				{
					sheet.insertRule(_selector + "{" + _rule + "}", index);
				}

				// Store the new index
				selectors[_selector] = index;
			}
		}
	};
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 */

/**
 * Date and timepicker
 *
 * @augments Class
 * @param {string} _app application name object is instanced for
 * @param {object} _wnd window object is instanced for
 */
egw.extend('calendar', egw.MODULE_GLOBAL, function (_app, _wnd)
{
	"use strict";

	let _holiday_cache = {};

	/**
	 * transform PHP date/time-format to jQuery date/time-format
	 *
	 * @param {string} _php_format
	 * @returns {string}
	 */
	function dateTimeFormat(_php_format)
	{
		return _php_format
			.replace("Y","yy")
			.replace("d","dd")
			.replace("m","mm")
			.replace("M", "M")
			.replace('H', 'hh')
			.replace('i', 'mm')	// datepicker uses mm for month and minutes, depending on where in format it's written!
			.replace('s', 'ss');
	}

	return {
		/**
		 * setup a calendar / date-selection
		 *
		 * @member of egw
		 * @param _input
		 * @param _time
		 * @param _callback
		 * @param _context
		 * @returns
		 */
		calendar: function(_input, _time, _callback, _context)
		{
			alert('jQueryUI datepicker is no longer supported!');
		},
		/**
		 * setup a time-selection
		 *
		 * @param _input
		 * @param _callback
		 * @param _context
		 * @returns
		 */
		time: function(_input, _callback, _context)
		{
			alert('jQueryUI datepicker is no longer supported!');
		},
		/**
		 * transform PHP date/time-format to jQuery date/time-format
		 *
		 * @param {string} _php_format
		 * @returns {string}
		 */
		dateTimeFormat: function(_php_format)
		{
			return dateTimeFormat(_php_format);
		},
		/**
		 * Get timezone offset of user in seconds
		 *
		 * If browser / OS is configured correct, identical to: (new Date()).getTimezoneOffset()
		 *
		 * @return {number} offset to UTC in minutes
		 */
		getTimezoneOffset: function() {
			return isNaN(egw.preference('timezoneoffset')) ? (new Date()).getTimezoneOffset() : parseInt(egw.preference('timezoneoffset'));
		},
		/**
		 * Calculate the start of the week, according to user's preference
		 *
		 * @param {string} date
		 * @return {Date}
		 */
		week_start: function(date) {
			var d = new Date(date);
			var day = d.getUTCDay();
			var diff = 0;
			switch(egw.preference('weekdaystarts','calendar'))
			{
				case 'Saturday':
					diff = day === 6 ? 0 : day === 0 ? -1 : -(day + 1);
					break;
				case 'Monday':
					diff = day === 0 ? -6 : 1 - day;
					break;
				case 'Sunday':
				default:
					diff = -day;
			}
			d.setUTCDate(d.getUTCDate() + diff);
			return d;
		},
		/**
		 * Get a list of holidays for the given year
		 *
		 * Returns a promise that resolves with a list of holidays indexed by date, in Ymd format:
		 * {20001225: [{day: 14, month: 2, occurence: 2021, name: "Valentinstag"}]}
		 *
		 * No need to cache the results, we do it here.
		 *
		 * @param year
		 * @returns Promise<{[key: string]: Array<object>}>
		 */
		holidays: function holidays(year) //: Promise<{ [key : string] : Array<object> }>
		{
			// No country selected causes error, so skip if it's missing
			if (!egw || !egw.preference('country', 'common'))
			{
				return Promise.resolve({});
			}

			if (typeof _holiday_cache[year] === 'undefined')
			{
				// Fetch with json instead of jsonq because there may be more than
				// one widget listening for the response by the time it gets back,
				// and we can't do that when it's queued.
				_holiday_cache[year] = window.fetch(
					egw.link('/calendar/holidays.php', {year: year, url: this.config('ical_holiday_url') || ''})
				).then((response) =>
				{
					return _holiday_cache[year] = response.json();
				});
			}
			return Promise.resolve(_holiday_cache[year]);
		}
	};
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel (as AT stylite.de)
 */

/**
 * @augments Class
 * @param {string} _app application name object is instanciated for
 * @param {object} _wnd window object is instanciated for
 */
egw.extend('ready', egw.MODULE_WND_LOCAL, function(_app, _wnd)
{
	"use strict";

	var egw = this;

	var registeredCallbacks = [];
	var registeredProgress = [];
	var readyPending = {'readyEvent': true};
	var readyPendingCnt = 1;
	var readyDoneCnt = 0;
	var isReady = false;

	function doReadyWaitFor() {
		if (!isReady)
		{
			var uid = egw.uid();
			readyPending[uid] = true;
			readyPendingCnt++;

			readyProgressChange();

			return uid;
		}

		this.debug('warning', 'ready has already been called!');

		return null;
	}

	function doReadyDone(_token) {
		if (typeof readyPending[_token] !== 'undefined')
		{
			delete readyPending[_token];
			readyPendingCnt--;
			readyDoneCnt++;

			readyProgressChange();

			testCallReady();
		}
	}

	function readyProgressChange()
	{
		// Call all registered progress callbacks
		for (var i = 0; i < registeredProgress.length; i++)
		{
			registeredProgress[i].callback.call(
				registeredProgress[i].context,
				readyDoneCnt,
				readyPendingCnt
			);
		}

		egw.debug('log', 'Ready events, processed %s/%s', readyDoneCnt,
				readyPendingCnt + readyDoneCnt);
	}

	function readyEventHandler() {
		doReadyDone('readyEvent');
	}

	function testCallReady()
	{
		// Check whether no further event is pending
		if (readyPendingCnt <= 1 && !isReady)
		{
			// If exactly one event is pending and that one is not the ready
			// event, abort
			if (readyPendingCnt === 1 && !readyPending['readyEvent'])
			{
				return;
			}

			// Set "isReady" to true, if readyPendingCnt is zero
			var isReady = readyPendingCnt === 0;

			// Call all registered callbacks
			for (var i = registeredCallbacks.length - 1; i >= 0; i--)
			{
				if (registeredCallbacks[i].before || readyPendingCnt === 0)
				{
					registeredCallbacks[i].callback.call(
						registeredCallbacks[i].context
					);

					// Delete the callback from the list
					registeredCallbacks.splice(i, 1);
				}
			}
		}
	}

	// Register the event handler for "ready" (code adapted from jQuery)

	// Mozilla, Opera and webkit nightlies currently support this event
	if (_wnd.document.addEventListener) {
		// Use the handy event callback
		_wnd.document.addEventListener("DOMContentLoaded", readyEventHandler, false);

		// A fallback to window.onload, that will always work
		_wnd.addEventListener("load", readyEventHandler, false);

	// If IE event model is used
	} else if (_wnd.document.attachEvent) {
		// ensure firing before onload,
		// maybe late but safe also for iframes
		_wnd.document.attachEvent("onreadystatechange", readyEventHandler);

		// A fallback to window.onload, that will always work
		_wnd.attachEvent("onload", readyEventHandler);
	}

	return {

		/**
		 * The readyWaitFor function can be used to register an event, that has
		 * to be marked as "done" before the ready function will call its
		 * registered callbacks. The function returns an id that has to be
		 * passed to the "readDone" function once
		 *
		 * @memberOf egw
		 */
		readyWaitFor: function() {
			return doReadyWaitFor();
		},

		/**
		 * The readyDone function can be used to mark a event token as
		 * previously requested by "readyWaitFor" as done.
		 *
		 * @param _token is the token which has now been processed.
		 */
		readyDone: function(_token) {
			doReadyDone(_token);
		},

		/**
		 * The ready function can be used to register a function that will be
		 * called, when the window is completely loaded. All ready handlers will
		 * be called exactly once. If the ready handler has already been called,
		 * the given function will be called defered using setTimeout.
		 *
		 * @param _callback is the function which will be called when the page
		 * 	is ready. No parameters will be passed.
		 * @param _context is the context in which the callback function should
		 * 	get called.
		 * @param _beforeDOMContentLoaded specifies, whether the callback should
		 * 	get called, before the DOMContentLoaded event has been fired.
		 */
		ready: function(_callback, _context, _beforeDOMContentLoaded) {
			if (!isReady)
			{
				registeredCallbacks.push({
					'callback': _callback,
					'context': _context ? _context : null,
					'before': _beforeDOMContentLoaded ? true : false
				});
			}
			else
			{
				setTimeout(function() {
					_callback.call(_context);
				}, 1);
			}
		},

		/**
		 * The readyProgress function can be used to register a function that
		 * will be called whenever a ready event is done or registered.
		 *
		 * @param _callback is the function which will be called when the
		 * 	progress changes.
		 * @param _context is the context in which the callback function which
		 * 	should get called.
		 */
		readyProgress: function(_callback, _context) {
			if (!isReady)
			{
				registeredProgress.unshift({
					'callback': _callback,
					'context': _context ? _context : null
				});
			}
			else
			{
				this.debug('warning', 'ready has already been called!');
			}
		},

		/**
		 * Returns whether the ready events have already been called.
		 */
		isReady: function() {
			return isReady;
		}
	};

});

/**
 * eGroupWare eTemplate2
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Andreas Stöckel
 * @copyright Stylite 2012
 * @version $Id$
 */

/**
 * Module storing and updating row data
 *
 * @param {string} _app application name object is instanciated for
 * @param {object} _wnd window object is instanciated for
 */
egw.extend("data", egw.MODULE_APP_LOCAL, function (_app, _wnd)
{
	"use strict";

	/**
	 * How many UIDs we'll tell the server we know about.  No need to pass the whole list around.
	 */
	var KNOWN_UID_LIMIT = 200;

	/**
	 * Cache lifetime
	 *
	 * If cached results are used, we check their timestamp.  If the timestamp
	 * is older than this, we will also ask for fresh data.  For cached data
	 * younger than this, we only return the cache
	 *
	 * 29 seconds, 1 less then the fastest nextmatch autorefresh option
	 */
	var CACHE_LIFETIME = 29; // seconds

	/**
	 * Cached fetches are differentiated from actual results by using this prefix
	 * @type String
	 */
	var CACHE_KEY_PREFIX = 'cached_fetch_';

	var lastModification = null;

	/**
	 * cacheCallback stores callbacks that determine if data is placed
	 * into cacheStorage, or simply kept temporarily.  It is indexed
	 * by prefix.
	 *
	 * @type Array
	 */
	var cacheCallback = {};

	/**
	 * The uid function generates a session-unique id for the current
	 * application by appending the application name to the given uid.
	 *
	 * @param {string} _uid
	 * @param {string} _prefix
	 */
	function UID(_uid, _prefix)
	{
		_prefix = _prefix ? _prefix : _app;

		return _prefix + "::" + _uid;
	}

	/**
	 * Looks like too much data is cached.  Forget some.
	 *
	 * Tries to free up localStorage by removing the oldest cached data for the
	 * given prefix, but if none is found it will look at all cached data.
	 *
	 * @param {string} _prefix UID / application prefix
	 * @returns {Number} Number of cached recordsets removed, normally 1.
	 */
	function _clearCache(_prefix)
	{
		// Find cached items for the prefix, we prefer to expire just within the app
		var indexes = [];
		for(var i = 0; i < window.localStorage.length; i++)
		{
			var key = window.localStorage.key(i);

			// This is a cached fetch for many rows
			if(key.indexOf(CACHE_KEY_PREFIX+_prefix) == 0)
			{
				var cached = JSON.parse(window.localStorage.getItem(key));

				if(cached.lastModification)
				{
					indexes.push({
						key: key,
						lastModification: cached.lastModification
					});
				}
				else
				{
					// No way to know how old it is, just remove it
					window.localStorage.removeItem(key);
				}
			}
			// Actual cached data
			else if (key.indexOf(_prefix) == 0)
			{
				try {
					let cached = JSON.parse(window.localStorage.getItem(key));
					if(cached.timestamp)
					{
						indexes.push({
							key: key,
							lastModification: cached.timestamp
						});
					}
					else
					{
						// No way to know how old it is, just remove it
						window.localStorage.removeItem(key);
					}
				}
				catch (e) {
					window.localStorage.removeItem(key);
				}
			}
		}
		// Nothing for that prefix?  Clear all cached data.
		if(_prefix && indexes.length == 0)
		{
			return _clearCache('');
		}
		// Found some cached for that prefix, only remove the oldest
		else if (indexes.length > 0)
		{
			indexes.sort(function(a,b) {
				if(a.lastModification < b.lastModification) return 1;
				if(a.lastModification > b.lastModification) return -1;
				return 0;
			});
			window.localStorage.removeItem(indexes.pop().key);
			return 1;
		}
		return indexes.length;
	}

	function parseServerResponse(_result, _callback, _context, _execId, _widgetId)
	{
		// Check whether the result is valid
		// This result is not for us, quietly return
		if(_result && typeof _result.type != 'undefined') return;

		// "result" has to be an object consisting of "order" and "data"
		if (!(_result && typeof _result.order !== "undefined"
		    && typeof _result.data !== "undefined"))
		{
			egw.debug("error", "Invalid result for 'dataFetch'");
		}

		if (_result.lastModification)
		{
			lastModification = _result.lastModification;
		}

		if (_result.order && _result.data)
		{
			// Assemble the correct order uids
			if(!(_result.order.length && _result.order[0] && _result.order[0].indexOf && _result.order[0].indexOf(_context.prefix) == 0))
			{
				for (var i = 0; i < _result.order.length; i++)
				{
					_result.order[i] = UID(_result.order[i], _context.prefix);
				}
			}

			// Load all data entries that have been sent or delete them
			for (var key in _result.data)
			{
				let uid = UID(key, (typeof _context == "object" && _context != null) ?_context.prefix : "");
				if (_result.data[key] === null &&
				(
					typeof _context.refresh == "undefined" || _context.refresh && !jQuery.inArray(key,_context.refresh)
				))
				{
					egw.dataDeleteUID(uid);
				}
				else
				{
					egw.dataStoreUID(uid, _result.data[key]);
				}
			}

			// Check if we tried to refresh a specific row and didn't get it, so set it to null
			// (triggers update for listeners), then remove it
			if(typeof _context == "object" && _context.refresh)
			{
				for(let i = 0; i < _context.refresh.length; i++)
				{
					let uid = UID(_context.refresh[i], _context.prefix);
					if(_result.order.indexOf(uid) >= 0)
					{
						continue;
					}
					egw.dataStoreUID(uid, null);
					egw.dataDeleteUID(uid);
				}
			}

			// Check to see if we need long-term caching of the query and its results
			if(window.localStorage && _context.prefix && cacheCallback[_context.prefix]  && !_context.no_cache)
			{
				// Ask registered callbacks if we should cache this
				for(var i = 0; i < cacheCallback[_context.prefix].length; i++)
				{
					var cc = cacheCallback[_context.prefix][i];
					var cache_key = cc.callback.call(cc.context, _context);
					if(cache_key)
					{
						cache_key = CACHE_KEY_PREFIX + _context.prefix + '::' + cache_key;
						try
						{
							for (var key in _result.data)
							{
								var uid = UID(key, (typeof _context == "object" && _context != null) ? _context.prefix : "");

								// Register a handler on each data so we can know if it is updated or removed
								egw.dataUnregisterUID(uid, null, cache_key);
								egw.dataRegisterUID(uid, function(data, _uid) {
									// If data item is removed, remove it from cached fetch too
									if(data == null)
									{
										var cached = JSON.parse(window.localStorage[this]) || false;
										if(cached && cached.order && cached.order.indexOf(_uid) >= 0)
										{
											cached.order.splice(cached.order.indexOf(_uid),1);
											if(cached.total) cached.total--;
											window.localStorage[this] = JSON.stringify(cached);
										}
										window.localStorage.removeItem(_uid);
									}
									else
									{
										// Update or store data in long-term storage
										window.localStorage[_uid] = JSON.stringify({timestamp: (new Date).getTime(), data: data});
									}
								}, cache_key, _execId, _widgetId);
							}
							// Don't keep data in long-term cache with request also
							_result.data = {};
							window.localStorage.setItem(cache_key,JSON.stringify(_result));
						}
						catch (e)
						{
							// Maybe ran out of space?  Free some up.
							if(e.name == 'QuotaExceededError'	// storage quota is exceeded, remove cached data
								|| e.name == 'NS_ERROR_DOM_QUOTA_REACHED')	// FF-name
							{
								var count = _clearCache(_context.prefix);
								egw.debug('info', 'localStorage full, removed ' + count + ' stored datasets');
							}
							// No, something worse happened
							else
							{
								egw.debug('warning', 'Tried to cache some data.  It did not work.', cache_key, e);
							}
						}
					}
				}
			}

			// Call the callback function and pass the calculated "order" array
			// as well as the "total" count and the "timestamp" to the listener.
			if (_callback)
			{
				_callback.call(_context, {
					"order": _result.order,
					"total": parseInt(_result.total),
					"readonlys": _result.readonlys,
					"rows": _result.rows,
					"lastModification": lastModification
				});
			}
		}
	}

	return {

		/**
		 * The dataFetch function provides an abstraction layer for the
		 * corresponding "EGroupware\Api\Etemplate\Widget\Nextmatch::ajax_get_rows" function.
		 * The server returns the following structure:
		 * 	{
		 * 		order: [uid, ...],
		 * 		data:
		 * 			{
		 * 				uid0: data,
		 * 				...
		 * 				uidN: data
		 * 			},
		 * 		total: <TOTAL COUNT>,
		 * 		lastModification: <LAST MODIFICATION TIMESTAMP>,
		 * 		readonlys: <READONLYS>
		 * 	}
		 * If a uid got deleted on the server above data is null.
		 * If a uid is omitted from data, is has not changed since lastModification.
		 *
		 * If order/data is null, this means that nothing has changed for the
		 * given range.
		 * The dataFetch function stores new data for the uid's inside the
		 * local data storage, the grid views are then capable of querying the
		 * data for those uids from the local storage using the
		 * "dataRegisterUID" function.
		 *
		 * @param _execId is the execution context of the etemplate instance
		 * 	you're querying the data for.
		 * @param _queriedRange is an object of the following form:
		 * 	{
		 * 		start: <START INDEX>,
		 * 		num_rows: <COUNT OF ENTRIES>
		 * 	}
		 * The range always corresponds to the given filter settings.
		 * @param _filters contains the filter settings. The filter settings are
		 * 	those which are crucial for the mapping between index and uid.
		 * @param _widgetId id with full namespace of widget
		 * @param _callback is the function that should get called, once the data
		 * 	is available. The data passed to the callback function has the
		 * 	following form:
		 * 	{
		 * 		order: [uid, ...],
		 * 		total: <TOTAL COUNT>,
		 * 		lastModification: <LAST MODIFICATION TIMESTAMP>,
		 * 		readonlys: <READONLYS>
		 * 	}
		 * 	Please note that the "uids" comming from the server and the ones
		 * 	being parsed to the callback function differ. While the uids
		 * 	which are returned from the server are only unique inside the
		 * 	application, the uids which are used on the client are "globally"
		 * 	unique.
		 * @param _context is the context in which the callback function will get
		 * 	called.
		 * @param _knownUids is an array of uids already known to the client.
		 *  This parameter may be null in order to indicate that the client
		 *  currently has no data for the given filter settings.
		 */
		dataFetch: function (_execId, _queriedRange, _filters, _widgetId,
				_callback, _context, _knownUids)
		{
			var lm = lastModification;
			if(typeof _context.lastModification != "undefined") lm = _context.lastModification;

			if (_queriedRange["no_data"])
			{
				lm = 0xFFFFFFFFFFFF;
			}
			else if (_queriedRange["only_data"])
			{
				lm = 0;
			}

			// Store refresh in context to not delete the other entries when server only returns these
			if (typeof _queriedRange.refresh != "undefined")
			{
				if(typeof _queriedRange.refresh == "string")
				{
					_context.refresh = [_queriedRange.refresh];
				}
				else
				{
					_context.refresh = _queriedRange.refresh;
				}
			}

			// Limit the amount of UIDs we say we know about to a sensible number, in case user is enjoying auto-pagination
			var knownUids = _knownUids ? _knownUids : egw.dataKnownUIDs(_context.prefix ? _context.prefix : _app);
			if(knownUids > KNOWN_UID_LIMIT)
			{
				knownUids.slice(typeof _queriedRange.start != "undefined" ? _queriedRange.start:0,KNOWN_UID_LIMIT);
			}

			// Check to see if we have long-term caching of the query and its results
			if(window.localStorage && _context.prefix && cacheCallback[_context.prefix])
			{
				// Ask registered callbacks if we should cache this
				for(var i = 0; i < cacheCallback[_context.prefix].length; i++)
				{
					var cc = cacheCallback[_context.prefix][i];
					var cache_key = cc.callback.call(cc.context, _context);
					if(cache_key)
					{
						cache_key = CACHE_KEY_PREFIX + _context.prefix + '::' + cache_key;

						var cached = window.localStorage.getItem(cache_key);
						if(cached)
						{
							cached = JSON.parse(cached);
							var needs_update = true;

							// Check timestamp
							if(cached.lastModification && ((Date.now()/1000) - cached.lastModification) < CACHE_LIFETIME)
							{
								needs_update = false;
							}

							egw.debug('log', 'Data cached query from ' + new Date(cached.lastModification*1000)+': ' + cache_key + '('+
								(needs_update ? 'will be' : 'will not be')+" updated)\nprocessing...");

							// Call right away with cached data, but set no_cache flag
							// to avoid re-caching this data with a new timestamp.
							// We may still ask the server though.
							var no_cache = _context.no_cache;
							_context.no_cache = true;
							parseServerResponse(cached, _callback, _context, _execId, _widgetId);
							_context.no_cache = no_cache;


							// If cache registrant wants notification of cache useage,
							// let it know
							if(cc.notification)
							{
								cc.notification.call(cc.context, needs_update);
							}

							if(!needs_update)
							{
								// Cached data is new enough, skip the server call
								return;
							}
						}
					}
				}
			}
			// create a clone of filters, which can be used in parseServerResponse and cache callbacks
			// independent of changes happening while waiting for the response
			_context.filters = jQuery.extend({}, _filters);
			var request = egw.json(
				"EGroupware\\Api\\Etemplate\\Widget\\Nextmatch::ajax_get_rows",
				[
					_execId,
					_queriedRange,
					_filters,
					_widgetId,
					knownUids,
					lm
				],
				function(result) {
					parseServerResponse(result, _callback, _context, _execId, _widgetId);
				},
				this,
				true
			);
			request.sendRequest();
		},

		/**
		 * Turn on long-term client side cache of a particular request
		 * (cache the nextmatch query results) for fast, immediate response
		 * with old data.
		 *
		 * The request is still sent to the server, and the cache is updated
		 * with fresh data, and any needed callbacks are called again with
		 * the fresh data.
		 *
		 * @param {string} prefix UID / Application prefix should match the
		 *	individual record prefix
		 * @param {function} callback_function A function that will analize the provided fetch
		 *	parameters and return a reproducable cache key, or false to not cache
		 *	the request.
		 * @param {function} notice_function A function that will be called whenever
		 *	cached data is used.  It is passed one parameter, a boolean that indicates
		 *	if the server is or will be queried to refresh the cache.  Do not fetch additional data
		 *	inside this callback, and return quickly.
		 * @param {object} context Context for callback function.
		 */
		dataCacheRegister: function(prefix, callback_function, notice_function, context)
		{
			if(typeof cacheCallback[prefix] == 'undefined')
			{
				cacheCallback[prefix] = [];
			}
			cacheCallback[prefix].push({
				callback: callback_function,
				notification: notice_function || false,
				context: context || null
			});
		},

		/**
		 * Unregister a previously registered cache callback
		 * @param {string} prefix UID / Application prefix should match the
		 *	individual record prefix
		 * @param {function} [callback] Callback function to un-register.  If
		 *	omitted, all functions for the prefix will be removed.
		 */
		dataCacheUnregister: function(prefix, callback)
		{
			if(typeof callback != 'undefined')
			{
				for(var i = 0; i < cacheCallback[prefix].length; i++)
				{
					if(cacheCallback[prefix][i].callback == callback)
					{
						cacheCallback[prefix].splice(i,1);
						return;
					}
				}
			}
			// Callback not provided or not found, reset by prefix
			cacheCallback[prefix] = [];
		}
	};

});

egw.extend("data_storage", egw.MODULE_GLOBAL, function (_app, _wnd) {

	/**
	 * The localStorage object is used to store the data for certain uids. An
	 * entry inside the localStorage object looks like the following:
	 * 	{
	 * 		timestamp: <CREATION TIMESTAMP (local)>,
	 * 		data: <DATA>
	 * 	}
	 */
	var localStorage = {};

	/**
	 * The registeredCallbacks map is used to store all callbacks registerd for
	 * a certain uid.
	 */
	var registeredCallbacks = {};



	/**
	 * Register the "data" plugin globally for single uids
	 * Multiple UIDs such as nextmatch results are still handled by egw.data
	 * using dataFetch() && parseServerResponse(), above.  Both update the
	 * GLOBAL data cache though this one is registered globally, and the above
	 * is registered app local.
	 *
	 * @param {string} type
	 * @param {object} res
	 * @param {object} req
	 * @returns {Boolean}
	 */
	egw.registerJSONPlugin(function(type, res, req) {
		if ((typeof res.data.uid != 'undefined') &&
			(typeof res.data.data != 'undefined'))
		{
			// Store it, which will call all registered listeners
			this.dataStoreUID(res.data.uid, res.data.data);
			return true;
		}
	}, egw, 'data',true);

	/**
	 * Uids and timers used for querying data uids, hashed by the first few
	 * bytes of the _execId, stored as an object of the form
	 * {
	 *     "timer": <QUEUE TIMER>,
	 *     "uids": <ARRAY OF UIDS>
	 * }
	 */
	var queue = {};

	/**
	 * Contains the queue timeout in milliseconds.
	 */
	var QUEUE_TIMEOUT = 10;

	/**
	 * This constant specifies the maximum age of entries in the local storrage
	 * in milliseconds
	 */
	var MAX_AGE = 5 * 60 * 1000; // 5 mins

	/**
	 * This constant specifies the interval in which the local storage gets
	 * cleaned up.
	 */
	var CLEANUP_INTERVAL = 30 * 1000; // 30 sec

	/**
	 * Register a cleanup function, which throws away all data entries which are
	 * older than the given age.
	 */
	_wnd.setInterval(function() {
		// Get the current timestamp
		var time = (new Date).getTime();

		// Iterate over the local storage
		for (var uid in localStorage)
		{
			// Expire old data, if there are no callbacks
			if (time - localStorage[uid].timestamp > MAX_AGE && typeof registeredCallbacks[uid] == "undefined")
			{
				// Unregister all registered callbacks for that uid
				egw.dataUnregisterUID(uid);

				// Delete the data from the localStorage
				delete localStorage[uid];

				// We don't clean long-term storage because of age until it runs
				// out of space
			}
		}
	}, CLEANUP_INTERVAL);

	return {

		/**
		 * Registers the intrest in a certain uid for a callback function. If
		 * the data for that uid changes or gets loaded, the given callback
		 * function is called. If the data for the given uid is available at the
		 * time of registering the callback, the callback is called immediately.
		 *
		 * @param _uid is the uid for which the callback should be registered.
		 * @param _callback is the callback which should get called.
		 * @param _context is the optional context in which the callback will be
		 * executed
		 * @param _execId is the exec id which will be used in case the data is
		 * not available
		 * @param _widgetId is the widget id which will be used in case the uid
		 * has to be fetched.
		 */
		dataRegisterUID: function (_uid, _callback, _context, _execId, _widgetId) {
			// Create the slot for the uid if it does not exist now
			if (typeof registeredCallbacks[_uid] === "undefined")
			{
				registeredCallbacks[_uid] = [];
			}

			// Store the given callback
			registeredCallbacks[_uid].push({
				"callback": _callback,
				"context": _context ? _context : null,
				"execId": _execId,
				"widgetId" : _widgetId
			});

			// Check whether the data is available -- if yes, immediately call
			// back the callback function
			if (typeof localStorage[_uid] !== "undefined")
			{
				// Update the timestamp and call the given callback function
				localStorage[_uid].timestamp = (new Date).getTime();
				_callback.call(_context, localStorage[_uid].data, _uid);
			}
			// Check long-term storage
			else if(window.localStorage && window.localStorage[_uid])
			{
				localStorage[_uid] = JSON.parse(window.localStorage[_uid]);
				_callback.call(_context, localStorage[_uid].data, _uid);
			}
			else if (_execId && _widgetId)
			{
				// Get the first 50 bytes of the exex id
				var hash = _execId.substring(0, 50);

				// Create a new queue if it does not exist yet
				if (typeof queue[hash] === "undefined")
				{
					var self = this;
					queue[hash] = {"uids": [], "timer": null};
					queue[hash].timer = window.setTimeout(function () {
						// Fetch the data
						self.dataFetch(_execId, {
								"start": 0,
								"num_rows": 0,
								"only_data": true,
								"refresh": queue[hash].uids
							},
							[], _widgetId, null, _context, null);

						// Delete the queue entry
						delete queue[hash];
					}, 100);
				}

				// Push the uid onto the queue, removing the prefix
				var parts = _uid.split("::");
				parts.shift();
				if (queue[hash].uids.indexOf(parts.join("::")) === -1)
				{
					queue[hash].uids.push(parts.join('::'));
				}
			}
			else
			{
				this.debug("log", "Data for uid " + _uid + " not available.");
			}
		},

		/**
		 * Unregisters the intrest of updates for a certain data uid.
		 *
		 * @param _uid is the data uid for which the callbacks should be
		 * 	unregistered.
		 * @param _callback specifies the specific callback that should be
		 * 	unregistered. If it evaluates to false, all callbacks (or those
		 * 	matching the optionally given context) are removed.
		 * @param _context specifies the callback context that should be
		 * 	unregistered. If it evaluates to false, all callbacks (or those
		 * 	matching the optionally given callback function) are removed.
		 */
		dataUnregisterUID: function (_uid, _callback, _context) {

			// Force the optional parameters to be exactly null
			_callback = _callback ? _callback : null;
			_context = _context ? _context : null;

			if (typeof registeredCallbacks[_uid] !== "undefined")
			{

				// Iterate over the registered callbacks for that uid and delete
				// all callbacks pointing to the given callback and context
				for (var i = registeredCallbacks[_uid].length - 1; i >= 0; i--)
				{
					if ((!_callback || registeredCallbacks[_uid][i].callback === _callback)
					    && (!_context || registeredCallbacks[_uid][i].context === _context))
					{
						registeredCallbacks[_uid].splice(i, 1);
					}
				}

				// Delete the slot if no callback is left for the uid
				if (registeredCallbacks[_uid].length === 0)
				{
					delete registeredCallbacks[_uid];
				}
			}
		},

		/**
		 * Returns whether data is available for the given uid.
		 *
		 * @param _uid is the uid for which should be checked whether it has some
		 * 	data.
		 */
		dataHasUID: function (_uid) {
			return typeof localStorage[_uid] !== "undefined";
		},

		/**
		 * Returns data of a given uid.
		 *
		 * @param _uid is the uid for which should be checked whether it has some
		 * 	data.
		 */
		dataGetUIDdata: function (_uid) {
			return localStorage[_uid];
		},

		/**
		 * Returns all uids that have the given prefix
		 *
		 * @param {string} _prefix
		 * @return {array}
		 * TODO: Improve this
		 */
		dataKnownUIDs: function (_prefix) {

			var result = [];

			for (var key in localStorage)
			{
				var parts = key.split("::");
				if (parts.shift() === _prefix && localStorage[key].data)
				{

					result.push(parts.join('::'));
				}
			}

			return result;
		},

		/**
		 * Stores data for the uid and calls all callback functions registered
		 * for that uid.
		 *
		 * @param _uid is the uid for which the data should be saved.
		 * @param _data is the data which should be saved.
		 */
		dataStoreUID: function (_uid, _data) {
			// Get the current unix timestamp
			var timestamp = (new Date).getTime();

			// Store the data in the local storage
			localStorage[_uid] = {
				"timestamp": timestamp,
				"data": _data
			};

			// Inform all registered callback functions and pass the data to
			// those.
			if (typeof registeredCallbacks[_uid] != "undefined")
			{
				for (var i = registeredCallbacks[_uid].length - 1; i >= 0; i--)
				{
					try {
						registeredCallbacks[_uid][i].callback.call(
							registeredCallbacks[_uid][i].context,
							_data,
							_uid
						);
					} catch (e) {
						// Remove this callback from the list
						if(typeof registeredCallbacks[_uid] != "undefined")
						{
							registeredCallbacks[_uid].splice(i, 1);
						}
					}
				}
			}
		},

		/**
		 * Deletes the data for a certain uid from the local storage and
		 * unregisters all callback functions associated to it.
		 *
		 * This does NOT update nextmatch!
		 * Application code should use: egw(window).refresh(msg, app, id, "delete");
		 *
		 * @param _uid is the uid which should be deleted.
		 */
		dataDeleteUID: function (_uid) {
			if (typeof localStorage[_uid] !== "undefined")
			{
				// Delete the element from the local storage
				delete localStorage[_uid];

				// Unregister all callbacks for that uid
				this.dataUnregisterUID(_uid);
			}
		},

		/**
		 * Force a refreash of the given uid from the server if known, and
		 * calls all associated callbacks.
		 *
		 * If the UID does not have any registered callbacks, it cannot be refreshed because the required
		 * execID and context are missing.
		 *
		 * @param {string} _uid is the uid which should be refreshed.
		 * @return {boolean} True if the uid is known and can be refreshed, false if unknown and will not be refreshed
		 */
		dataRefreshUID: function (_uid) {
			if (typeof localStorage[_uid] === "undefined") return false;

			if(typeof registeredCallbacks[_uid] !== "undefined" && registeredCallbacks[_uid].length > 0)
			{
				var _execId = registeredCallbacks[_uid][0].execId;
				// This widget ID MUST be a nextmatch, because the data call is to Etemplate\Widget\Nexmatch
				var nextmatchId = registeredCallbacks[_uid][0].widgetId;
				var uid = _uid.split("::");
				var context = {
					"prefix":uid.shift()
				};
				uid = uid.join("::");

				// find filters, even if context is not always from nextmatch, eg. caching uses it's a string context
				var filters = {};
				for(var i=0; i < registeredCallbacks[_uid].length; i++)
				{
					var callback = registeredCallbacks[_uid][i];
					if (typeof callback.context == 'object' &&
						typeof callback.context.self == 'object' &&
						typeof callback.context.self._filters == 'object')
					{
						filters = callback.context.self._filters;
						break;
					}
				}

				// need to send nextmatch filters too, as server-side will merge old version from request otherwise
				this.dataFetch(_execId, {'refresh':uid}, filters, nextmatchId, false, context, [uid]);

				return true;
			}
			return false;
		},

		/**
		 * Search for exact UID string or regular expression and return widgets using it
		 *
		 * @param {string|RegExp} _uid is the uid which should be refreshed.
		 * @return {object} UID: array of (nextmatch-)wigetIds
		 */
		dataSearchUIDs: function(_uid)
		{
			var matches = {};
			var f = function(_uid)
			{
				if (typeof matches[_uid] == "undefined")
				{
					matches[_uid] = [];
				}
				if (typeof registeredCallbacks[_uid] !== "undefined")
				{
					for(var n=0; n < registeredCallbacks[_uid].length; ++n)
					{
						var callback = registeredCallbacks[_uid][n];
						if (typeof callback.context != "undefined" &&
							typeof callback.context.self != "undefined" &&
							typeof callback.context.self._widget != "undefined")
						{
							matches[_uid].push(callback.context.self._widget);
						}
					}
				}
			};
			if (typeof _uid == "object" && _uid.constructor.name == "RegExp")
			{
				for(var uid in localStorage)
				{
					if (_uid.test(uid))
					{
						f(uid);
					}
				}
			}
			else if (typeof localStorage[_uid] != "undefined")
			{
				f(_uid);
			}
			return matches;
		},

		/**
		 * Search for exact UID string or regular expression and call registered (nextmatch-)widgets refresh function with given _type
		 *
		 * This method is preferable over dataRefreshUID for app code, as it takes care of things like counters too.
		 *
		 * It does not do anything for _type="add"!
		 *
		 * @param {string|RegExp) _uid is the uid which should be refreshed.
		 * @param {string} _type "delete", "edit", "update", not useful for "add"!
		 * @return {array} (nextmatch-)wigets refreshed
		 */
		dataRefreshUIDs: function(_uid, _type)
		{
			var uids = this.dataSearchUIDs(_uid);
			var widgets = [];
			var uids4widget = [];
			for(var uid in uids)
			{
				for(var n=0; n < uids[uid].length; ++n)
				{
					var widget = uids[uid][n];
					var idx = widgets.indexOf(widget);
					if (idx == -1)
					{
						widgets.push(widget);
						idx = widgets.length-1;
					}
					// uids for nextmatch.refesh do NOT contain the prefix
					var nm_uid = uid.replace(RegExp('^'+widget.controller.dataStorePrefix+'::'), '');
					if (typeof uids4widget[idx] == "undefined")
					{
						uids4widget[idx] = [nm_uid];
					}
					else
					{
						uids4widget[idx].push(nm_uid);
					}
				}
			}
			for(var w=0; w < widgets.length; ++w)
			{
				widgets[w].refresh(uids4widget[w], _type);
			}
			return widgets;
		}
	};
});

/**
 * EGroupware clientside egw tail
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package api
 * @subpackage jsapi
 * @link https://www.egroupware.org
 * @author Hadi Nategh (as AT stylite.de)
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 */

jQuery(function()
{
	"use strict";

	var log_tail_start=0;
	var filename = jQuery('pre[id^="log"]');
	if (typeof filename !='undefined' && filename.length > 0)
	{
		filename = filename.attr('data-filename');
	}
	function button_log(buttonId)
	{
		if (buttonId != "clear_log")
		{
			egw.json("api.EGroupware\\Api\\Json\\Tail.ajax_delete",[filename,buttonId=="empty_log"])
				.sendRequest(true);
		}
		jQuery("#log").text("");
	}
	function refresh_log()
	{
		egw.json("api.EGroupware\\Api\\Json\\Tail.ajax_chunk",[filename,log_tail_start], function(_data)
		{
			if (_data.length) {
				log_tail_start = _data.next;
				var log = jQuery("#log").append(_data.content.replace(/</g,"&lt;"));
				log.animate({ scrollTop: log.prop("scrollHeight") - log.height() + 20 }, 500);
			}
			if (_data.size === false)
			{
				jQuery("#download_log").hide();
			}
			else
			{
				jQuery("#download_log").show().attr("title", egw(window).lang('Size')+_data.size);
			}
			if (_data.writable === false)
			{
				jQuery("#purge_log").hide();
				jQuery("#empty_log").hide();
			}
			else
			{
				jQuery("#purge_log").show();
				jQuery("#empty_log").show();
			}
			window.setTimeout(refresh_log,_data.length?200:2000);
		}).sendRequest(true);
	}
	function resize_log()
	{
		jQuery("#log").width(egw_getWindowInnerWidth()-20).height(egw_getWindowInnerHeight()-33);
	}
	jQuery('input[id^="clear_log"]').on('click',function(){
		button_log(this.getAttribute('id'));
	});
	jQuery('input[id^="purge_log"]').on('click',function(){
		button_log(this.getAttribute('id'));
	});
	jQuery('input[id^="empty_log"]').on('click',function(){
		button_log(this.getAttribute('id'));
	});
	//egw_LAB.wait(function() {
		jQuery(document).ready(function()
		{
			if (typeof filename !='undefined' && filename.length > 0)
			{
				resize_log();
				refresh_log();
			}
		});
		jQuery(window).resize(resize_log);
	//});
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 * @version $Id$
 */

/**
 * Methods to display a success or error message and the app-header
 *
 * @augments Class
 * @param {string} _app application name object is instanciated for
 * @param {object} _wnd window object is instanciated for
 */
egw.extend('message', egw.MODULE_WND_LOCAL, function(_app, _wnd)
{
	"use strict";

	_app;	// not used, but required by function signature
	var error_reg_exp;
	var a_href_reg = /<a href="([^"]+)">([^<]+)<\/a>/img;
	var new_line_reg = /<\/?(p|br)\s*\/?>\n?/ig;
	// keeps alive messages stored
	var alive_messages = [];
	// Register an 'error' plugin, displaying using the message system
	window.setTimeout(() =>
	{
		egw(_wnd).registerJSONPlugin(function (type, res, req)
		{
			if (typeof res.data == 'string')
			{
				egw.message(res.data, 'error');
				return true;
			}
			throw 'Invalid parameters';
		}, null, 'error');
	}, 0);

	/**
	 * Decode html entities so they can be added via .text(_str), eg. html_entity_decode('&amp;') === '&'
	 *
	 * @param {string} _str
	 * @returns {string}
	 */
	function html_entity_decode(_str)
	{
		return _str && _str.indexOf('&') != -1 ? jQuery('<span>'+_str+'</span>').text() : _str;
	}

	return {
		/**
		 * Display an error or regular message
		 *
		 * All messages, but type "success", are displayed 'til next message or user clicks on it.
		 *
		 * @param {string} _msg message to show or empty to remove previous message
		 * @param {string} _type 'help', 'info', 'error', 'warning' or 'success' (default)
		 * @param {string} _discardID unique string id (appname:id) in order to register
		 * the message as discardable. If no appname given, the id will be prefixed with
		 * current app. The discardID will be stored in local storage.
		 *
		 * @return {object} returns an object containing data and methods related to the message
		 */
		message: function(_msg, _type, _discardID)
		{
			var jQuery = _wnd.jQuery;
			var wrapper = jQuery('.egw_message_wrapper').length > 0 ? jQuery('.egw_message_wrapper')
				: jQuery(_wnd.document.createElement('div')).addClass('egw_message_wrapper noPrint').css('position', 'absolute');

			// add popup indicator class to be able to distinguish between mainframe message or popup message
			if (this.is_popup()) wrapper.addClass('isPopup');

			if (_msg && !_type)
			{
				if (typeof error_reg_exp == 'undefined') error_reg_exp = new RegExp('(error|'+egw.lang('error')+')', 'i');

				_type = _msg.match(error_reg_exp) ? 'error' : 'success';
			}

			// if we are NOT in a popup then call the message on top window
			if (!this.is_popup() && _wnd !== egw.top)
			{
				egw(egw.top).message(_msg, _type);
				return;
			}

			var parent = jQuery('div#divAppboxHeader');
			// popup has no app-header (idots) or it is hidden by onlyPrint class (jdots) --> use body
			if (!parent.length || parent.hasClass('onlyPrint'))
			{
				parent = jQuery('body');
			}

			for (var m in alive_messages)
			{
				// Do not add a same message twice if it's still not dismissed
				if (alive_messages[m] == _msg) return;
			}

			if (_msg)	// empty _msg just removes pervious message
			{
				// keeps alive messages
				alive_messages.push(_msg);
				// message index in stack
				var msg_index = alive_messages.length-1;

				// replace p and br-tags with newlines
				_msg = _msg.replace(new_line_reg, "\n");
				var msg_div = jQuery(_wnd.document.createElement('div'))
					.attr('id','egw_message')
					.text(_msg)
					.addClass(_type+'_message')
					.click(function(){
						if (_type == 'success')
						{
							delete(alive_messages[msg_index]);
							jQuery(msg_div).remove();
						}
					})
					.prependTo(wrapper);
				var msg_close = jQuery(_wnd.document.createElement('span'))
					.click(function() {
						//check if the messeage should be discarded forever
						if (_type == 'info' && _discardID
							&& msg_chkbox && msg_chkbox.is(':checked'))
						{
							var discarded = egw.getLocalStorageItem(discardAppName,'discardedMsgs');

							if (!isDiscarded(_discardID))
							{
								if (!discarded)
								{
									discarded = [_discardID];
								}
								else
								{
									if (jQuery.isArray(discarded = JSON.parse(discarded))) discarded.push(_discardID);
								}
								egw.setLocalStorageItem(discardAppName,'discardedMsgs',JSON.stringify(discarded));
							}
						}
						delete(alive_messages[msg_index]);
						jQuery(msg_div).remove();
					})
					.addClass('close')
					.appendTo(msg_div);
				if (_type == 'success')	msg_close.hide();
				// discard checkbox implementation
				if (_discardID && _type === 'info')
				{
					var discardID = _discardID.split(':');
					if (discardID.length<2)
					{
						_discardID = egw.app_name() +":"+_discardID;
					}
					var discardAppName = discardID.length>1? discardID[0]: egw.app_name();


					// helper function to check if the messaege is discarded
					var isDiscarded = function (_id)
					{

						var discarded = JSON.parse(egw.getLocalStorageItem(discardAppName,'discardedMsgs'));

						if (jQuery.isArray(discarded))
						{
							for(var i=0; i< discarded.length; i++)
							{
								if (discarded[i] === _id) return true;
							}
						}
						return false;
					};

					//discard div container
					var msg_discard =jQuery(_wnd.document.createElement('div')).addClass('discard');

					// checkbox
					var msg_chkbox = jQuery(_wnd.document.createElement('input'))
							.attr({type:"checkbox",name:"msgChkbox"})
							.click(function(e){e.stopImmediatePropagation();})
							.appendTo(msg_discard);
					// Label
					jQuery(_wnd.document.createElement('label'))
								.text(egw.lang("Don't show this again"))
								.css({"font-weight":"bold"})
								.attr({for:'msgChkbox'})
								.appendTo(msg_discard);

					if (isDiscarded(_discardID)) return;
					msg_div.append(msg_discard);
				}

				parent.prepend(wrapper);

				// replace simple a href (NO other attribute, to gard agains XSS!)
				var matches = a_href_reg.exec(_msg);
				if (matches)
				{
					var parts = _msg.split(matches[0]);
					var href = html_entity_decode(matches[1]);
					msg_div.text(parts[0]);
					msg_div.append(jQuery(_wnd.document.createElement('a'))
						.attr({href: href, target: href.indexOf(egw.webserverUrl) != 0 ? '_blank' : '_self'})
						.text(matches[2]));
					msg_div.append(jQuery(_wnd.document.createElement('span')).text(parts[1]));
				}
				// center the message
				wrapper.css('right', ((jQuery(_wnd).innerWidth()-msg_div.width())/2)+'px');

				if (_type == 'success')	// clear message again after some time, if no error
				{
					_wnd.setTimeout(function() {
						msg_div.remove();
						delete(alive_messages[msg_index]);
					}, 5000);
				}
			}
			return {
				node: msg_div,
				message: _msg,
				index: msg_index,
				close: function(){msg_close.click();}
			};
		},

		/**
		 * Are we running in a popup
		 *
		 * @returns {boolean} true: popup, false: main window
		 */
		is_popup: function ()
		{
			var popup = false;
			try {
				if (_wnd.opener && _wnd.opener != _wnd && typeof _wnd.opener.top.egw == 'function')
				{
					popup = true;
				}
			}
			catch(e) {
				// ignore SecurityError exception if opener is different security context / cross-origin
			}
			return popup;
		},

		/**
		 * Active app independent if we are using a framed template-set or not
		 *
		 * @returns {string}
		 */
		app_name: function()
		{
			return !this.is_popup() && _wnd.framework && _wnd.framework.activeApp ? _wnd.framework.activeApp.appName : _wnd.egw_appName;
		},

		/**
		 * Update app-header and website-title
		 *
		 * @param {string} _header
		 * @param {string} _app Application name, if not for the current app
		 */
		app_header: function(_header,_app)
		{
			// not for popups and only for framed templates
			if (!this.is_popup() && _wnd.framework && _wnd.framework.setWebsiteTitle)
			{
				var app = _app || this.app_name();
				var title = _wnd.document.title.replace(/[.*]$/, '['+_header+']');

				_wnd.framework.setWebsiteTitle.call(_wnd.framework, app, title, _header);
				return;
			}
			if (_wnd.document.querySelector('div#divAppboxHeader'))
			{
				_wnd.document.querySelector('div#divAppboxHeader').textContent = _header;
			}

			_wnd.document.title = _wnd.document.title.replace(/[.*]$/, '['+_header+']');
		},

		/**
		 * Loading prompt is for building a loading animation and show it to user
		 * while a request is under progress.
		 *
		 * @param {string} _id a unique id to be able to distinguish loading-prompts
		 * @param {boolean} _stat true to show the loading and false to remove it
		 * @param {string} _msg a message to show while loading
		 * @param {string|jQuery _node} _node DOM selector id or jquery DOM object, default is body
		 * @param {string} _mode	defines the animation mode, default mode is spinner
		 *	animation modes:
		 *		- spinner: a sphere with a spinning bar inside
		 *		- horizental: a horizental bar
		 *
		 * @returns {jquery dom object|null} returns jQuery DOM object or null in case of hiding
		 */
		loading_prompt: function(_id,_stat,_msg,_node, _mode)
		{
			var $container = '';
			var jQuery = _wnd.jQuery;

			var id = _id? 'egw-loadin-prompt_'+_id: 'egw-loading-prompt_1';
			var mode = _mode || 'spinner';
			if (_stat)
			{
				var $node = _node? jQuery(_node): jQuery('body');

				var $container = jQuery(_wnd.document.createElement('div'))
						.attr('id', id)
						.addClass('egw-loading-prompt-container ui-front');

				var $text = jQuery(_wnd.document.createElement('span'))
						.addClass('egw-loading-prompt-'+mode+'-msg')
						.text(_msg)
						.appendTo($container);
				var $animator = jQuery(_wnd.document.createElement('div'))
						.addClass('egw-loading-prompt-'+mode+'-animator')
						.appendTo($container);
				if (!_wnd.document.getElementById(id)) $container.insertBefore($node);
				return $container;
			}
			else
			{
				$container = jQuery(_wnd.document.getElementById(id));
				if ($container.length > 0) $container.remove();
				return null;
			}
		},

		/**
		 * Refresh given application _targetapp display of entry _app _id, incl. outputting _msg
		 *
		 * Default implementation here only reloads window with it's current url with an added msg=_msg attached
		 *
		 * @param {string} _msg message (already translated) to show, eg. 'Entry deleted'
		 * @param {string} _app application name
		 * @param {(string|number)} _id id of entry to refresh or null
		 * @param {string} _type either 'update', 'edit', 'delete', 'add' or null
		 * - update: request just modified data from given rows.  Sorting is not considered,
		 *		so if the sort field is changed, the row will not be moved.
		 * - update-in-place: update row, but do NOT move it, or refresh if uid does not exist
		 * - edit: rows changed, but sorting may be affected.  Requires full reload.
		 * - delete: just delete the given rows clientside (no server interaction neccessary)
		 * - add: requires full reload for proper sorting
		 * @param {string} _targetapp which app's window should be refreshed, default current
		 * @param {(string|RegExp)} _replace regular expression to replace in url
		 * @param {string} _with
		 * @param {string} _msg_type 'error', 'warning' or 'success' (default)
		 * @param {object|null} _links app => array of ids of linked entries
		 * or null, if not triggered on server-side, which adds that info
		 */
	   refresh: function(_msg, _app, _id, _type, _targetapp, _replace, _with, _msg_type, _links)
	   {
			// Log for debugging purposes
			this.debug("log", "egw_refresh(%s, %s, %s, %o, %s, %s)", _msg, _app, _id, _type, _targetapp, _replace, _with, _msg_type, _links);

			var win = _targetapp ? _wnd.egw_appWindow(_targetapp) : _wnd;

			this.message(_msg, _msg_type);

			if(typeof _links == "undefined")
			{
				_links = [];
			}

			// notify app observers: if observer for _app itself returns false, no regular refresh will take place
			// app's own observer can replace current app_refresh functionality
			var no_regular_refresh = false;
			for(var app_obj of _wnd.egw.window.EgwApp)	// run observers in main window (eg. not iframe, which might be opener!)
			{
				if (typeof app_obj.observer == 'function' &&
					app_obj.observer(_msg, _app, _id, _type, _msg_type, _links) === false && app_obj.appname === _app)
				{
					no_regular_refresh = true;
				}
			}
			if (no_regular_refresh) return;

			// if we have a framework template, let it deal with refresh, unless it returns a DOMwindow for us to refresh
			if (win.framework && win.framework.refresh &&
				!(win = win.framework.refresh(_msg, _app, _id, _type, _targetapp, _replace, _with, _msg_type)))
			{
				return;
			}

			// if window registered an app_refresh method or overwritten app_refresh, just call it
			if(typeof win.app_refresh == "function" && typeof win.app_refresh.registered == "undefined" ||
				typeof win.app_refresh != "undefined" && win.app_refresh.registered(_app))
			{
				win.app_refresh(_msg, _app, _id, _type);
				return;
			}

			// etemplate2 specific to avoid reloading whole page
			if(typeof win.etemplate2 != "undefined" && win.etemplate2.app_refresh)
			{
				var refresh_done = win.etemplate2.app_refresh(_msg, _app, _id, _type);

				// Refresh target or current app too
				if ((_targetapp || this.app_name()) != _app)
				{
					refresh_done = win.etemplate2.app_refresh(_msg, _targetapp || this.app_name()) || refresh_done;
				}
				//In case that we have etemplate2 ready but it's empty and refresh is not done
				if (refresh_done) return;
			}

			// fallback refresh by reloading window
			var href = win.location.href;

			if (typeof _replace != 'undefined')
			{
				href = href.replace(typeof _replace == 'string' ? new RegExp(_replace) : _replace, (typeof _with != 'undefined' && _with != null) ? _with : '');
			}

			if (href.indexOf('msg=') != -1)
			{
				href = href.replace(/msg=[^&]*/,'msg='+encodeURIComponent(_msg));
			}
			else if (_msg)
			{
				href += (href.indexOf('?') != -1 ? '&' : '?') + 'msg=' + encodeURIComponent(_msg);
			}
			//alert('egw_refresh() about to call '+href);
			win.location.href = href;
	   },


		/**
		 * Handle a push notification about entry changes from the websocket
		 *
		 * @param  pushData
		 * @param {string} pushData.app application name
		 * @param {(string|number)} pushData.id id of entry to refresh or null
		 * @param {string} pushData.type either 'update', 'edit', 'delete', 'add' or null
		 * - update: request just modified data from given rows.  Sorting is not considered,
		 *		so if the sort field is changed, the row will not be moved.
		 * - edit: rows changed, but sorting may be affected.  Requires full reload.
		 * - delete: just delete the given rows clientside (no server interaction neccessary)
		 * - add: requires full reload for proper sorting
		 * @param {object|null} pushData.acl Extra data for determining relevance.  eg: owner or responsible to decide if update is necessary
		 * @param {number} pushData.account_id User that caused the notification
		 */
		push: function(pushData)
		{
			// Log for debugging purposes
			this.debug("log", "push(%o)", pushData);

			if (typeof pushData == "undefined")
			{
				this.debug('warn', "Push sent nothing");
				return;
			}

			// notify app observers
			for (var app_obj of _wnd.egw.window.EgwApp)	// run observers in main window (eg. not iframe, which might be opener!)
			{
				if (typeof app_obj.push == 'function')
				{
					app_obj.push(pushData);
				}
			}

			// call the global registered push callbacks
			this.registerPush(pushData);
		}
	};

});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link http://www.egroupware.org
 * @author Hadi Nategh <hn-AT-stylite.de>
 * @version $Id: $
 */

/**
 * Methods to display browser notification
 *
 * @augments Class
 * @param {string} _app application name object is instanciated for
 * @param {object} _wnd window object is instanciated for
 *
 * @return {object} defined functions of module
 */
egw.extend('notification', egw.MODULE_WND_LOCAL, function(_app, _wnd)
{
	"use strict";

	// Notification permission, the default value is 'default' which is equivalent to 'denied'
	var permission = 'default';
	// Keeps alive notifications
	var alive_notifications = [];

	if (typeof Notification != 'undefined')
	{
		permission = Notification.permission;
	}

	return {

		/**
		 *
		 * @param {string} _title a string to be shown as notification message
		 * @param {object} _options an object of Notification possible options:
		 *		options = {
		 *			dir:  // direction of notification to be shown rtl, ltr or auto
		 *			lang: // a valid BCP 47 language tag
		 *			body: // DOM body
		 *			icon: // parse icon URL, default icon is app icon
		 *			tag: // a string value used for tagging an instance of notification, default is app name
		 *			requireInteraction: // boolean value indicating that a notification should remain active until the user clicks or dismisses it
		 *			onclick: // Callback function dispatches on click on notification message
		 *			onshow: // Callback function dispatches when notification is shown
		 *			onclose: // Callback function dispateches on notification close
		 *			onerror: // Callback function dispatches on error, default is a egw.debug log
		 *		}
		 *	@return {boolean} false if Notification is not supported by browser
		 */
		notification: function (_title, _options)
		{
			// Check if the notification is supported by  browser
			if (typeof Notification == 'undefined') return false;

			var self = this;
			// Check and ask for permission
			if (Notification && Notification.requestPermission && permission === 'default') Notification.requestPermission (function(_permission){
				permission = _permission;
				if (permission === 'granted') self.notification(_title,_options);
			});

			// All options including callbacks
			var options = _options || {};

			// Options used for creating Notification instane
			var inst_options = {
				tag: options.tag || egw.app_name(),
				dir: options.dir || 'ltr',
				lang: options.lang || egw.preference('lang', 'common'),
				body: options.body || '',
				icon: options.icon || egw.image('navbar', egw.app_name()),
				requireInteraction: options.requireInteraction || false
			};

			// Create an  instance of Notification object
			var notification = new Notification(_title, inst_options);

			//set timer to close shown notification in 10 s, some browsers do not
			//close it automatically.
			setTimeout(notification.close.bind(notification), 10000);

			// Callback function dispatches on click on notification message
			notification.onclick = options.onclick || '';
			// Callback function dispatches when notification is shown
			notification.onshow = options.onshow || '';
			// Callback function dispateches on notification close
			notification.onclose = options.onclose || '';
			// Callback function dispatches on error
			notification.onerror = options.onerror || function (e) {egw.debug('Notification failed because of ' + e);};

			// Collect all running notifications in case if want to close them all,
			// for instance on logout action.
			alive_notifications.push(notification);
		},

		/**
		 * Check Notification availability by browser
		 *
		 * @returns {Boolean} true if notification is supported and permitted otherwise false
		 */
		checkNotification: function () {
			// Check if the notification is supported by  browser
			if (typeof Notification == 'undefined') return false;
			
			// Ask for permission if there's nothing decided yet
			if (Notification && Notification.requestPermission && permission == 'default') {
				Notification.requestPermission (function(_permission){
					permission = _permission;
				});
			}
			return (Notification && Notification.requestPermission && permission == 'granted');
		},

		/**
		 * Check if there's any runnig notifications and will close them all
		 *
		 */
		killAliveNotifications: function ()
		{
			if (alive_notifications && alive_notifications.length > 0)
			{
				for (var i=0; i<alive_notifications.length;i++)
				{
					if (typeof alive_notifications[i].close == 'function') alive_notifications[i].close();
				}
				alive_notifications = [];
			}
		}
	};
});

/**
 * EGroupware clientside API object
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package etemplate
 * @subpackage api
 * @link https://www.egroupware.org
 * @author Ralf Becker <RalfBecker@outdoor-training.de>
 */

egw.extend('timer', egw.MODULE_GLOBAL, function()
{
	"use strict";

	/**
	 * Overall timer state
	 */
	let overall = {};
	/**
	 * Specific timer state
	 */
	let specific = {};
	/**
	 * Disable config with values "overall", "specific" or "overwrite"
	 * @type {string[]}
	 */
	let disable = [];
	/**
	 * Timer container in top-menu
	 * @type {Element}
	 */
	const timer = document.querySelector('#topmenu_timer');
	/**
	 * Reference from setInterval to stop periodic update
	 */
	let timer_interval;
	/**
	 * Reference to open dialog or undefined if not open
	 * @type {Et2-dialog}
	 */
	let dialog;

	/**
	 * Set state of timer
	 *
	 * @param _state
	 */
	function setState(_state)
	{
		disable = _state.disable;
		// initiate overall timer
		startTimer(overall, _state.overall?.start, _state.overall?.offset);	// to show offset / paused time
		overall.started = _state.overall?.started ? new Date(_state.overall.started) : undefined;
		overall.started_id = _state.overall?.started_id;
		if (_state.overall?.paused)
		{
			stopTimer(overall, true);
		}
		else if (!_state.overall?.start)
		{
			stopTimer(overall);
		}
		overall.last = _state.overall.last ? new Date(_state.overall.last) : undefined;
		overall.id = _state.overall?.id;

		// initiate specific timer, only if running or paused
		if (_state.specific?.start || _state.specific?.paused)
		{
			startTimer(specific, _state.specific?.start, _state.specific?.offset, _state.specific.app_id);	// to show offset / paused time
			specific.started = _state.specific?.started ? new Date(_state.specific.started) : undefined;
			specific.started_id = _state.specific?.started_id;
			specific.id = _state.specific.id;
			if (_state.specific?.paused)
			{
				stopTimer(specific, true);
			}
			else if (!_state.specific?.start)
			{
				stopTimer(specific);
			}
		}
		specific.last = _state.specific.last ? new Date(_state.specific.last) : undefined;
		specific.id = _state.specific?.id;
	}

	/**
	 * Get state of timer
	 * @param string _action last action
	 * @param string|Date|undefined _time time to report
	 * @returns {{action: string, overall: {}, specific: {}, ts: Date}}
	 */
	function getState(_action, _time)
	{
		return {
			action: _action,
			ts: new Date(_time || new Date),
			overall: overall,
			specific: specific
		}
	}

	/**
	 * Run timer action eg. start/stop
	 *
	 * @param {string} _action
	 * @param {string} _time
	 * @param {string} _app_id
	 * @return Promise from egw.request() to wait for state being persisted on server
	 * @throws string error-message
	 */
	function timerAction(_action, _time, _app_id)
	{
		const [type, action] = _action.split('-');
		switch(_action)
		{
			case 'overall-start':
				startTimer(overall, _time);
				break;

			case 'overall-pause':
				stopTimer(overall,true, _time);
				if (specific?.start) stopTimer(specific, true, _time);
				break;

			case 'overall-stop':
				stopTimer(overall, false, _time);
				if (specific?.start) stopTimer(specific, false, _time);
				break;

			case 'specific-start':
				if (overall?.paused) startTimer(overall, _time);
				startTimer(specific, _time);
				break;

			case 'specific-pause':
				stopTimer(specific,true, _time);
				break;

			case 'specific-stop':
				stopTimer(specific, false, _time);
				break;
		}
		// set _app_id on timer, if specified
		if (_app_id && type === 'specific')
		{
			specific.app_id = _app_id;
		}
		// persist state
		return egw.request('timesheet.EGroupware\\Timesheet\\Events.ajax_event', [getState(_action, _time)]).then((tse_id) =>
		{
			const timer = type === 'specific' ? specific : overall;
			// do NOT set/change timer.id, if a paused timer get stopped (to show and update paused time, not irrelevant stop)
			if (timer.start || typeof timer.paused !== 'undefined')
			{
				timer.id = tse_id;
			}
			if (action === 'start')
			{
				timer.started_id = tse_id;
			}
			if (_action === 'specific-stop')
			{
				let type = 'add';
				let extra = {events: 'specific'};
				if (specific.app_id && specific.app_id.substring(0, 11) === 'timesheet::')
				{
					extra.ts_id = specific.app_id.substring(11);
					type = 'edit';
				}
				egw.open(null, 'timesheet', type, extra);

				// unset the app_id and the tse_id to not associate the next start with it
				specific.app_id = undefined;
			}
		});
	}

	/**
	 * Enable/disable buttons based on timer state
	 */
	function setButtonState()
	{
		if (!dialog) return;

		// disable not matching / available menu-items
		dialog.querySelectorAll('et2-button').forEach(button =>
		{
			if (button.id.substring(0, 7) === 'overall')
			{
				// timer running: disable only start, enable pause and stop
				if (overall?.start)
				{
					button.disabled = button.id === 'overall[start]';
				}
				// timer paused: disable pause, enable start and stop
				else if (overall?.paused)
				{
					button.disabled = button.id === 'overall[pause]';
				}
				// timer stopped: disable stop and pause, enable start
				else
				{
					button.disabled = button.id !== 'overall[start]';
				}
			}
			else if (button.id.substring(0, 8) === 'specific')
			{
				// timer running: disable only start, enable pause and stop
				if (specific?.start)
				{
					button.disabled = button.id === 'specific[start]';
				}
				// timer paused: disable pause, enable start and stop
				else if (specific?.paused)
				{
					button.disabled = button.id === 'specific[pause]';
				}
				// timer stopped: disable stop and pause, enable start
				else
				{
					button.disabled = button.id !== 'specific[start]';
				}
			}
		});
	}

	/**
	 * Update the timer DOM node according to _timer state
	 *
	 * @param {DOMNode} _node
	 * @param {object} _timer
	 */
	function updateTimer(_node, _timer)
	{
		let sep = ':';
		let diff = Math.round((_timer.offset || 0) / 60000.0);
		if (_timer.start)
		{
			const now = Math.round((new Date()).valueOf() / 1000.0);
			sep = now % 2 ? ' ' : ':';
			diff = Math.round((now - Math.round(_timer.start.valueOf() / 1000.0)) / 60.0);
		}
		_node.textContent = sprintf('%d%s%02d', (diff / 60)|0, sep, diff % 60);
		// set CSS classes accordingly
		_node.classList.toggle('running', !!_timer.start);
		_node.classList.toggle('paused', _timer.paused || false);
		_node.classList.toggle('overall', _timer === overall);
	}

	/**
	 * Update all timers: topmenu and dialog (if open)
	 */
	function update()
	{
		// topmenu only shows either specific, if running or paused, or the overall timer
		updateTimer(timer, specific.start || specific.paused ? specific : overall);

		// if dialog is open, it shows both timers
		if (dialog)
		{
			const specific_timer = dialog.querySelector('div#_specific_timer');
			const overall_timer = dialog.querySelector('div#_overall_timer');
			if (specific_timer)
			{
				updateTimer(specific_timer, specific);
			}
			if (overall_timer)
			{
				updateTimer(overall_timer, overall);
			}
		}
	}


	/**
	 * Start given timer
	 *
	 * @param object _timer
	 * @param string|Date|undefined _start to initialise with time different from current time
	 * @param number|undefined _offset to set an offset
	 */
	function startTimer(_timer, _start, _offset)
	{
		_timer.started = _start ? new Date(_start) : new Date();
		_timer.started.setSeconds(0);	// only use full minutes, as this is what we display
		if (_timer.last && _timer.started.valueOf() < _timer.last.valueOf())
		{
			throw egw.lang('Start-time can not be before last stop- or pause-time %1!', formatUTCTime(_timer.last));
		}
		// update _timer state object
		_timer.start = new Date(_timer.last = _timer.started);

		if (_offset || _timer.offset && _timer.paused)
		{
			_timer.start.setMilliseconds(_timer.start.getMilliseconds()-(_offset || _timer.offset));
		}
		_timer.offset = 0;	// it's now set in start-time
		_timer.paused = false;
		_timer.app_id = undefined;

		// update now
		update();

		// initiate periodic update, if not already runing
		if (!timer_interval)
		{
			timer_interval = window.setInterval(update, 1000);
		}
	}

	/**
	 * Stop or pause timer
	 *
	 * If specific timer is stopped, it will automatically display the overall timer, if running or paused
	 *
	 * @param object _timer
	 * @param bool|undefined _pause true: pause, else: stop
	 * @param string|Date|undefined _time stop-time, default current time
	 * @throws string error-message when timer.start < _time
	 */
	function stopTimer(_timer, _pause, _time)
	{
		const time = _time ? new Date(_time) : new Date();
		time.setSeconds(0);	// only use full minutes, as this is what we display
		if (time.valueOf() < _timer.last.valueOf())
		{
			const last_time = formatUTCTime(_timer.last);
			if (_timer.start)
			{
				throw egw.lang('Stop- or pause-time can not be before the start-time %1!', last_time);
			}
			else
			{
				throw egw.lang('Start-time can not be before last stop- or pause-time %1!', last_time);
			}
		}
		// update _timer state object
		if (_timer.start)
		{
			if (time.valueOf() < _timer.start.valueOf())
			{
			}
			_timer.offset = time.valueOf() - _timer.start.valueOf();
			_timer.start = undefined;
		}
		// if we stop an already paused timer, we keep the paused event as last, not the stop
		if (_timer.paused)
		{
			_timer.paused = _pause || undefined;
		}
		else
		{
			_timer.last = time;
			_timer.paused = _pause || false;
		}
		// update timer display
		updateTimer(timer, _timer);

		// if dialog is shown, update its timer(s) too
		if (dialog)
		{
			const specific_timer = dialog.querySelector('div#_specific_timer');
			const overall_timer = dialog?.querySelector('div#_overall_timer');
			if (specific_timer && _timer === specific)
			{
				updateTimer(specific_timer, specific);
			}
			if (overall_timer && _timer === overall)
			{
				updateTimer(overall_timer, overall);
			}
		}

		// stop periodic update, only if NO more timer is running
		if (timer_interval && !specific.start && !overall.start)
		{
			window.clearInterval(timer_interval);
			timer_interval = undefined;
		}
	}

	/**
	 * Format a time according to user preference
	 *
	 * Cant import from DateTime.ts, gives an error ;)
	 *
	 * @param {Date} date
	 * @param {Object|undefined} options object containing attribute timeFormat=12|24, default user preference
	 * @returns {string}
	 */
	function formatTime(date, options)
	{
		if(!date || !(date instanceof Date))
		{
			return "";
		}
		let _value = '';

		let timeformat = options?.timeFormat || egw.preference("timeformat") || "24";
		let hours = (timeformat == "12" && date.getUTCHours() > 12) ? (date.getUTCHours() - 12) : date.getUTCHours();
		if(timeformat == "12" && hours == 0)
		{
			// 00:00 is 12:00 am
			hours = 12;
		}

		_value = (timeformat == "24" && hours < 10 ? "0" : "") + hours + ":" +
			(date.getUTCMinutes() < 10 ? "0" : "") + (date.getUTCMinutes()) +
			(timeformat == "24" ? "" : (date.getUTCHours() < 12 ? " am" : " pm"));

		return _value;
	}

	/**
	 * Format a UTC time according to user preference
	 *
	 * @param {Date} date
	 * @returns {string}
	 */
	function formatUTCTime(date)
	{
		// eT2 operates in user-time, while timers here always operate in UTC
		return formatTime(new Date(date.valueOf() - egw.getTimezoneOffset() * 60000));
	}

	/**
	 * Open the timer dialog to start/stop timers
	 *
	 * @param {string} _title default "Start & stop timer"
	 */
	function timerDialog(_title)
	{
		// Pass egw in the constructor
		dialog = new Et2Dialog(egw);

		// Set attributes.  They can be set in any way, but this is convenient.
		dialog.transformAttributes({
			// If you use a template, the second parameter will be the value of the template, as if it were submitted.
			callback: (button_id, value) =>		// return false to prevent dialog closing
			{
				dialog = undefined;
			},
			id: "timer_dialog",
			title: _title || 'Start & stop timer',
			template: egw.webserverUrl + '/timesheet/templates/default/timer.xet',
			buttons: [
				{label: egw.lang("Close"), id: "close", default: true, image: "cancel"},
			],
			value: {
				content: {
					disable: disable.join(':'),
					times: {
						specific: getTimes(specific),
						overall: getTimes(overall)
					}
				},
				sel_options: {}
			}
		});
		// Add to DOM, dialog will auto-open
		document.body.appendChild(dialog);
		dialog.updateComplete.then(() =>
		{
			// enable/disable buttons based on timer state
			setButtonState();
			// update timers in dialog
			update();
		});
	}

	/**
	 * Update times displayed under buttons
	 */
	function updateTimes()
	{
		if (!dialog) return;

		const times = {
			specific: getTimes(specific),
			overall: getTimes(overall)
		};

		// disable not matching / available menu-items
		dialog.querySelectorAll('et2-date-time-today').forEach(_widget =>
		{
			const [, timer, action] = _widget.id.match(/times\[([^\]]+)\]\[([^\]]+)\]/);
			_widget.value = times[timer][action];
		});
	}

	/**
	 * Get start, pause and stop time of timer to display in UI
	 *
	 * @param {Object} _timer
	 * @return {Object} with attributes start, pause, stop
	 */
	function getTimes(_timer)
	{
		const started = _timer.started ? new Date(_timer.started.valueOf() - egw.getTimezoneOffset() * 60000) : undefined;
		const last = _timer.last ? new Date(_timer.last.valueOf() - egw.getTimezoneOffset() * 60000) : undefined;
		return {
			start: started,
			paused: _timer.paused ? last : undefined,
			stop: !_timer.start && !_timer.paused ? last : undefined
		};
	}

	return {
		/**
		 * Change/overwrite time
		 *
		 * @param {PointerEvent} _ev
		 * @param {Et2DateTimeToday} _widget
		 */
		change_timer: function(_ev, _widget)
		{
			// if there is no value, or timer overwrite is disabled --> ignore click
			if (!_widget?.value || disable.indexOf('overwrite') !== -1) {
				return;
			}
			const [, which, action] = _widget.id.match(/times\[([^\]]+)\]\[([^\]]+)\]/);
			const timer = which === 'overall' ? overall : specific;
			const tse_id = timer[action === 'start' ? 'started_id' : 'id'];
			const dialog = new Et2Dialog(egw);

			// Set attributes.  They can be set in any way, but this is convenient.
			dialog.transformAttributes({
				callback: (_button, _values) => {
					const change = (new Date(_widget.value)).valueOf() - (new Date(_values.time)).valueOf();
					if (_button === Et2Dialog.OK_BUTTON && change)
					{
						_widget.value = _values.time;
						timer[action === 'start' ? 'started' : action] = new Date((new Date(_values.time)).valueOf() + egw.getTimezoneOffset() * 60000);
						// for a stopped or paused timer, we need to adjust the offset (duration) and the displayed timer too
						if (timer.offset)
						{
							timer.offset -= action === 'start' ? -change : change;
							update();
							// for stop/pause set last time, otherwise we might not able to start again directly after
							if (action !== 'start')
							{
								timer.last = new Date(timer[action]);
							}
						}
						// for a running timer, we need to adjust the (virtual) start too
						else if (timer.start)
						{
							timer.start = new Date(timer.start.valueOf() - change);
							// for running timer set last time, otherwise we might not able to stop directly after
							timer.last = new Date(timer.start);
						}
						egw.request('timesheet.EGroupware\\Timesheet\\Events.ajax_updateTime',
							[tse_id, new Date((new Date(_values.time)).valueOf() + egw.getTimezoneOffset() * 60000)]);
					}
				},
				title: egw.lang('Change time'),
				template: 'timesheet.timer.change',
				buttons: Et2Dialog.BUTTONS_OK_CANCEL,
				value: {
					content: { time: _widget.value }
				}
			});
			// Add to DOM, dialog will auto-open
			document.body.appendChild(dialog);
		},

		/**
		 * Start, Pause or Stop clicked in timer-dialog
		 *
		 * @param {Event} _ev
		 * @param {Et2Button} _button
		 */
		timer_button: function(_ev, _button)
		{
			const value = dialog.value;
			try {
				timerAction(_button.id.replace(/^([a-z]+)\[([a-z]+)\]$/, '$1-$2'),
					// eT2 operates in user-time, while timers here always operate in UTC
					value.time ? new Date((new Date(value.time)).valueOf() + egw.getTimezoneOffset() * 60000) : undefined);
			}
			catch (e) {
				Et2Dialog.alert(e, egw.lang('Invalid Input'), Et2Dialog.ERROR_MESSAGE);
			}
			setButtonState();
			updateTimes();
			return false;
		},

		/**
		 * Start timer for given app and id
		 *
		 * @param {Object} _action
		 * @param {Array} _senders
		 */
		start_timer: function(_action, _senders)
		{
			if (_action.parent.data.nextmatch?.getSelection().all || _senders.length !== 1)
			{
				egw.message(egw.lang('You must select a single entry!'), 'error');
				return;
			}
			// timer already running, ask user if he wants to associate it with the entry, or cancel
			if (specific.start || specific.paused)
			{
				Et2Dialog.show_dialog((_button) => {
						if (_button === Et2Dialog.OK_BUTTON)
						{
							if (specific.paused)
							{
								timerAction('specific-start', undefined, _senders[0].id);
							}
							else
							{
								specific.app_id = _senders[0].id;
								egw.request('timesheet.EGroupware\\Timesheet\\Events.ajax_updateAppId', [specific.id, specific.app_id]);
							}
						}
					},
					egw.lang('Do you want to associate it with the selected %1 entry?', egw.lang(_senders[0].id.split('::')[0])),
					egw.lang('Timer already running or paused'), {},
					Et2Dialog.BUTTONS_OK_CANCEL, Et2Dialog.QUESTION_MESSAGE, undefined, egw);
				return;
			}
			timerAction('specific-start', undefined, _senders[0].id);
		},

		/**
		 * Create timer in top-menu
		 *
		 * @param {string} _parent parent to create selectbox in
		 */
		add_timer: function(_parent)
		{
			const timer_container = document.getElementById(_parent);
			if (!timer_container) return;

			// set state if given
			const timer = document.getElementById('topmenu_timer');
			const state = timer && timer.getAttribute('data-state') ? JSON.parse(timer.getAttribute('data-state')) : undefined;
			if (timer && state)
			{
				setState(state);
			}

			// bind click handler
			timer_container.addEventListener('click', (ev) => {
				timerDialog();
			});

			// check if overall working time is not disabled
			if (state.disable.indexOf('overall') === -1)
			{
				// we need to wait that all JS is loaded
				window.egw_ready.then(() => { window.setTimeout(() =>
				{
					// check if we should ask on login to start working time
					this.preference('workingtime_session', 'timesheet', true).then(pref =>
					{
						if (pref === 'no') return;

						// overall timer not running, ask to start
						if (overall && !overall.start && !state.overall.dont_ask)
						{
							Et2Dialog.show_dialog((button) => {
								if (button === Et2Dialog.YES_BUTTON)
								{
									timerAction('overall-start');
								}
								else
								{
									egw.request('EGroupware\\Timesheet\\Events::ajax_dontAskAgainWorkingTime', button !== Et2Dialog.NO_BUTTON);
								}
							}, 'Do you want to start your working time?', 'Working time', {}, 		[
								{button_id: Et2Dialog.YES_BUTTON, label: egw.lang('yes'), id: 'dialog[yes]', image: 'check', "default": true},
								{button_id: Et2Dialog.NO_BUTTON, label: egw.lang('no'), id: 'dialog[no]', image: 'cancel'},
								{button_id: "dont_ask_again", label: egw.lang("Don't ask again!"), id: 'dialog[dont_ask_again]', image:'save', align: "right"}
							]);
						}
						// overall timer running for more than 16 hours, ask to stop
						else if (overall?.start && (((new Date()).valueOf() - overall.start.valueOf()) / 3600000) >= 16)
						{
							timerDialog('Forgot to switch off working time?');
						}
					});

				}, 2000);});
			}
		},

		/**
		 * Ask user to stop working time
		 *
		 * @returns {Promise<void>} resolved once user answered, to continue logout
		 */
		onLogout_timer: function()
		{
			let promise;
			if (overall.start || overall.paused)
			{
				promise = new Promise((_resolve, _reject) =>
				{
					Et2Dialog.show_dialog((button) => {
						if (button === Et2Dialog.YES_BUTTON)
						{
							timerAction('overall-stop').then(_resolve);
						}
						else
						{
							_resolve();
						}
					}, 'Do you want to stop your working time?', 'Working time', {}, Et2Dialog.BUTTONS_YES_NO);
				});
			}
			else
			{
				promise = Promise.resolve();
			}
			return promise;
		}
	};
});

/**
 * eGroupWare - API
 * http://www.egroupware.org
 *
 * This file was originally created Tyamad, but their content is now completly removed!
 * It still contains some commonly used javascript functions, always included by EGroupware.
 *
 * @license http://opensource.org/licenses/gpl-license.php GPL - GNU General Public License
 * @package api
 * @subpackage jsapi
 * @version $Id$
 */

'use strict';

/**
 * Check whether the console object is defined - if not, define one
 */
if (typeof window.console == 'undefined')
{
	window.console = {
		'log': function() {
		},
		'warn': function() {
		},
		'error': function() {
		},
		'info': function() {
		}
	};
}

/**
 * Seperates all script tags from the given html code and returns the seperately
 *
 * @param object _html object that the html code
 *  from which the script should be seperated
 *	The html code has to be stored in _html.html,
 *	the result js will be written to _html.js.
 */

window.egw_seperateJavaScript = function(_html)
{
	var html = typeof _html.html == 'string'?_html.html:'';

	var in_pos = html.search(/<script/im);
	var out_pos = html.search(/<\/script>/im);
	while (in_pos > -1 && out_pos > -1)
	{
		/*Seperate the whole <script...</script> tag */
		var js_str = html.substring(in_pos, out_pos+9);

		/*Remove the initial tag */
		/*js_str = js_str.substring(js_str.search(/>/) + 1);*/
		_html.js += js_str;


		html = html.substring(0, in_pos) + html.substring(out_pos + 9);

		var in_pos = html.search(/<script/im);
		var out_pos = html.search(/<\/script>/im);
	}

	_html.html = html;
};

/**
 * Inserts the script tags inside the given html into the dom tree
 */
window.egw_insertJS = function(_html)
{
	// Insert each script element seperately
	if (_html)
	{

		var in_pos = -1;
		var out_pos = -1;

		do {

			// Search in and out position
			var in_pos = _html.search(/<script/im);
			var out_pos = _html.search(/<\/script>/im);

			// Copy the text inside the script tags...
			if (in_pos > -1 && out_pos > -1)
			{
				if (out_pos > in_pos)
				{
					var scriptStart = _html.indexOf("\>", in_pos);
					if (scriptStart > in_pos)
					{
						var script = _html.substring(scriptStart + 1,
							out_pos);
						try
						{
							// And insert them as real script tags
							var tag = document.createElement("script");
							tag.setAttribute("type", "text/javascript");
							tag.text = script;
							document.getElementsByTagName("head")[0].appendChild(tag);
						}
						catch (e)
						{
							if (typeof console != "undefined" && typeof console.log != "undefined")
							{
								console.log('Error while inserting JS code:', _e);
							}
						}
					}
				}
				_html = _html.substr(out_pos + 9);
			}

		} while (in_pos > -1 && out_pos > -1)
	}
};

/**
 * Returns the top window which contains the current egw_instance, even for popup windows
 */
window.egw_topWindow = function()
{
	return egw.top;
};

/**
 * Returns the window object of the current application
 * @param string _app is the name of the application which requests the window object
 */
window.egw_appWindow = function(_app)
{
	var framework = egw_getFramework();
	if(framework && framework.egw_appWindow) return framework.egw_appWindow(_app);
	return window;
};

/**
 * Open _url in window of _app
 * @param _app
 * @param _url
 */
window.egw_appWindowOpen = function(_app, _url)
{
	if (typeof _url == "undefined") {
		_url = "about:blank";
	}
	window.location = _url;
};

/**
 * Returns the current egw application
 * @param string _name is only used for fallback, if an onlder version of jdots is used.
 */
window.egw_getApp = function(_name)
{
	return window.parent.framework.getApplicationByName(_name);
};

/**
 * Returns the name of the currently active application
 *
 * @deprecated use egw(window).app_name()
 */
window.egw_getAppName = function()
{
	if (typeof egw_appName == 'undefined')
	{
		return 'egroupware';
	}
	else
	{
		return egw_appName;
	}
};

/**
 * Refresh given application _targetapp display of entry _app _id, incl. outputting _msg
 *
 * Default implementation here only reloads window with it's current url with an added msg=_msg attached
 *
 * @param {string} _msg message (already translated) to show, eg. 'Entry deleted'
 * @param {string} _app application name
 * @param {(string|number)} _id id of entry to refresh or null
 * @param {string} _type either 'update', 'edit', 'delete', 'add' or null
 * - update: request just modified data from given rows.  Sorting is not considered,
 *		so if the sort field is changed, the row will not be moved.
 * - edit: rows changed, but sorting may be affected.  Requires full reload.
 * - delete: just delete the given rows clientside (no server interaction neccessary)
 * - add: requires full reload for proper sorting
 * @param {string} _targetapp which app's window should be refreshed, default current
 * @param {(string|RegExp)} _replace regular expression to replace in url
 * @param {string} _with
 * @param {string} _msg_type 'error', 'warning' or 'success' (default)
  * @deprecated use egw(window).refresh() instead
 */
window.egw_refresh = function(_msg, _app, _id, _type, _targetapp, _replace, _with, _msg_type)
{
	egw(window).refresh(_msg, _app, _id, _type, _targetapp, _replace, _with, _msg_type);
};

/**
 * Display an error or regular message
 *
 * @param {string} _msg message to show
 * @param {string} _type 'error', 'warning' or 'success' (default)
 * @deprecated use egw(window).message(_msg, _type)
 */
window.egw_message = function(_msg, _type)
{
	egw(window).message(_msg, _type);
};

/**
 * Update app-header and website-title
 *
 * @param {string} _header
 * @param {string} _app Application name, if not for the current app
   @deprecated use egw(window).app_header(_header, _app)
*/
window.egw_app_header = function(_header,_app)
{
	egw(window).app_header(_header, _app);
};

/**
 * View an EGroupware entry: opens a popup of correct size or redirects window.location to requested url
 *
 * Examples:
 * - egw_open(123,'infolog') or egw_open('infolog:123') opens popup to edit or view (if no edit rights) infolog entry 123
 * - egw_open('infolog:123','timesheet','add') opens popup to add new timesheet linked to infolog entry 123
 * - egw_open(123,'addressbook','view') opens addressbook view for entry 123 (showing linked infologs)
 * - egw_open('','addressbook','view_list',{ search: 'Becker' }) opens list of addresses containing 'Becker'
 *
 * @param string|int id either just the id or "app:id" if app==""
 * @param string app app-name or empty (app is part of id)
 * @param string type default "edit", possible "view", "view_list", "edit" (falls back to "view") and "add"
 * @param object|string extra extra url parameters to append as object or string
 * @param string target target of window to open
 * @deprecated use egw.open()
 */
window.egw_open = function(id, app, type, extra, target)
{
	window.egw.open(id, app, type, extra, target);
};

window.egw_getFramework = function()
{
	if (typeof window.framework != 'undefined')
	{
		return framework;
	}
	try {
		if (typeof window.parent.egw_getFramework != "undefined" && window != window.parent)
		{
			return window.parent.egw_getFramework();
		}
	}
	catch (e) {}

	return null;
};

/**
 * Register a custom method to refresh an application in an intelligent way
 *
 * This function will be called any time the application needs to be refreshed.
 * The default is to just reload, but with more detailed knowledge of the application
 * internals, it should be possible to only refresh what is needed.
 *
 * The refresh function signature is:
 * function (_msg, _app, _id, _type);
 * returns void
 * @see egw_refresh()
 *
 * @param appname String Name of the application
 * @param refresh_func function to call when refreshing
 */
window.register_app_refresh = function(appname, refresh_func)
{
	var framework = window.egw_getFramework();
	if(framework != null && typeof framework.register_app_refresh == "function")
	{
		framework.register_app_refresh(appname,refresh_func);
	}
	else
	{
		if(typeof window.app_refresh != "function")
		{
			// Define the app_refresh stuff here
			window.app_refresh = function(_msg, appname, _id, _type) {
				if(window.app_refresh.registry[appname])
				{
					window.app_refresh.registry[appname].call(this,_msg,appname,_id,_type);
				}
			};
			window.app_refresh.registry = {};
			window.app_refresh.registered = function(appname) {
				return (typeof window.app_refresh.registry[appname] == "function");
			};
		}
		window.app_refresh.registry[appname] = refresh_func;
	}
};


window.egw_set_checkbox_multiselect_enabled = function(_id, _enabled)
{
	//Retrieve the checkbox_multiselect base div
	var ms = document.getElementById('exec['+_id+']');
	if (ms !== null)
	{
		//Set the background color
		var label_color = "";
		if (_enabled)
		{
			ms.style.backgroundColor = "white";
			label_color = "black";
		}
		else
		{
			ms.style.backgroundColor = "#EEEEEE";
			label_color = "gray";
		}

		//Enable/Disable all children input elements
		for (var i = 0; i <ms.childNodes.length; i++)
		{
			if (ms.childNodes[i].nodeName == 'LABEL')
			{
				ms.childNodes[i].style.color = label_color;
				if ((ms.childNodes[i].childNodes.length >= 1) &&
					(ms.childNodes[i].childNodes[0].nodeName == 'INPUT'))
				{
					ms.childNodes[i].childNodes[0].disabled = !_enabled;
					ms.childNodes[i].childNodes[0].checked &= _enabled;
				}
			}
		}
	}
};

/**
 * Open a (centered) popup window with given size and url
 *
 * @param {string} _url
 * @param {string} _windowName or "_blank"
 * @param {number} _width
 * @param {number} _height
 * @param {type} _status "yes" or "no" to display status bar of popup
 * @param {string|boolean} _app app-name for framework to set correct opener or false for current app
 * @param {boolean} _returnID true: return window, false: return undefined
 * @returns {DOMWindow|undefined}
 * @deprecated use egw.openPopup(_url, _width, _height, _windowName, _app, _returnID, _status)
 */
window.egw_openWindowCentered2 = function(_url, _windowName, _width, _height, _status, _app, _returnID)
{
	return egw(window).openPopup(_url, _width, _height, _windowName, _app, _returnID, _status);
};

/**
 * @deprecated use egw.openPopup(_url, _width, _height, _windowName, _app, _returnID, _status)
 */
window.egw_openWindowCentered = function(_url, _windowName, _width, _height)
{
	return egw_openWindowCentered2(_url, _windowName, _width, _height, 'no', false, true);
};

// return the left position of the window
window.egw_getWindowLeft = function()
{
	// workaround for Fennec bug https://bugzilla.mozilla.org/show_bug.cgi?format=multiple&id=648250 window.(outerHeight|outerWidth|screenX|screenY) throw exception
	try {
		return window.screenX;
	}
	catch (e) {}

	return window.screenLeft;
};

// return the left position of the window
window.egw_getWindowTop = function()
{
	// workaround for Fennec bug https://bugzilla.mozilla.org/show_bug.cgi?format=multiple&id=648250 window.(outerHeight|outerWidth|screenX|screenY) throw exception
	try {
		return window.screenY;
	}
	catch (e) {}

	return window.screenTop-90;
};

// get the outerWidth of the browser window. For IE we simply return the innerWidth
window.egw_getWindowInnerWidth = function()
{
	return window.innerWidth;
};

// get the outerHeight of the browser window. For IE we simply return the innerHeight
window.egw_getWindowInnerHeight = function()
{
	return window.innerHeight;
};

// get the outerWidth of the browser window. For IE we simply return the innerWidth
window.egw_getWindowOuterWidth = function()
{
	// workaround for Fennec bug https://bugzilla.mozilla.org/show_bug.cgi?format=multiple&id=648250 window.(outerHeight|outerWidth|screenX|screenY) throw exception
	try {
		return window.outerWidth;
	}
	catch (e) {}

	return egw_getWindowInnerWidth();
};

// get the outerHeight of the browser window. For IE we simply return the innerHeight
window.egw_getWindowOuterHeight = function()
{
	// workaround for Fennec bug https://bugzilla.mozilla.org/show_bug.cgi?format=multiple&id=648250 window.(outerHeight|outerWidth|screenX|screenY) throw exception
	try {
		return window.outerHeight;
	}
	catch (e) {}

	return egw_getWindowInnerHeight();
};

// ie selectbox dropdown menu hack. as ie is not able to resize dropdown menus from selectboxes, we
// read the content of the dropdown menu and present it as popup resized for the user. if the user
// clicks/seleckts a value, the selection is posted back to the origial selectbox
window.dropdown_menu_hack = function(el)
{
	if(el.runtimeStyle)
	{
		if(typeof(enable_ie_dropdownmenuhack) !== 'undefined')
		{
			if (enable_ie_dropdownmenuhack==1){

			}
			else
				return;
		} else {
			return;
		}
		if(el.runtimeStyle.behavior.toLowerCase()=="none"){return;}
		el.runtimeStyle.behavior="none";

		if (el.multiple ==1) {return;}
		if (el.size > 1) {return;}

		var ie5 = (document.namespaces==null);
		el.ondblclick = function(e)
		{
			window.event.returnValue=false;
			return false;
		};

		if(window.createPopup==null)
		{
			var fid = "dropdown_menu_hack_" + Date.parse(new Date());

			window.createPopup = function()
			{
				if(window.createPopup.frameWindow==null)
				{
					el.insertAdjacentHTML("MyFrame","<iframe id='"+fid+"' name='"+fid+"' src='about:blank' frameborder='1' scrolling='no'></></iframe>");
					var f = document.frames[fid];
					f.document.open();
					f.document.write("<html><body></body></html>");
					f.document.close();
					f.fid = fid;


					var fwin = document.getElementById(fid);
					fwin.style.cssText="position:absolute;top:0;left:0;display:none;z-index:99999;";


					f.show = function(px,py,pw,ph,baseElement)
					{
						py = py + baseElement.getBoundingClientRect().top + Math.max( document.body.scrollTop, document.documentElement.scrollTop) ;
						px = px + baseElement.getBoundingClientRect().left + Math.max( document.body.scrollLeft, document.documentElement.scrollLeft) ;
						fwin.style.width = pw + "px";
						fwin.style.height = ph + "px";
						fwin.style.posLeft =px ;
						fwin.style.posTop = py ;
						fwin.style.display="block";
					};


					f.hide = function(e)
					{
						if(window.event && window.event.srcElement && window.event.srcElement.tagName && window.event.srcElement.tagName.toLowerCase()=="select"){return true;}
						fwin.style.display="none";
					};
					document.attachEvent("onclick",f.hide);
					document.attachEvent("onkeydown",f.hide);

				}
				return f;
			};
		}

		function showMenu()
		{

			function selectMenu(obj)
			{
				var o = document.createElement("option");
				o.value = obj.value;
				//alert("val"+o.value+', text:'+obj.innerHTML+'selected:'+obj.selectedIndex);
				o.text = obj.innerHTML;
				o.text = o.text.replace('<NOBR>','');
				o.text = o.text.replace('</NOBR>','');
				//if there is no value, you should not try to set the innerHTML, as it screws up the empty selection ...
				if (o.value != '') o.innerHTML = o.text;
				while(el.options.length>0){el.options[0].removeNode(true);}
				el.appendChild(o);
				el.title = o.innerHTML;
				el.contentIndex = obj.selectedIndex ;
				el.menu.hide();
				if(el.onchange)
				{
					el.onchange();
				}
			}


			el.menu.show(0 , el.offsetHeight , 10, 10, el);
			var mb = el.menu.document.body;

			mb.style.cssText ="border:solid 1px black;margin:0;padding:0;overflow-y:auto;overflow-x:auto;background:white;font:12px Tahoma, sans-serif;";
			var t = el.contentHTML;
			//alert("1"+t);
			t = t.replace(/<select/gi,'<div');
			//alert("2"+t);
			t = t.replace(/<option/gi,'<span');
			//alert("3"+t);
			t = t.replace(/<\/option/gi,'</span');
			//alert("4"+t);
			t = t.replace(/<\/select/gi,'</div');
			t = t.replace(/<optgroup label=\"([\w\s\wäöüßÄÖÜ]*[^>])*">/gi,'<span value="i-opt-group-lable-i">$1</span>');
			t = t.replace(/<\/optgroup>/gi,'<span value="">---</span>');
			mb.innerHTML = t;
			//mb.innerHTML = "<div><span value='dd:ff'>gfgfg</span></div>";

			el.select = mb.all.tags("div")[0];
			el.select.style.cssText="list-style:none;margin:0;padding:0;";
			mb.options = el.select.getElementsByTagName("span");

			for(var i=0;i<mb.options.length;i++)
			{
				//alert('Value:'+mb.options[i].value + ', Text:'+ mb.options[i].innerHTML);
				mb.options[i].selectedIndex = i;
				mb.options[i].style.cssText = "list-style:none;margin:0;padding:1px 2px;width/**/:100%;white-space:nowrap;";
				if (mb.options[i].value != 'i-opt-group-lable-i') mb.options[i].style.cssText = mb.options[i].style.cssText + "cursor:hand;cursor:pointer;";
				mb.options[i].title =mb.options[i].innerHTML;
				mb.options[i].innerHTML ="<nobr>" + mb.options[i].innerHTML + "</nobr>";
				if (mb.options[i].value == 'i-opt-group-lable-i') mb.options[i].innerHTML = "<b><i>"+mb.options[i].innerHTML+"</b></i>";
				if (mb.options[i].value != 'i-opt-group-lable-i') mb.options[i].onmouseover = function()
				{
					if( mb.options.selected )
					{mb.options.selected.style.background="white";mb.options.selected.style.color="black";}
					mb.options.selected = this;
					this.style.background="#333366";this.style.color="white";
				};
				mb.options[i].onmouseout = function(){this.style.background="white";this.style.color="black";};
				if (mb.options[i].value != 'i-opt-group-lable-i')
				{
					mb.options[i].onmousedown = function(){selectMenu(this); };
					mb.options[i].onkeydown = function(){selectMenu(this); };
				}
				if(i == el.contentIndex)
				{
					mb.options[i].style.background="#333366";
					mb.options[i].style.color="white";
					mb.options.selected = mb.options[i];
				}
			}
			var mw = Math.max( ( el.select.offsetWidth + 22 ), el.offsetWidth + 22 );
			mw = Math.max( mw, ( mb.scrollWidth+22) );
			var mh = mb.options.length * 15 + 8 ;
			var mx = (ie5)?-3:0;
			var docW = document.documentElement.offsetWidth ;
			var sideW = docW - el.getBoundingClientRect().left ;
			if (sideW < mw)
			{
				//alert(el.getBoundingClientRect().left+' Avail: '+docW+' Mx:'+mx+' My:'+my);
				// if it does not fit into the window on the right side, move it to the left
				mx = mx -mw + sideW-5;
			}
			var my = el.offsetHeight -2;
			my=my+5;
			var docH = document.documentElement.offsetHeight ;
			var bottomH = docH - el.getBoundingClientRect().bottom ;
			mh = Math.min(mh, Math.max(( docH - el.getBoundingClientRect().top - 50),100) );
			if(( bottomH < mh) )
			{
				mh = Math.max( (bottomH - 12),10);
				if( mh <100 )
				{
					my = -100 ;
				}
				mh = Math.max(mh,100);
			}
			self.focus();
			el.menu.show( mx , my , mw, mh , el);
			if(mb.options.selected)
			{
				mb.scrollTop = mb.options.selected.offsetTop;
			}
			window.onresize = function(){el.menu.hide();};
		}

		function switchMenu()
		{
			if(event.keyCode)
			{
				if(event.keyCode==40){ el.contentIndex++ ;}
				else if(event.keyCode==38){ el.contentIndex--; }
			}
			else if(event.wheelDelta )
			{
				if (event.wheelDelta >= 120)
					el.contentIndex++ ;
				else if (event.wheelDelta <= -120)
					el.contentIndex-- ;
			}
			else {return true;}
			if( el.contentIndex > (el.contentOptions.length-1) ){ el.contentIndex =0;}
			else if (el.contentIndex<0){el.contentIndex = el.contentOptions.length-1 ;}
			var o = document.createElement("option");
			o.value = el.contentOptions[el.contentIndex].value;
			o.innerHTML = el.contentOptions[el.contentIndex].text;
			while(el.options.length>0){el.options[0].removeNode(true);}
			el.appendChild(o);
			el.title = o.innerHTML;
		}
		if(dropdown_menu_hack.menu ==null)
		{
			dropdown_menu_hack.menu = window.createPopup();
			document.attachEvent("onkeydown",dropdown_menu_hack.menu.hide);
		}
		el.menu = dropdown_menu_hack.menu ;
		el.contentOptions = new Array();
		el.contentIndex = el.selectedIndex;
		el.contentHTML = el.outerHTML;

		for(var i=0;i<el.options.length;i++)
		{

			el.contentOptions [el.contentOptions.length] =
			{
				"value": el.options[i].value,"text": el.options[i].innerHTML
			};
			if(!el.options[i].selected){el.options[i].removeNode(true);i--;};
		}
		el.onkeydown = switchMenu;
		el.onclick = showMenu;
		el.onmousewheel= switchMenu;
	}
};

/**
 * Use frameworks (framed template) link handler to open a url
 *
 * @param _link
 * @param _app
 * @deprecated use egw(window).link_handler(_link, _app) instead
 */
window.egw_link_handler = function(_link, _app)
{
	egw(window).link_handler(_link, _app);
};

/**
 * Support functions for uiaccountselection class
 *
 * @ToDo: should be removed if uiaccountsel class is no longer in use
 */
window.addOption = function(id,label,value,do_onchange)
{
	let selectBox = document.getElementById(id);
	for (var i=0; i < selectBox.length; i++) {
	//		check existing entries if they're already there and only select them in that case
		if (selectBox.options[i].value == value) {
			selectBox.options[i].selected = true;
			break;
		}
	}
	if (i >= selectBox.length) {
		if (!do_onchange) {
			if (selectBox.length && selectBox.options[0].value=='') selectBox.options[0] = null;
			selectBox.multiple=true;
			selectBox.size=4;
		}
		selectBox.options[selectBox.length] = new Option(label,value,false,true);
	}
	if (selectBox.onchange && do_onchange) selectBox.onchange();
};

/**
 * @deprecated use egw.file_editor_prefered_mimes()
 */
window.egw_get_file_editor_prefered_mimes = function(_mime)
{
	return egw.file_editor_prefered_mimes(_mime);
};
/**
 * Install click handlers for popup and multiple triggers of uiaccountselection
 */
jQuery(function(){
	jQuery(document).on('click', '.uiaccountselection_trigger',function(){
		var selectBox = document.getElementById(this.id.replace(/(_multiple|_popup)$/, ''));
		if (selectBox)
		{
			var link = selectBox.getAttribute('data-popup-link');

			if (selectBox.multiple || this.id.match(/_popup$/))
			{
				window.open(link, 'uiaccountsel', 'width=600,height=420,toolbar=no,scrollbars=yes,resizable=yes');
			}
			else
			{
				selectBox.size = 4;
				selectBox.multiple = true;
				if (selectBox.options[0].value=='') selectBox.options[0] = null;

				if (!jQuery(selectBox).hasClass('groupmembers') && !jQuery(selectBox).hasClass('selectbox'))	// no popup!
				{
					this.src = egw.image('search');
					this.title = egw.lang('Search accounts');
				}
				else
				{
					this.style.display = 'none';
					selectBox.style.width = '100%';
				}
			}
		}
	});
	jQuery(document).on('change', 'select.uiaccountselection',function(e){
		if (this.value == 'popup')
		{
			var link = this.getAttribute('data-popup-link');
			window.open(link, 'uiaccountsel', 'width=600,height=420,toolbar=no,scrollbars=yes,resizable=yes');
			e.preventDefault();
		}
	});
});

// IE does not support ES6 therefore we need to use polyfill function
Number.isInteger = Number.isInteger || function(value) {
  return typeof value === 'number' &&
    isFinite(value) &&
    Math.floor(value) === value;
};

/**
 * Circular dependancy resolution file
 * Here we force the order of includes
 */
//# sourceMappingURL=egw.min.js.map
