"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.watchFilesMiddleware = void 0;
const dependency_graph_1 = require("dependency-graph");
const debounce_1 = __importDefault(require("debounce"));
const path_1 = __importDefault(require("path"));
const index_js_1 = require("../../index.js");
const IGNORED_404s = ['favicon.ico'];
const { PARAM_SESSION_ID } = index_js_1.constants;
function onRerunSessions(runSessions, sessions, sessionIds) {
    const sessionsToRerun = sessionIds
        ? sessionIds
            .map(id => {
            const session = sessions.get(id);
            if (!session && !sessions.getDebug(id)) {
                throw new Error(`Could not find session ${id}`);
            }
            return session;
        })
            .filter(s => s)
        : sessions.all();
    runSessions(sessionsToRerun);
}
function onRequest404(sessions, sessionId, url) {
    if (sessions.getDebug(sessionId)) {
        return;
    }
    const session = sessions.get(sessionId);
    if (!session) {
        throw new Error(`Could not find session ${sessionId}`);
    }
    const { request404s } = session;
    if (!request404s.includes(url) && !IGNORED_404s.some(i => url.endsWith(i))) {
        sessions.update(Object.assign(Object.assign({}, session), { request404s: [...request404s, url] }));
    }
}
function toFilePath(browserPath) {
    return browserPath.replace(new RegExp('/', 'g'), path_1.default.sep);
}
function watchFilesMiddleware({ rootDir, fileWatcher, sessions, runSessions, testFrameworkImport, }) {
    const fileGraph = new dependency_graph_1.DepGraph({ circular: true });
    const urlGraph = new dependency_graph_1.DepGraph({ circular: true });
    let pendingChangedFiles = new Set();
    function getSessionIdsForPath(path) {
        const ids = new Set();
        if (urlGraph.hasNode(path)) {
            for (const dependant of urlGraph.dependantsOf(path)) {
                if (dependant.startsWith('\0')) {
                    const url = new URL(dependant.substring(1));
                    const id = url.searchParams.get(PARAM_SESSION_ID);
                    if (!id) {
                        throw new Error('Missing session id parameter');
                    }
                    ids.add(id);
                }
            }
        }
        return ids;
    }
    function getSessionIdsForFile(file) {
        const ids = new Set();
        if (fileGraph.hasNode(file)) {
            for (const dependant of fileGraph.dependantsOf(file)) {
                if (dependant.startsWith('\0')) {
                    const url = new URL(dependant.substring(1));
                    const id = url.searchParams.get(PARAM_SESSION_ID);
                    if (!id) {
                        throw new Error('Missing session id parameter');
                    }
                    ids.add(id);
                }
            }
        }
        return ids;
    }
    function syncRerunSessions() {
        const sessionsToRerun = new Set();
        // search dependants of changed files for test HTML files, and reload only those
        for (const file of pendingChangedFiles) {
            for (const id of getSessionIdsForFile(file)) {
                sessionsToRerun.add(id);
            }
        }
        if (sessionsToRerun.size > 0) {
            // re run specified sessions
            onRerunSessions(runSessions, sessions, Array.from(sessionsToRerun));
        }
        else {
            // re run alll sessions
            onRerunSessions(runSessions, sessions);
        }
        pendingChangedFiles = new Set();
    }
    const rerunSessions = (0, debounce_1.default)(syncRerunSessions, 300);
    function onFileChanged(filePath) {
        pendingChangedFiles.add(filePath);
        rerunSessions();
    }
    fileWatcher.addListener('change', onFileChanged);
    fileWatcher.addListener('unlink', onFileChanged);
    function addDependencyMapping(ctx, testFrameworkImport) {
        if ((testFrameworkImport && ctx.path === testFrameworkImport) || ctx.path.endsWith('/')) {
            // the test framework or test runner HTML don't need to be tracked
            // we specifcally don't want to track the test framework, because it
            // requests all test files which would mess up the dependancy graph
            return;
        }
        // who is requesting this dependency
        let dependantUrl;
        if (ctx.URL.searchParams.has(PARAM_SESSION_ID)) {
            // if the requested file itself has a session id in the URL, use that as the dependant
            // this way when this file changes, we know session it belongs to
            dependantUrl = ctx.URL;
        }
        else if (ctx.headers.referer) {
            // most requests contain a referer, the file which requested this file
            dependantUrl = new URL(ctx.headers.referer);
        }
        else {
            // certain files like source maps are fetched without a referer, we skip those
            return;
        }
        const dependencyPath = ctx.path;
        const dependencyUrl = ctx.url;
        const dependantFilePath = dependantUrl.searchParams.has(PARAM_SESSION_ID)
            ? // the dependant is the test file itself,
                // we remember the full href and mark it with a null byte so that
                // later we can extract the session id
                `\0${dependantUrl.href}`
            : // the dependant is a "regular" file, we resolve it to the file path
                path_1.default.join(rootDir, toFilePath(dependantUrl.pathname));
        const dependencyFilePath = path_1.default.join(rootDir, toFilePath(dependencyPath));
        const dependantPath = dependantUrl.searchParams.has(PARAM_SESSION_ID)
            ? dependantFilePath
            : dependantUrl.pathname;
        fileGraph.addNode(dependantFilePath);
        fileGraph.addNode(dependencyFilePath);
        fileGraph.addDependency(dependantFilePath, dependencyFilePath);
        urlGraph.addNode(dependantPath);
        urlGraph.addNode(dependencyUrl);
        urlGraph.addDependency(dependantPath, dependencyUrl);
    }
    return async (ctx, next) => {
        addDependencyMapping(ctx, testFrameworkImport);
        await next();
        if (ctx.status === 404) {
            for (const sessionId of getSessionIdsForPath(ctx.url)) {
                onRequest404(sessions, sessionId, ctx.url.substring(1));
            }
        }
    };
}
exports.watchFilesMiddleware = watchFilesMiddleware;
//# sourceMappingURL=watchFilesMiddleware.js.map