// src/web-types-generator.ts
import fs2 from "fs";

// ../../tools/utilities/src/utilities.ts
var toKebabCase = (value) => value.replace(/([a-z0–9])([A-Z])/g, "$1-$2").toLowerCase();
function has(arr) {
  return Array.isArray(arr) && arr.length > 0;
}

// ../../tools/cem-utils/src/cem-utilities.ts
function getComponentDescription(component, descriptionSrc) {
  var _a;
  let description = ((_a = descriptionSrc ? component[descriptionSrc] : component.summary || component.description) == null ? void 0 : _a.replace(/\\n/g, "\n")) || "";
  if (component.deprecated) {
    const deprecation = typeof component.deprecated === "string" ? `@deprecated ${component.deprecated}` : "@deprecated";
    description = `${deprecation}

${description}`;
  }
  return description;
}
function getComponents(customElementsManifest, exclude) {
  var _a;
  return (((_a = customElementsManifest.modules) == null ? void 0 : _a.map(
    (mod) => {
      var _a2;
      return ((_a2 = mod == null ? void 0 : mod.declarations) == null ? void 0 : _a2.filter(
        (dec) => !(exclude == null ? void 0 : exclude.includes(dec.name)) && (dec.customElement || dec.tagName)
      )) || [];
    }
  )) || []).flat();
}
function getComponentMethods(component) {
  var _a;
  return (_a = component.members) == null ? void 0 : _a.filter(
    (member) => {
      var _a2;
      return member.kind === "method" && member.privacy !== "private" && ((_a2 = member.description) == null ? void 0 : _a2.length);
    }
  );
}

// ../../tools/cem-utils/src/description-templates.ts
function getComponentDetailsTemplate(component, options, isComment = false) {
  var _a, _b, _c, _d, _e;
  const slots = getSlotsTemplate(
    component == null ? void 0 : component.slots,
    options == null ? void 0 : options.hideSlotDocs,
    (_a = options == null ? void 0 : options.labels) == null ? void 0 : _a.slots
  );
  const events = getEventsTemplate(
    component == null ? void 0 : component.events,
    options == null ? void 0 : options.hideEventDocs,
    (_b = options == null ? void 0 : options.labels) == null ? void 0 : _b.events
  );
  const cssProps = getCssPropsTemplate(
    component == null ? void 0 : component.cssProperties,
    options == null ? void 0 : options.hideCssPropertiesDocs,
    (_c = options == null ? void 0 : options.labels) == null ? void 0 : _c.cssProperties
  );
  const parts = getPartsTemplate(
    component == null ? void 0 : component.cssParts,
    options == null ? void 0 : options.hideCssPartsDocs,
    (_d = options == null ? void 0 : options.labels) == null ? void 0 : _d.cssParts
  );
  const methods = getMethodsTemplate(
    getComponentMethods(component),
    options == null ? void 0 : options.hideMethodDocs,
    (_e = options == null ? void 0 : options.labels) == null ? void 0 : _e.methods
  );
  let description = getComponentDescription(component, options == null ? void 0 : options.descriptionSrc) + "\n---\n" + events + methods + slots + cssProps + parts;
  if (isComment) {
    description = description.split("\n").map((x) => ` * ${x}`).join("\n");
  }
  return description;
}
function getSlotsTemplate(slots, hide, label = "Slots") {
  return has(slots) && !hide ? `

### **${label}:**
 ${getSlotDocs(slots)}` : "";
}
function getEventsTemplate(events, hide, label = "Events") {
  return has(events) && !hide ? `

### **${label}:**
 ${getEventDocs(events)}` : "";
}
function getCssPropsTemplate(cssProperties, hide, label = "CSS Properties") {
  return has(cssProperties) && !hide ? `

### **${label}:**
 ${getCssPropertyDocs(cssProperties)}` : "";
}
function getPartsTemplate(cssParts, hide, label = "CSS Parts") {
  return has(cssParts) && !hide ? `

### **${label}:**
 ${getCssPartsDocs(cssParts)}` : "";
}
function getMethodsTemplate(methods, hide, label = "Methods") {
  return has(methods) && !hide ? `

### **${label}:**
 ${getMethodDocs(methods)}` : "";
}
function getEventDocs(events) {
  return events == null ? void 0 : events.map((event) => `- **${event.name}** - ${event.description}`).join("\n");
}
function getCssPropertyDocs(properties) {
  return properties == null ? void 0 : properties.map(
    (prop) => `- **${prop.name}** - ${prop.description} _(default: ${prop.default})_`
  ).join("\n");
}
function getCssPartsDocs(parts) {
  return parts == null ? void 0 : parts.map((part) => `- **${part.name}** - ${part.description}`).join("\n");
}
function getSlotDocs(slots) {
  return slots == null ? void 0 : slots.map(
    (slot) => `- ${slot.name ? `**${slot.name}**` : "_default_"} - ${slot.description}`
  ).join("\n");
}
function getMethodDocs(methods) {
  return methods == null ? void 0 : methods.map((method) => {
    var _a;
    if (method.privacy === "private" || method.privacy === "protected") {
      return;
    }
    return `- **${method.name}${getParameters(method.parameters)}${method.return ? `: _${(_a = method.return.type) == null ? void 0 : _a.text}_` : ""}** - ${getMemberDescription(method.description, method.deprecated)}`;
  }).join("\n");
}
function getParameters(parameters) {
  return parameters ? "(" + parameters.map(
    (x) => {
      var _a, _b;
      return `${x.name + (((_a = x == null ? void 0 : x.type) == null ? void 0 : _a.text) ? `: _${(_b = x == null ? void 0 : x.type) == null ? void 0 : _b.text}_` : "")}`;
    }
  ).join(", ") + ")" : "()";
}
function getMemberDescription(description, deprecated) {
  if (!deprecated) {
    return description || "";
  }
  const desc = description ? ` - ${description}` : "";
  return typeof deprecated === "string" ? `@deprecated ${deprecated}${desc}` : `@deprecated${desc}`;
}

// src/cem-utilities.ts
function getCssPropertyList(components) {
  return ((components == null ? void 0 : components.map((component) => {
    var _a;
    return ((_a = component.cssProperties) == null ? void 0 : _a.map((prop) => {
      return {
        name: prop.name,
        description: prop.description
      };
    })) || [];
  })) || []).flat();
}
function getCssPartList(components) {
  return ((components == null ? void 0 : components.map((component) => {
    var _a;
    return ((_a = component.cssParts) == null ? void 0 : _a.map((prop) => {
      return {
        name: `part(${prop.name})`,
        description: prop.description
      };
    })) || [];
  })) || []).flat();
}
function getComponentAttributes(component) {
  var _a;
  const attributes = [];
  (_a = component == null ? void 0 : component.attributes) == null ? void 0 : _a.forEach((attr) => {
    var _a2;
    const existingAttr = attributes.find(
      (x) => x.name === attr.name || x.name === attr.fieldName
    );
    if (existingAttr) {
      return;
    }
    attributes.push({
      name: attr.name || attr.fieldName,
      description: attr.description,
      value: {
        type: (_a2 = attr.type) == null ? void 0 : _a2.text
      }
    });
  });
  return attributes;
}

// ../../tools/integrations/src/files.ts
import fs from "fs";
import path from "path";
import prettier from "prettier";
function createOutDir(outDir) {
  if (outDir !== "./" && !fs.existsSync(outDir)) {
    fs.mkdirSync(outDir, { recursive: true });
  }
}
function saveFile(outDir, fileName, contents, parser = "json", printWidth = 80) {
  const outputPath = path.join(outDir, fileName);
  fs.writeFileSync(
    outputPath,
    prettier.format(contents, { parser, printWidth })
  );
  return outputPath;
}

// ../../tools/integrations/src/logs.ts
function logRed(message) {
  console.log("\x1B[31m%s\x1B[0m", message);
}
function logGreen(message) {
  console.log("\x1B[32m%s\x1B[0m", message);
}
function logBlue(message) {
  console.log("\x1B[34m%s\x1B[0m", message);
}

// ../../tools/configurations/src/config.ts
var baseConfig = {
  outdir: "./",
  exclude: [],
  descriptionSrc: void 0,
  hideSlotDocs: false,
  hideEventDocs: false,
  hideCssPropertiesDocs: false,
  hideCssPartsDocs: false,
  hideMethodDocs: false,
  labels: {
    slots: "Slots",
    events: "Events",
    cssProperties: "CSS Properties",
    cssParts: "CSS Parts",
    methods: "Methods"
  }
};
function updateConfig(params) {
  const config = { ...baseConfig, ...params };
  config.labels = { ...baseConfig.labels, ...params == null ? void 0 : params.labels };
  return config;
}

// src/web-types-generator.ts
var packageJson = JSON.parse(fs2.readFileSync("package.json", "utf8"));
function getTagList(components, options, referenceTemplate) {
  return components.map((component) => {
    const reference = referenceTemplate ? referenceTemplate(component.name, component.tagName) : void 0;
    return {
      name: `${options.prefix}${component.tagName || toKebabCase(component.name)}${options.suffix}`,
      description: getComponentDetailsTemplate(component, options),
      ["doc-url"]: (reference == null ? void 0 : reference.url) || "",
      attributes: getComponentAttributes(component),
      js: getJsProperties(component, options.typesSrc)
    };
  });
}
function getJsProperties(component, typesSrc) {
  return {
    properties: getWebTypeProperties(component, typesSrc),
    events: getWebTypeEvents(component)
  };
}
function getWebTypeProperties(component, typesSrc = "types") {
  var _a;
  return ((_a = component.attributes || component.members) == null ? void 0 : _a.map(
    (attr) => {
      var _a2, _b;
      return {
        name: attr.name,
        description: attr.description,
        value: {
          type: ((_a2 = attr[`${typesSrc}`]) == null ? void 0 : _a2.text) || ((_b = attr.type) == null ? void 0 : _b.text)
        }
      };
    }
  )) || [];
}
function getWebTypeEvents(component) {
  var _a;
  return ((_a = component.events) == null ? void 0 : _a.map((event) => {
    return {
      name: event.name,
      description: event.description
    };
  })) || [];
}
function generateJetBrainsWebTypes(customElementsManifest, options) {
  options = getOptions(options);
  const components = getComponents(
    customElementsManifest,
    options.exclude
  ).filter((x) => x.tagName);
  if (!components.length) {
    logRed("No components found in custom-elements.json");
    return;
  }
  const elements = options.webTypesFileName ? getTagList(components, options) : [];
  const cssProperties = getCssPropertyList(components);
  const cssParts = getCssPartList(components);
  const outputPath = saveWebTypeFile(
    elements,
    cssProperties,
    cssParts,
    options
  );
  logBlue(`[jet-brains-web-type-generator] - Generated "${outputPath}".`);
}
function getOptions(options) {
  options = updateConfig(options);
  options.webTypesFileName = options.webTypesFileName === void 0 ? "web-types.json" : options.webTypesFileName;
  options.prefix = options.prefix === void 0 ? "" : options.prefix;
  options.suffix = options.suffix === void 0 ? "" : options.suffix;
  return options;
}
function saveWebTypeFile(tags, cssProperties, parts, options) {
  createOutDir(options.outdir);
  if (options.webTypesFileName) {
    savePackageJson(packageJson, options);
    return saveFile(
      options.outdir,
      options.webTypesFileName,
      getWebTypesFileContents(tags, cssProperties, parts, options)
    );
  }
  return "";
}
function savePackageJson(packageJson2, options) {
  var _a;
  packageJson2["web-types"] = (!((_a = options.outdir) == null ? void 0 : _a.endsWith("/")) ? options.outdir + "/" : options.outdir || "") + options.webTypesFileName;
  saveFile("./", "package.json", JSON.stringify(packageJson2, null, 2));
}
function getWebTypesFileContents(tags, cssProperties, parts, options) {
  return `{
    "$schema": "https://json.schemastore.org/web-types",
    "name": "${packageJson.name}",
    "version": "${packageJson.version}",
    "description-markup": "markdown",
    "contributions": {
      ${options.excludeHtml ? "" : `"html": {
        "elements": ${JSON.stringify(tags)}
      },`}
      ${options.excludeCss ? "" : `"css": {
        "properties": ${JSON.stringify(cssProperties)},
        "pseudo-elements": ${JSON.stringify(parts)}
      }`}
    }
  }`;
}

// src/cem-analyzer-plugin.ts
function customElementJetBrainsPlugin(options = {}) {
  return {
    name: "custom-element-jet-brains-integration",
    packageLinkPhase({ customElementsManifest }) {
      console.log(
        "[jet-brains-web-type-generator] - Generating config files..."
      );
      generateJetBrainsWebTypes(customElementsManifest, options);
      logGreen("[jet-brains-web-type-generator] - File generation complete.");
    }
  };
}
export {
  customElementJetBrainsPlugin,
  generateJetBrainsWebTypes
};
