const fs = require("fs");
const TOML = require("@iarna/toml");
const debug = require("debug")("Eleventy:Serverless");

class NetlifyRedirects {
  constructor(name) {
    this.name = name;
  }

  // keep non-generated redirects or those generated by a different serverless function
  filterRedirects(redirects = [], newRedirects = []) {
    let newRedirectNameLookup = {};
    for (let entry of newRedirects) {
      if (entry._generated_by_eleventy_serverless) {
        newRedirectNameLookup[entry._generated_by_eleventy_serverless] = true;
      }
    }

    return redirects.filter((entry) => {
      // keep in place if not eleventy serverless redirect
      if (!entry._generated_by_eleventy_serverless) {
        return true;

        // keep in place eleventy serverless (but for another plugin)
      } else if (entry._generated_by_eleventy_serverless !== this.name) {
        return true;
      }

      // keep in place if still in use and not stale!
      if (newRedirectNameLookup[entry._generated_by_eleventy_serverless]) {
        return true;
      }

      return false;
    });
  }

  getResults(newRedirects, config = {}) {
    let redirects = this.filterRedirects(config.redirects, newRedirects);

    // Sort for stable order
    newRedirects.sort((a, b) => {
      if (a.from < b.from) {
        return -1;
      } else if (a.from > b.from) {
        return 1;
      }
      return 0;
    });

    for (let r of newRedirects) {
      let found = false;
      for (let entry of redirects) {
        if (r.from === entry.from && r.to === entry.to) {
          found = true;
        }
      }

      if (!found) {
        redirects.unshift(r);
      }
    }

    if (redirects.length) {
      config.redirects = redirects;
    } else {
      delete config.redirects;
    }

    return config;
  }

  getNewRedirects(outputMap, target) {
    if (!target) {
      throw new Error(
        `Missing redirect target in Eleventy Serverless Bundler Plugin. Received ${target}`
      );
    }

    let newRedirects = [];
    for (let url in outputMap) {
      newRedirects.push({
        from: url,
        to: `${target}${this.name}`,
        status: 200,
        force: true,
        _generated_by_eleventy_serverless: this.name,
      });
    }

    return newRedirects;
  }

  // parse existing netlify.toml
  getExistingConfig(filename = "./netlify.toml") {
    if (fs.existsSync(filename)) {
      return TOML.parse(fs.readFileSync(filename));
    }

    return {};
  }

  writeFile(outputMap, target) {
    let newRedirects = this.getNewRedirects(outputMap, target);

    let configFilename = "./netlify.toml";
    let existingCfg = this.getExistingConfig(configFilename);
    let results = this.getResults(newRedirects, existingCfg);

    fs.writeFileSync(configFilename, TOML.stringify(results));

    debug(
      `Eleventy Serverless (${this.name}), writing (×${newRedirects.length}): ${configFilename}`
    );
  }
}

module.exports = NetlifyRedirects;
