"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createTestSessions = void 0;
const nanoid_1 = require("nanoid");
const path_1 = __importDefault(require("path"));
const TestSessionStatus_js_1 = require("../test-session/TestSessionStatus.js");
const collectTestFiles_js_1 = require("./collectTestFiles.js");
function createTestSessions(config, groupConfigs) {
    var _a;
    const groups = [];
    if (config.files) {
        groups.push({
            name: 'default',
            files: config.files,
            browsers: config.browsers,
        });
    }
    for (const groupConfig of groupConfigs) {
        // merge group with config defaults
        const mergedGroupConfig = {
            name: groupConfig.name,
            configFilePath: groupConfig.configFilePath,
            testRunnerHtml: config.testRunnerHtml,
            browsers: config.browsers,
            files: (_a = config.files) !== null && _a !== void 0 ? _a : [],
        };
        if (typeof mergedGroupConfig.name !== 'string') {
            throw new Error('Group in config is missing a name.');
        }
        if (groupConfig.browsers != null) {
            mergedGroupConfig.browsers = groupConfig.browsers;
        }
        if (groupConfig.files != null) {
            mergedGroupConfig.files = groupConfig.files;
        }
        if (groupConfig.testRunnerHtml != null) {
            mergedGroupConfig.testRunnerHtml = groupConfig.testRunnerHtml;
        }
        groups.push(mergedGroupConfig);
    }
    const sessionGroups = [];
    const testSessions = [];
    const testFiles = new Set();
    const browsers = new Set();
    for (const group of groups) {
        const baseDir = group.configFilePath ? path_1.default.dirname(group.configFilePath) : process.cwd();
        const testFilesForGroup = (0, collectTestFiles_js_1.collectTestFiles)(group.files, baseDir)
            // Normalize file path because glob returns windows paths with forward slashes:
            // C:/foo/bar -> C:\foo\bar
            .map(testFile => path_1.default.normalize(testFile));
        if (testFilesForGroup.length === 0) {
            throw new Error(`Could not find any test files with pattern(s): ${group.files}`);
        }
        for (const file of testFilesForGroup) {
            testFiles.add(file);
        }
        const sessionGroup = {
            name: group.name,
            browsers: group.browsers,
            testFiles: testFilesForGroup,
            testRunnerHtml: group.testRunnerHtml,
            sessionIds: [],
        };
        for (const browser of group.browsers) {
            browsers.add(browser);
        }
        for (const testFile of testFilesForGroup) {
            for (const browser of group.browsers) {
                const session = {
                    id: (0, nanoid_1.nanoid)(),
                    group: sessionGroup,
                    debug: false,
                    testRun: -1,
                    browser,
                    status: TestSessionStatus_js_1.SESSION_STATUS.SCHEDULED,
                    testFile,
                    errors: [],
                    logs: [],
                    request404s: [],
                };
                testSessions.push(session);
                sessionGroup.sessionIds.push(session.id);
            }
        }
    }
    if (testFiles.size === 0 || testSessions.length === 0) {
        throw new Error('Did not find any tests to run.');
    }
    return {
        sessionGroups,
        testSessions,
        testFiles: Array.from(testFiles),
        browsers: Array.from(browsers),
    };
}
exports.createTestSessions = createTestSessions;
//# sourceMappingURL=createSessionGroups.js.map