<?php defined('FLATBOARD') or die('Flatboard Community.');

/*
 * Nom du projet : Flatboard
 * URL du projet : https://flatboard.org
 * Auteur : Frédéric Kaplon et contributeurs
 * Tout le code de Flatboard est publié sous la licence MIT.
 *
 * -- CODE : -----------------------------------------------------------------------
 *
 * $css = array(
 *       'http://example/files/sample_css_file_1.css',
 *       'http://example/files/sample_css_file_2.css',
 *       );
 *       
 * echo Asset::Stylesheet($css, 'minified_files/', md5("my_mini_file") . ".css");
 *
 * --------------------------------------------------------------------------------
 */

class Asset {

    /**
     * Constructeur protégé car il s'agit d'une classe statique.
     *
     * @access protected
     */
    protected function __construct()
    {
        // Rien ici
    }

    /**
     * Générer un lien vers une feuille de style CSS minifiée.
     *
     * @param array $array_files Liste des fichiers CSS à combiner
     * @param string $destination_dir Répertoire de destination pour le fichier minifié
     * @param string $dest_file_name Nom du fichier minifié
     * @return string Lien vers la feuille de style minifiée
     */
    public static function Stylesheet($array_files, $destination_dir, $dest_file_name = 'styles.min.css')
    {
        global $config;
        $content = '';

        $cache_dir = THEME_CONFIG_PATH . 'cache' . DS;
        $index_file = $cache_dir . 'index.html';
        $css_cache_file = $cache_dir . $dest_file_name;

        // Créer le dossier de cache
        if (!is_dir($cache_dir) && (!@mkdir($cache_dir) || !@chmod($cache_dir, 0777))) {
            return false; // Retourner false si le dossier ne peut pas être créé
        }

        // Créer index.html
        if (!file_exists($index_file)) {
            file_put_contents($index_file, ''); // Utiliser file_put_contents pour simplifier
        }

        // Continuer uniquement si le fichier de cache n'existe pas
        if (!file_exists($css_cache_file)) {
            foreach ($array_files as $file) {
                // Vérifier si le fichier est vide
                if (empty($file)) {
                    continue;
                }
                // Ajouter le contenu compressé du fichier CSS
                $content .= Asset::compressCSS($file);
            }

            // Écrire le contenu dans le fichier de cache
            if (($fd = fopen($css_cache_file, 'w')) !== false) {
                fwrite($fd, $content);
                fclose($fd);
            }

            return '<link rel="stylesheet" href="' . $css_cache_file . '?ver=' . VERSION . '" />' . PHP_EOL; // Retourner le lien vers le fichier combiné
        } else {
            // Utiliser le fichier de cache existant
            return '<link rel="stylesheet" href="' . $css_cache_file . '?ver=' . date("Y.m.d.H:i:s", filemtime($css_cache_file)) . '" />' . PHP_EOL; // Retourner le lien vers le fichier de cache
        }
    }

    /**
     * Compresser le contenu d'un fichier CSS.
     *
     * @param string $file Chemin du fichier CSS
     * @return string Contenu CSS compressé
     */
    public static function compressCSS($file)
    {
        $filedata = @file_get_contents($file);
        if ($filedata === false) {
            return ''; // Retourner une chaîne vide si le fichier ne peut pas être lu
        }

        // Supprimer les espaces et les commentaires
        $filedata = str_replace(array("\r\n", "\r", "\n", "\t", '  ', '    ', '    '), '', $filedata);
        $filedata = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $filedata);
        $filedata = str_replace(['{ ', ' }', '; ', ', ', ' {', '} ', ': ', ' ,', ' ;'], ['{', '}', ';', ',', '{', '}', ':', ',', ';'], $filedata);

        return $filedata;
    }

    /**
     * Mettre à jour le fichier CSS minifié si nécessaire.
     *
     * @return string Contenu du fichier CSS minifié
     */
    public static function updateCSS()
    {
        global $css_cache_file, $css_files, $css_path;

        if (file_exists($css_cache_file)) {
            $cache_time = filemtime($css_cache_file);
            foreach ($css_files as $file) {
                if (file_exists($css_path . $file)) {
                    $time = filemtime($css_path . $file);
                    if ($time > $cache_time) {
                        return Asset::joinCSSFiles(); // Rejoindre les fichiers CSS si l'un d'eux a été modifié
                    }
                }
            }
        } else {
            return Asset::joinCSSFiles(); // Rejoindre les fichiers CSS si le fichier de cache n'existe pas
        }

        return file_get_contents($css_cache_file); // Retourner le contenu du fichier de cache
    }

    /**
     * Joindre et compresser tous les fichiers CSS.
     *
     * @return string Contenu des fichiers CSS compressés
     */
    public static function joinCSSFiles()
    {
        global $css_cache_file, $css_files, $css_path;
        $data = '';

        foreach ($css_files as $file) {
            if (file_exists($css_path . $file)) {
                $data .= Asset::compressCSS($css_path . $file); // Ajouter le contenu compressé de chaque fichier
            }
        }

        file_put_contents($css_cache_file, $data); // Écrire le contenu compressé dans le fichier de cache
        return $data; // Retourner le contenu compressé
    }
}