<?php defined('FLATBOARD') or die('Flatboard Community.');

/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
*/

class HTMLForm
{
    /**
     * Constructeur protégé car il s'agit d'une classe statique.
     *
     * @access  protected
     */
    protected function __construct()
    {
        // Rien ici
    }

    /**
     * Méthode qui convertit du HTML en texte brut
     * HTMLForm::clean('test');
     *
     * @param string $text String
     * @return string
     */   
    public static function clean($text)
    {
        // Assurez-vous que CHARSET est défini, par exemple, define('CHARSET', 'UTF-8');
        if (!defined('CHARSET')) {	
            throw new Exception('CHARSET is not defined.');	
        }		
        // Trim l'entrée et convertit les caractères spéciaux en entités HTML
        return htmlspecialchars(trim($text ?? ''), ENT_QUOTES, CHARSET);	
    } 

	/**
	 * Méthode qui vérifie si une couleur est valide (hexadécimale ou nom de couleur)
	 *
	 * @param string $color La couleur à vérifier
	 * @return bool
	 */
	public static function isValidColor($color)
	{
	    // Vérifie si la couleur est au format hexadécimal
	    if (preg_match('/^#([0-9A-Fa-f]{3}|[0-9A-Fa-f]{6})$/', $color)) {
	        return true;
	    }
	
	    // Liste de toutes les couleurs CSS
	    $validColors = [
	        'aqua', 'black', 'blue', 'fuchsia', 'gray', 'green', 'lime', 'maroon', 
	        'navy', 'olive', 'purple', 'red', 'silver', 'teal', 'white', 'yellow',
	        'aliceblue', 'antiquewhite', 'aquamarine', 'azure', 'beige', 
	        'bisque', 'blanchedalmond', 'blueviolet', 'brown', 
	        'burlywood', 'cadetblue', 'chartreuse', 'chocolate', 'coral', 
	        'cornflowerblue', 'cornsilk', 'crimson', 'cyan', 'darkblue', 
	        'darkcyan', 'darkgoldenrod', 'darkgray', 'darkgreen', 'darkkhaki', 
	        'darkmagenta', 'darkolivegreen', 'darkorange', 'darkorchid', 
	        'darkred', 'darksalmon', 'darkseagreen', 'darkslateblue', 
	        'darkslategray', 'darkturquoise', 'darkviolet', 'deeppink', 
	        'deepskyblue', 'dimgray', 'dodgerblue', 'firebrick', 'floralwhite', 
	        'forestgreen', 'gainsboro', 'ghostwhite', 'gold', 
	        'goldenrod', 'greenyellow', 'honeydew', 'hotpink', 'indianred', 
	        'indigo', 'ivory', 'khaki', 'lavender', 'lavenderblush', 
	        'lawngreen', 'lemonchiffon', 'lightblue', 'lightcoral', 
	        'lightcyan', 'lightgoldenrodyellow', 'lightgray', 'lightgreen', 
	        'lightpink', 'lightsalmon', 'lightseagreen', 'lightskyblue', 
	        'lightslategray', 'lightsteelblue', 'lightyellow', 'limegreen', 
	        'linen', 'magenta', 'mediumaquamarine', 'mediumblue', 
	        'mediumorchid', 'mediumpurple', 'mediumseagreen', 
	        'mediumslateblue', 'mediumspringgreen', 'mediumturquoise', 
	        'mediumvioletred', 'midnightblue', 'mintcream', 'mistyrose', 
	        'moccasin', 'navajowhite', 'oldlace', 'olivedrab', 'orange', 
	        'orangered', 'orchid', 'palegoldenrod', 'palegreen', 
	        'paleturquoise', 'palevioletred', 'papayawhip', 
	        'peachpuff', 'peru', 'pink', 'plum', 'powderblue', 
	        'rosybrown', 'royalblue', 'saddlebrown', 'salmon', 
	        'sandybrown', 'seagreen', 'seashell', 'sienna', 'skyblue', 'slateblue', 'slategray', 
	        'snow', 'springgreen', 'steelblue', 'tan', 'teal', 
	        'thistle', 'tomato', 'turquoise', 'violet', 'wheat', 
	        'whitesmoke', 'yellow', 'yellowgreen'
	    ];
	
	    // Vérifie si la couleur est dans la liste des couleurs valides
	    return in_array(strtolower($color), $validColors);
	}

	/**
	 * Normalise les sauts de ligne dans le texte donné.
	 * Cette méthode remplace plusieurs sauts de ligne consécutifs par deux sauts de ligne
	 * et convertit toutes les représentations de saut de ligne en un format unique.
	 *
	 * @param string $text Le texte d'entrée
	 * @return string Le texte normalisé
	 */
	public static function transNL($text)
	{
	    // Convertir toutes les représentations de saut de ligne en "\n"
	    $normalizedText = str_replace(array("\r\n", "\r"), "\n", $text);
	    
	    // Remplacer trois ou plusieurs sauts de ligne consécutifs par deux sauts de ligne
	    return preg_replace('/\n{3,}/', "\n\n", $normalizedText);
	}
	
	/**
	 * Génère un tripcode à partir d'un nom et d'un identifiant.
	 * HTMLForm::trip('login@password');
	 *
	 * @param string $name Le nom à traiter
	 * @param string $id L'identifiant à utiliser
	 * @return string Le tripcode généré
	 */
	public static function trip($name, $id)
	{
	    global $config;
	
	    // Si le nom est vide, retourner les 8 derniers caractères de l'identifiant
	    if (empty($name)) {
	        return is_string($id) ? substr($id, -8) : ''; 
	    }
	
	    // Définir le séparateur
	    $sign = '@';
	    
	    // Diviser le nom en deux parties
	    $parts = explode($sign, $name, 2);
	    $first = trim($parts[0]); // Utiliser trim pour enlever les espaces
	    $last = isset($parts[1]) ? trim($parts[1]) : ''; 
	
	    // Utiliser une valeur par défaut pour le sel
	    $fallback = isset($config['salt']) ? $config['salt'] : KEY;
	
	    // Générer le sel
	    $salt = sha1($fallback . md5($last));
	
	    // Nettoyer le premier segment pour créer le tripcode
	    $trip = str_replace(array('http://', 'https://'), '', $first);
	    $trip = preg_replace('/[^A-Za-z0-9\-]/', '', $trip); // Utiliser $trip au lieu de $first
	
	    // Retourner le tripcode
	    return $trip . (!empty($last) ? $sign . substr($salt, -8) : '');
	}

	/**
	 * Gère l'affichage des messages d'erreur.
	 *
	 * @param string $eid L'identifiant de l'erreur
	 * @param string $msg Le message d'erreur à afficher
	 * @return string Le message d'erreur formaté ou une chaîne vide
	 */
	public static function err($eid, $msg)
	{
	    // Vérifier si l'identifiant d'erreur est défini dans la session
	    if (isset($_SESSION[$eid])) {
	        // Supprimer l'identifiant d'erreur de la session
	        unset($_SESSION[$eid]);
	        
	        // Retourner le message d'erreur formaté
	        return '&nbsp;<span style="color:red; font-weight:500;">' . $msg . '</span>';
	    }
	    
	    // Retourner une chaîne vide si l'identifiant d'erreur n'est pas présent
	    return '';
	}	

	/**
	 * Génère un champ de mot de passe HTML.
	 *
	 * @param string $name Le nom du champ
	 * @param string $default La valeur par défaut du champ
	 * @param string $class La classe CSS à appliquer
	 * @param string $placeholder Le texte d'espace réservé
	 * @param string $desc La description à afficher sous le champ
	 * @return string Le code HTML du champ de mot de passe
	 */
	public static function password($name, $default = '', $class = '', $placeholder = '', $desc = '')
	{
	    global $lang;
	
	    // Récupérer la valeur du champ à partir de la méthode POST ou utiliser la valeur par défaut
	    $value = Util::isPOST($name) ? HTMLForm::clean($_POST[$name]) : $default;
	
	    // Définir la classe CSS
	    $classAttr = !empty($class) ? ' class="' . htmlspecialchars($class) . '"' : ' class="form-control"';
	
	    // Définir l'attribut placeholder
	    $placeholderAttr = !empty($placeholder) ? ' placeholder="' . htmlspecialchars($lang[$placeholder]) . '"' : '';
	
	    // Définir la description
	    $descHtml = !empty($desc) ? '<small class="form-text text-muted">' . htmlspecialchars($lang[$desc]) . '</small>' : '';
	
	    // Générer le code HTML pour le champ de mot de passe
	    return '<div class="form-group pass_show">
	                <label for="' . htmlspecialchars($name) . '">' . $lang[$name] . '
	                    ' . HTMLForm::err($name . 'ErrNotMatch', $lang['errNotMatch']) . HTMLForm::err('bad_user_syntax', $lang['bad_user_syntax']) . '
	                </label>
	                <input type="password" name="' . htmlspecialchars($name) . '" value="' . htmlspecialchars($value) . '"' . $classAttr . $placeholderAttr . ' required autofocus>
	                ' . $descHtml . '
	            </div>';
	}		

	/**
	 * Méthode qui affiche une zone de saisie.
	 *
	 * @param string $name      Nom de la zone de saisie
	 * @param string $default   Valeur contenue dans la zone de saisie
	 * @param string $type      Type du champ (text, password, hidden)
	 * @param string $class     Classe CSS à utiliser pour formater l'affichage
	 * @param string $placeholder Valeur du placeholder du champ (HTML5)
	 * @param string $desc      Valeur de la description
	 * @param bool $disabled    Désactivation d'un champ
	 * @return string          Code HTML de la zone de saisie
	 */
	public static function text($name, $default = '', $type = 'text', $class = '', $placeholder = '', $desc = '', $disabled = false)
	{
	    global $lang;
	
	    // Récupérer la valeur du champ à partir de la méthode POST ou utiliser la valeur par défaut
	    $value = Util::isPOST($name) ? HTMLForm::clean($_POST[$name]) : $default;
	
	    // Définir la classe CSS
	    $classAttr = !empty($class) ? ' class="form-control ' . htmlspecialchars($class) . '"' : ' class="form-control"';
	
	    // Définir l'attribut placeholder
	    $placeholderAttr = !empty($placeholder) ? ' placeholder="' . $lang[$placeholder] . '"' : '';
	
	    // Définir l'attribut disabled
	    $disabledAttr = $disabled ? ' disabled' : '';
	
	    // Définir la description
	    $descHtml = !empty($desc) ? '<small class="form-text text-muted">' . $lang[$desc] . '</small>' : '';
	
	    // Générer le code HTML pour la zone de saisie
	    return '<div class="form-group">
	                <label for="' . htmlspecialchars($name) . '">' . $lang[$name] . '
	                    ' . HTMLForm::err($name . 'ErrLen', $lang['errLen']) . 
	                    HTMLForm::err($name . 'ErrNb', $lang['errNb']) . 
	                    HTMLForm::err($name . 'the_specified_color_is_not_valid', $lang['the_specified_color_is_not_valid']) .
	                    HTMLForm::err($name . 'ErrContentFilter', $lang['ErrContentFilter']) . '
	                </label>
	                <input type="' . htmlspecialchars($type) . '" id="' . htmlspecialchars($name) . '" name="' . htmlspecialchars($name) . '" value="' . htmlspecialchars($value) . '"' . $classAttr . $placeholderAttr . $disabledAttr . '>
	                ' . $descHtml . '
	            </div>';
	}

	/**
	 * Génère une zone de texte HTML.
	 *
	 * @param string $name      Nom de la zone de texte
	 * @param string $default   Valeur par défaut de la zone de texte
	 * @param string $class     Classe CSS à utiliser pour formater l'affichage
	 * @param string $desc      Valeur de la description
	 * @param int $rows         Nombre de lignes de la zone de texte
	 * @param string $placeholder Valeur du placeholder du champ (HTML5)
	 * @param bool $disabled    Désactivation de la zone de texte
	 * @return string          Code HTML de la zone de texte
	 */
	public static function textarea($name, $default = '', $class = '', $desc = '', $rows = '', $placeholder = '', $disabled = false)
	{
	    global $lang;
	    
	    // Récupérer la valeur du champ à partir de la méthode POST ou utiliser la valeur par défaut
	    $value = Util::isPOST($name) ? HTMLForm::transNL(HTMLForm::clean($_POST[$name])) : $default;
	
	    // Définir la classe CSS
	    $classAttr = !empty($class) ? ' class="form-control ' . htmlspecialchars($class) . '"' : ' class="form-control"';
	
	    // Définir la description
	    $descHtml = !empty($desc) ? '<small class="form-text text-muted">' . $lang[$desc] . '</small>' : '';
	
	    // Définir le nombre de lignes
	    $rowsAttr = !empty($rows) ? (int)$rows : 10; // Convertir en entier
	
	    // Définir l'attribut placeholder
	    $placeholderAttr = !empty($placeholder) ? ' placeholder="' . htmlspecialchars($lang[$placeholder]) . '"' : '';
	
	    // Définir l'attribut disabled
	    $disabledAttr = $disabled ? ' disabled' : '';
	
	    // Générer le code HTML pour la zone de texte
	    return '<div class="form-group">
	                <label for="' . htmlspecialchars($name) . '">' . $lang[$name] . '</label>
	                ' . Plugin::hook('editor') . '
	                <textarea id="' . htmlspecialchars($name) . '" name="' . htmlspecialchars($name) . '" rows="' . $rowsAttr . '"' . $classAttr . $placeholderAttr . $disabledAttr . '>' . htmlspecialchars($value) . '</textarea>
	                ' . $descHtml . 
	                HTMLForm::err($name . 'ErrLen', $lang['errLen']) . 
	                HTMLForm::err($name . 'ErrContentFilter', $lang['ErrContentFilter']) . '
	            </div>';
	}

	/**
	 * Génère un bouton de soumission avec un captcha.
	 *
	 * @param string $button  Le texte du bouton
	 * @param string $class   Classe CSS à appliquer au bouton
	 * @param string $icon    Classe de l'icône à afficher
	 * @param bool $cancel    Indique si un bouton d'annulation doit être affiché
	 * @return string        Code HTML du bouton de soumission
	 */
	public static function submit($button = 'submit', $class = '', $icon = '', $cancel = false)
	{
	    global $lang;
	    $out = '';
	
	    // Définir la classe CSS du bouton
	    $classAttr = !empty($class) ? ' class="' . htmlspecialchars($class) . '"' : ' class="btn btn-primary btn-lg"';
	
	    // Définir l'icône
	    $iconHtml = !empty($icon) ? '<i class="' . htmlspecialchars($icon) . '"></i>&nbsp;' : '';
	
	    // Définir le bouton d'annulation
	    $cancelButton = $cancel ? '&nbsp;<button type="reset" class="btn btn-secondary btn-lg" onclick="$(\'#form\').remove();"><i class="fa fa-times" aria-hidden="true"></i>&nbsp;' . htmlspecialchars($lang['cancel']) . '</button>' : '';
	
	    // Générer le captcha si nécessaire
	    if (CAPTCHA) {
	        $num1 = mt_rand(1, 20);
	        $num2 = mt_rand(1, 20);
	        if (mt_rand(0, 1) === 1) {
	            $math = "$num1 + $num2";
	            $rand = (string)($num1 + $num2);
	        } else {
	            $math = "$num1 - $num2";
	            $rand = (string)($num1 - $num2);
	        }
	
	        // Stocker le code captcha dans la session
	        if (empty(session_id())) {
	            session_start(); // Démarrer la session si elle n'est pas déjà démarrée
	        }
	        $_SESSION['captcha'] = $rand;
	
	        $out .= '
	            <div class="input-group mb-3">
	                <div class="input-group-prepend">
	                    <span class="input-group-text">' . $math . ' = ?</span>
	                </div>
	                <label for="captcha" class="sr-only">' . htmlspecialchars($lang['captcha']) . '</label>
	                <input name="captcha" type="text" class="form-control" placeholder="' . htmlspecialchars($lang['math_result']) . '">
	            </div>';
	    } else {
	        $out .= '
	            <div class="input-group mb-3">
	                <div class="input-group-prepend">
	                    <span class="input-group-text">' . Plugin::hook('captcha') . '<img id="cap-img" src="' . HTML_BASEPATH . DS . 'lib' . DS . 'Captcha.lib.php?rand=' . rand() . '" alt="' . htmlspecialchars($lang['captcha']) . '" /></span>
	                    <a class="input-group-text" onClick="reload();" data-toggle="tooltip" data-placement="top" title="' . htmlspecialchars($lang['r_captcha']) . '">
	                        <i class="fa fa-refresh" aria-hidden="true"></i>
	                    </a>
	                </div>
	                <label for="captcha" class="sr-only">' . htmlspecialchars($lang['captcha']) . '</label>
	                <input name="captcha" type="text" class="form-control" placeholder="' . htmlspecialchars($lang['enter_code']) . '">
	            </div>';
	    }
	
	    // Retourner le code HTML du bouton de soumission
	    return $out . HTMLForm::err('ErrToken', $lang['invalid_token']) . HTMLForm::err('ErrBot', $lang['errBot']) . '
	            <button' . $classAttr . ' type="submit">' . $iconHtml . htmlspecialchars($lang[$button]) . '</button>' .
	            $cancelButton . '
	            <script>
	                function reload() {
	                    var img = document.images["cap-img"];
	                    img.src = img.src.substring(0, img.src.lastIndexOf("?")) + "?rand=" + Math.random() * 1000;
	                }
	            </script>';
	}
	
	/**
	 * Génère un bouton de soumission simple.
	 *
	 * @param string $button  Le texte du bouton
	 * @param string $class   Classe CSS à appliquer au bouton
	 * @param string $icon    Classe de l'icône à afficher
	 * @param bool $cancel    Indique si un bouton d'annulation doit être affiché
	 * @return string        Code HTML du bouton de soumission
	 */
	public static function simple_submit($button = 'submit', $class = '', $icon = '', $cancel = false)
	{
	    global $lang;
	
	    // Définir la classe CSS du bouton
	    $classAttr = !empty($class) ? ' class="' . htmlspecialchars($class) . '"' : ' class="btn btn-primary"';
	
	    // Définir l'icône
	    $iconHtml = !empty($icon) ? '<i class="' . htmlspecialchars($icon) . '"></i>&nbsp;' : '';
	
	    // Définir le bouton d'annulation
	    $cancelButton = $cancel ? '&nbsp;<button type="reset" class="btn btn-secondary" onclick="$(\'#form\').remove();"><i class="fa fa-times" aria-hidden="true"></i>&nbsp;' . htmlspecialchars($lang['cancel']) . '</button>' : '';
	
	    // Retourner le code HTML du bouton de soumission
	    return HTMLForm::err('ErrToken', $lang['invalid_token']) . '<button' . $classAttr . ' type="submit">' . $iconHtml . htmlspecialchars($lang[$button]) . '</button>' . $cancelButton;
	}
	
	/**
	 * Génère un menu déroulant (select).
	 *
	 * @param string $name      Nom du champ
	 * @param array $options    Options du menu déroulant
	 * @param string $default   Valeur par défaut sélectionnée
	 * @param string $class     Classe CSS à appliquer au select
	 * @param string $desc      Description à afficher sous le select
	 * @param bool $disabled    Indique si le select doit être désactivé
	 * @return string          Code HTML du menu déroulant
	 */
	public static function select($name, $options, $default = '', $class = '', $desc = '', $disabled = false)
	{
	    global $lang;
	
	    // Définir la classe CSS du select
	    $classAttr = !empty($class) ? ' class="form-control ' . htmlspecialchars($class) . '"' : ' class="custom-select"';
	
	    // Définir la description
	    $descHtml = !empty($desc) ? '<small class="form-text text-muted">' . $lang[$desc] . '</small>' : '';
	
	    // Définir l'attribut disabled
	    $disabledAttr = $disabled ? ' disabled' : '';
	
	    // Déterminer la valeur sélectionnée
	    $selected = Util::isPOST($name) && isset($options[$_POST[$name]]) ? $_POST[$name] : $default;
	
	    // Générer le code HTML pour le select
	    $out = '<div class="form-group">
	                <label class="form-label" for="' . htmlspecialchars($name) . '">' . $lang[$name] . '</label>
	                <select id="' . htmlspecialchars($name) . '" name="' . htmlspecialchars($name) . '"' . $classAttr . $disabledAttr . '>';
	
	    foreach ($options as $value => $option) {
	        $out .= '<option value="' . htmlspecialchars($value) . '"' . ($value == $selected ? ' selected="selected"' : '') . '>' . htmlspecialchars($option) . '</option>';
	    }
	
	    $out .= '</select>' . $descHtml . '</div>';
	
	    return $out;
	}
	
	/**
	 * Génère une case à cocher (checkbox).
	 *
	 * @param string $name  Le nom de la case à cocher
	 * @param string $default  La valeur par défaut (peut être 'on' ou 'off')
	 * @param string $desc  Description à afficher sous la case à cocher
	 * @return string  Code HTML de la case à cocher
	 */
	public static function checkBox($name, $default = '', $desc = '')
	{
	    global $lang;
	
	    // Récupérer la valeur de la case à cocher à partir de la méthode POST ou utiliser la valeur par défaut
	    $value = Util::isPOST($name) ? HTMLForm::clean($_POST[$name]) : $default;
	
	    // Définir la description
	    $descHtml = !empty($desc) ? '<small class="form-text text-muted">' . htmlspecialchars($lang[$desc]) . '</small>' : '';
	
	    // Générer le code HTML pour la case à cocher
	    return '<div class="form-group">
	                <div class="custom-control custom-switch">
	                    <input class="custom-control-input" id="' . htmlspecialchars($name) . '" name="' . htmlspecialchars($name) . '" type="checkbox"' . ($value ? ' checked' : '') . '>
	                    <label class="custom-control-label" for="' . htmlspecialchars($name) . '">' . $lang[$name] . '</label>
	                </div>
	                ' . $descHtml . '
	            </div>';
	}
	
	/**
	 * Génère un formulaire HTML.
	 *
	 * @param string $action  L'URL d'action du formulaire
	 * @param string $controls  Les contrôles du formulaire (champs, boutons, etc.)
	 * @param string $class  Classe CSS à appliquer au formulaire
	 * @param string $method  Méthode d'envoi du formulaire (GET ou POST)
	 * @param bool $enctype  Indique si le formulaire doit utiliser l'encodage multipart
	 * @return string  Code HTML du formulaire
	 */
	public static function form($action, $controls, $class = '', $method = 'post', $enctype = true)
	{
	    global $token;
	
	    // Définir la classe CSS du formulaire
	    $classAttr = !empty($class) ? ' class="' . htmlspecialchars($class) . '"' : '';
	
	    // Définir l'attribut enctype
	    $enctypeAttr = $enctype ? ' enctype="multipart/form-data"' : '';
	
	    // Générer le code HTML pour le formulaire
	    $form = '<form id="form" action="' . htmlspecialchars($action) . '" method="' . htmlspecialchars($method) . '"' . $classAttr . $enctypeAttr . '>
	                <input type="hidden" name="_token" value="' . htmlspecialchars($token) . '">';
	    $form .= $controls;
	    $form .= '</form>';
	
	    return $form;
	}

    /**
     * Méthode qui permet d'envoyer des fichiers
     * HTMLForm::upload($name, $dir, $size);
     *
     * @param string $text String
     * @return string
     */   
    public static function slugupload($text){
        $find = array("/Ğ/","/Ü/","/Ş/","/İ/","/Ö/","/Ç/","/ğ/","/ü/","/ş/","/ı/","/ö/","/ç/", "/I/");
        $replace = array("g","u","s","i","o","c","g","u","s","i","o","c","i");
        $text = preg_replace("/[^0-9a-zA-ZÄzÜŞİÖÇğüşıöç]/"," ",$text);
        $text = preg_replace($find,$replace,$text);
        $text = preg_replace("/ +/"," ",$text);
        $text = preg_replace("/ /","-",$text);
        $text = preg_replace("/\s/","",$text);
        $text = strtolower($text);
        $text = preg_replace("/^-/","",$text);
        $text = preg_replace("/-$/","",$text);
        return $text;
    }

    public static function upload($name, $dir, $size){
	    global $lang;
        if(isset($_FILES[$name])){
            if(!empty($_FILES[$name]["name"])){
                if(strpos($size, "KB")){
                    $img_size = floatval($size)*1024;
                }
                elseif(strpos($size, "MB")){
                    $img_size = floatval($size)*1048576;
                }
                elseif(strpos($size, "GB")){
                    $img_size = floatval($size)*1073741824;
                }

                $upload_path = $dir;
                $file_info = pathinfo($_FILES[$name]["name"]);
                $filename = self::slugupload($file_info["filename"]). '.' .$file_info["extension"];
                $unique_filename = rand(0, 999).uniqid().self::slug($file_info["filename"]). '.' .$file_info["extension"];
                $filename_with_path = $upload_path.$unique_filename;
      
                if($_FILES[$name]["size"] < $img_size){
                    $upload_control = move_uploaded_file($_FILES[$name]["tmp_name"], $filename_with_path);
                    if($upload_control){
                        $array = ["status"=> "success", "msg"=> "The file has been uploaded successfully!", "name"=> $filename_with_path];
                        return $array;
                    }else{
                        $array = ["status"=> "error", "msg"=> "An error occurred while uploading the file!", "name"=> ""];
                        return $array;
                    }
                }else{
                    return ["status"=> "error", "msg"=> "The file you are trying to upload exceeds the maximum file size limit!", "name"=> ""];
                }
            }else{
                return ["status"=> "error", "msg"=> "Please, select a file.", "name"=> ""];
            }
        }else{
            return false;
        }
    }
		
	/**
	 * Affiche un aperçu du contenu soumis.
	 *
	 * @param string $name Le nom du champ à prévisualiser
	 * @return string HTML de l'alerte de prévisualisation ou une chaîne vide
	 */
	public static function preview($name)
	{
	    return Util::isPOST($name) ? '<div class="alert alert-warning" role="alert">' . Parser::content(HTMLForm::transNL(HTMLForm::clean($_POST[$name]))) . '</div>' : '';
	}

	/**
	 * Vérifie la validité d'un champ de texte.
	 *
	 * @param string $name Le nom du champ à vérifier
	 * @param int $min La longueur minimale autorisée
	 * @param int $max La longueur maximale autorisée
	 * @return bool True si le champ est valide, sinon false
	 */
	public static function check($name, $min = 1, $max = 40)
	{
	    if (!Util::isPOST($name) || !isset($_POST[$name])) {
	        return false;
	    }
	
	    // Récupération et nettoyage du contenu
	    $content = filter_var($_POST[$name], FILTER_SANITIZE_FULL_SPECIAL_CHARS);
	
	    // Vérification du filtre de contenu
	    if (preg_match(CONTENT_FILTER, $content)) {
	        $_SESSION[$name . 'ErrContentFilter'] = 1;
	        return false;
	    }
	
	    // Vérification de la longueur du texte
	    $len = strlen(trim($content));
	    if ($len >= $min && $len <= $max) {
	        return true;
	    }
	
	    $_SESSION[$name . 'ErrLen'] = 1;
	    return false;
	}
	
	/**
	 * Vérifie la correspondance des mots de passe.
	 *
	 * @param string $name Le nom du champ de mot de passe
	 * @return bool True si les mots de passe correspondent, sinon false
	 */
	public static function checkPass($name)
	{
	    if (HTMLForm::check($name) && Util::isPOST($name . 'Confirm') && $_POST[$name] === $_POST[$name . 'Confirm']) {
	        return true;
	    }
	
	    $_SESSION[$name . 'ErrNotMatch'] = 1;
	    return false;
	}
	
	/**
	 * Vérifie si un nombre est valide.
	 *
	 * @param string $name Le nom du champ à vérifier
	 * @return bool True si le nombre est valide, sinon false
	 */
	public static function checkNb($name)
	{
	    if (isset($_POST[$name])) {
	        $num = $_POST[$name];
	        if (ctype_digit($num) && $num > 0) {
	            return true;
	        }
	    }
	
	    $_SESSION[$name . 'ErrNb'] = 1;
	    return false;
	}
	
	/**
	 * Vérifie si une adresse e-mail est valide.
	 *
	 * @param string $email L'adresse e-mail à vérifier
	 * @return bool True si l'adresse e-mail est valide, sinon false
	 */
	public static function checkMail($email)
	{
	    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
	}
	
	/**
	 * Vérifie si le captcha est valide.
	 *
	 * @return bool True si le captcha est valide, sinon false
	 */
	public static function checkBot()
	{
	    if (!Util::isPOST('captcha')) {
	        return false;
	    }
	
	    if (isset($_SESSION['captcha']) && $_POST['captcha'] === $_SESSION['captcha']) {
	        return true;
	    }
	
	    $_SESSION['ErrBot'] = 1;
	    return false;
	}
	
	/**
	 * Vérifie si l'utilisateur a une session active pour décider de l'utilisation du captcha.
	 *
	 * @return bool True si le captcha n'est pas nécessaire, sinon false
	 */
	public static function sessionTrip()
	{
	    global $sessionTrip;
	
	    // Vérifie si la session est active ou si l'utilisateur est un travailleur
	    return $sessionTrip || User::isWorker() || HTMLForm::checkBot();
	}
	
	/**
	 * Génère un bouton de soumission avec ou sans captcha selon la session de l'utilisateur.
	 *
	 * @param string $name Le nom du bouton
	 * @param string $icon La classe de l'icône à afficher
	 * @param string $default La valeur par défaut du bouton
	 * @return string Code HTML du bouton de soumission
	 */
	public static function tripCaptcha($name = 'submit', $icon = 'fa fa-paper-plane', $default = '')
	{
	    global $sessionTrip;
	
	    // Vérifie si la session est active ou si l'utilisateur est un travailleur
	    if ($sessionTrip || User::isWorker()) {
	        return HTMLForm::simple_submit($name, $default, $icon);
	    } else {
	        return HTMLForm::submit($name, $default, $icon);
	    }
	}
}