<?php defined('FLATBOARD') or die('Flatboard Community.');

/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
*/

class User
{
    /**
     * Constructeur protégé car il s'agit d'une classe statique.
     *
     * @access protected
     */
    protected function __construct()
    {
        // Rien ici
    }

    /**
     * Vérifie si l'utilisateur a des droits administratifs.
     * 
     * @return bool Vrai si l'utilisateur est admin, faux sinon.
     */
    public static function isAdmin()
    {
        return $_SESSION['role'] === 'admin';
    }

    /**
     * Vérifie si l'utilisateur a des droits de modérateur.
     * 
     * @return bool Vrai si l'utilisateur est modérateur ou admin, faux sinon.
     */
    public static function isWorker()
    {
        return $_SESSION['role'] === 'worker' || $_SESSION['role'] === 'admin';
    }

    /**
     * Vérifie si l'utilisateur est l'auteur d'une entrée.
     * 
     * @param mixed $entry L'entrée à vérifier
     * @param string $data Type d'entrée ('topic' ou 'reply')
     * @return bool Vrai si l'utilisateur est l'auteur, faux sinon.
     */
    public static function isAuthor($entry, $data = false)
    {
        global $sessionTrip;

        if ($data === 'topic') {
            $topicEntry = flatDB::readEntry('topic', $entry);
            $tripCrypt = HTMLForm::trip($sessionTrip, $entry);
            $authorLogin = isset($sessionTrip) ? $tripCrypt === $topicEntry['trip'] : false;

            return strpos($tripCrypt, '@') !== false ? $authorLogin : isset($_SESSION[$entry]);
        } elseif ($data === 'reply') {
            $replyEntry = flatDB::readEntry('reply', $entry);
            $tripCrypt = HTMLForm::trip($sessionTrip, $entry);
            $authorLogin = isset($sessionTrip) ? $tripCrypt === $replyEntry['trip'] : false;

            return strpos($tripCrypt, '@') !== false ? $authorLogin : isset($_SESSION[$entry]);
        } else {
            return isset($_SESSION[$entry]);
        }
    }

    /**
     * Authentifie un utilisateur.
     * 
     * @param string $trip Identifiant de l'utilisateur
     * @return bool Vrai si l'authentification réussit, faux sinon.
     */
    public static function login($trip)
    {
        global $config, $lang;

        $tripNocrypt = HTMLForm::clean(Parser::translitIt($trip));
        $tripCrypt = HTMLForm::trip($tripNocrypt, $trip);

        // Session Admin
        if ($tripCrypt === $config['admin']) {
            $_SESSION['role'] = 'admin';
            $_SESSION['trip'] = $tripNocrypt;
            $_SESSION['mail'] = $config['mail'];
            return true;
        }
        // Session Modérateur
        elseif (isset($config['worker'][$tripCrypt])) {
            $_SESSION['role'] = 'worker';
            $_SESSION['trip'] = $tripNocrypt;
            $_SESSION['mail'] = $config['worker'][$tripCrypt];
            return true;
        }
        // Session utilisateur
        elseif (strpos($trip, '@') !== false) {
            $parts = explode('@', $trip);
            if (count($parts) === 2 && strlen($parts[1]) >= 4) {
                $_SESSION['trip'] = $tripNocrypt;
                return true;
            } else {
                $_SESSION['bad_user_syntax'] = 1;
                return false;
            }
        }
        // Erreur d'identification
        else {
            $_SESSION['bad_user_syntax'] = 1;
            return false;
        }
    }

    /**
     * Regénère le fichier de clé.
     * 
     * @param string $trip Identifiant de l'utilisateur
     */
    public static function generateKey($trip)
    {
        // Suppression de la clé actuelle
        if (file_exists(DATA_DIR . 'key.php')) {
            @unlink(DATA_DIR . 'key.php');
        }

        // Création de la clé
        $key = password_hash($_POST['password'], PASSWORD_DEFAULT);
        // Sauvegarde la clé du mot de passe dans un fichier
        @file_put_contents(DATA_DIR . 'key.php', "<?php define('KEY', '$key'); ?>", 0666);

        // Nouveau hash Flatboard 2.0
        $config['admin'] = hash_hmac('sha1', $_POST['password'], KEY);
        flatDB::saveEntry('config', 'config', $config);
        $_SESSION['role'] = '';
    }

    /**
     * Fonction de cryptage et décryptage simple.
     * 
     * @param string $string Chaîne à crypter/décrypter
     * @param string $action 'e' pour crypter, 'd' pour décrypter
     * @return string|false Chaîne cryptée ou décrytée, ou false en cas d'erreur
     */
    public static function simple_crypt($string, $action = 'e')
    {
        // Vous pouvez changer ces valeurs par les vôtres
        $secret_key = KEY;
        $secret_iv  = KEY;

        $output = false;
        $encrypt_method = "AES-256-CBC";
        $key = hash('sha256', $secret_key);
        $iv = substr(hash('sha256', $secret_iv), 0, 16);

        if ($action === 'e') {
            $output = base64_encode(openssl_encrypt($string, $encrypt_method, $key, 0, $iv));
        } elseif ($action === 'd') {
            $output = openssl_decrypt(base64_decode($string), $encrypt_method, $key, 0, $iv);
        }

        return $output;
    }

    /**
     * Protège l'affichage du mot de passe d'un modérateur.
     * 
     * @param string $password Mot de passe à protéger
     * @return string Mot de passe masqué
     */
    public static function get_starred($password)
    {
        $len = strlen($password);
        return substr($password, 0, 1) . str_repeat('*', $len - 2) . substr($password, $len - 1, 1);
    }

    /**
     * Protège l'email via un affichage JavaScript.
     * 
     * @param string $email Email à protéger
     * @param string $word Texte à afficher
     * @return string Code HTML pour afficher l'email
     */
    public static function protect_email($email, $word)
    {
        $pieces = explode("@", $email);
        return '<script>
                    var a = "<a href=\'mailto:";
                    var b = "' . $pieces[0] . '";
                    var c = "' . $pieces[1] . '";
                    var d = "\' class=\'badge badge-dark\'><i class=\'fa fa-envelope\'></i> ";
                    var e = "' . $word . '";
                    var f = "</a>";
                    document.write(a + b + "@" + c + d + e + f);
                </script>
                <noscript>Activer JavaScript pour afficher le mail</noscript>';
    }

    /**
     * Récupère l'adresse IP réelle de l'utilisateur.
     * 
     * @return string Adresse IP
     */
    public static function getRealIpAddr()
    {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            // Vérifie l'IP depuis un réseau partagé
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            // Vérifie si l'IP est passée par un proxy
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            // Sinon, utilise l'adresse IP distante
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        return $ip;
    }

    /**
     * Vérifie si une adresse IP est bannie.
     * 
     * @param string $ip Adresse IP à vérifier
     * @return bool Vrai si l'IP est bannie, faux sinon
     */
    public static function is_ban($ip)
    {
        $IPlist = BAN_FILE;
        if (file_exists($IPlist)) {
            $blacklist = file_get_contents($IPlist);
            return $blacklist !== '' && strpos($blacklist, $ip . "\n") !== false;
        }
        return false;
    }

    /**
     * Vérifie si l'adresse IP de l'utilisateur est bannie.
     * Si l'IP est bannie, redirige l'utilisateur vers une page spéciale.
     */
    public static function checkIP()
    {
        global $lang, $config;
        $ip = User::getRealIpAddr();
        $list = file(BAN_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        foreach ($list as $line) {
            if (strpos($ip, $line) !== false) {
                require THEME_DIR . $config['theme'] . DS . 'banned.tpl.php';
                exit();
            }
        }
    }
}