<?php defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
 * 
 * Classe pour la gestion des actifs (CSS) dans Flatboard.
 *
 * Cette classe fournit des méthodes statiques pour combiner, compresser et mettre en cache des fichiers CSS.
 * Elle est conçue pour être utilisée dans le cadre du projet Flatboard sous la licence MIT.
 */
class Asset
{
    /** @var string Chemin du dossier de cache */
    private const CACHE_DIR = THEME_CONFIG_PATH . 'cache' . DS;

    /** @var string Nom du fichier d'index dans le dossier de cache */
    private const INDEX_FILE = 'index.html';

    /** @var string Nom par défaut du fichier CSS minifié */
    private const DEFAULT_CSS_FILENAME = 'styles.min.css';

    /**
     * Constructeur protégé pour empêcher l'instanciation.
     */
    protected function __construct()
    {
        // Classe statique, aucune instanciation nécessaire
    }

    /**
     * Génère un lien vers une feuille de style CSS minifiée.
     *
     * Cette méthode combine plusieurs fichiers CSS, les compresse et les met en cache.
     * Si le fichier de cache existe et est à jour, il est utilisé directement.
     *
     * @param string[] $files Liste des URLs ou chemins des fichiers CSS à combiner.
     * @param string $destinationDir Répertoire de destination pour le fichier minifié.
     * @param string $filename Nom du fichier minifié (par défaut : 'styles.min.css').
     * @param string $version Version de l'application pour le cache busting (par défaut : VERSION).
     * @return string Lien HTML vers la feuille de style minifiée.
     * @throws RuntimeException Si le dossier de cache ne peut pas être créé ou si une erreur d'écriture survient.
     */
    public static function stylesheet(
        array $files,
        string $destinationDir,
        string $filename = self::DEFAULT_CSS_FILENAME,
        string $version = VERSION
    ): string {
        // Valider les paramètres
        if (empty($files)) {
            throw new InvalidArgumentException('La liste des fichiers CSS ne peut pas être vide.');
        }

        $cacheDir = rtrim(self::CACHE_DIR, DS);
        $cacheFile = $cacheDir . DS . $filename;
        $indexFile = $cacheDir . DS . self::INDEX_FILE;

        // Créer le dossier de cache si nécessaire
        self::ensureCacheDirectory($cacheDir, $indexFile);

        // Vérifier si le fichier de cache existe et est à jour
        if (file_exists($cacheFile) && self::isCacheValid($files, $cacheFile, $destinationDir)) {
            return sprintf(
                '<link rel="stylesheet" href="%s?ver=%s" />' . PHP_EOL,
                htmlspecialchars($cacheFile),
                date('Y.m.d.H:i:s', filemtime($cacheFile))
            );
        }

        // Générer le contenu minifié
        $content = self::generateMinifiedContent($files);

        // Écrire dans le fichier de cache
        if (file_put_contents($cacheFile, $content) === false) {
            throw new RuntimeException("Impossible d'écrire le fichier de cache : {$cacheFile}");
        }

        return sprintf(
            '<link rel="stylesheet" href="%s?ver=%s" />' . PHP_EOL,
            htmlspecialchars($cacheFile),
            htmlspecialchars($version)
        );
    }

    /**
     * Compresse le contenu d'un fichier CSS.
     *
     * Supprime les espaces inutiles, les retours à la ligne et les commentaires.
     *
     * @param string $file Chemin ou URL du fichier CSS.
     * @return string Contenu CSS compressé.
     */
    private static function compressCSS(string $file): string
    {
        $content = @file_get_contents($file);
        if ($content === false) {
            return ''; // Retourner une chaîne vide si le fichier est inaccessible
        }

        // Supprimer les retours à la ligne, tabulations et espaces multiples
        $content = str_replace(["\r\n", "\r", "\n", "\t", '  ', '    '], '', $content);

        // Supprimer les commentaires CSS
        $content = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $content);

        // Réduire les espaces autour des délimiteurs
        $content = str_replace(
            ['{ ', ' }', '; ', ', ', ' {', '} ', ': ', ' ,', ' ;'],
            ['{', '}', ';', ',', '{', '}', ':', ',', ';'],
            $content
        );

        return $content;
    }

    /**
     * Crée le dossier de cache et le fichier index.html si nécessaire.
     *
     * @param string $cacheDir Chemin du dossier de cache.
     * @param string $indexFile Chemin du fichier index.html.
     * @throws RuntimeException Si le dossier de cache ne peut pas être créé.
     */
    private static function ensureCacheDirectory(string $cacheDir, string $indexFile): void
    {
        if (!is_dir($cacheDir)) {
            if (!mkdir($cacheDir, 0777, true) || !chmod($cacheDir, 0777)) {
                throw new RuntimeException("Impossible de créer le dossier de cache : {$cacheDir}");
            }
        }

        if (!file_exists($indexFile)) {
            file_put_contents($indexFile, ''); // Créer un fichier index.html vide
        }
    }

    /**
     * Vérifie si le fichier de cache est valide (non obsolète).
     *
     * @param string[] $files Liste des fichiers CSS source.
     * @param string $cacheFile Chemin du fichier de cache.
     * @param string $sourceDir Répertoire des fichiers source.
     * @return bool True si le cache est valide, false sinon.
     */
    private static function isCacheValid(array $files, string $cacheFile, string $sourceDir): bool
    {
        $cacheTime = filemtime($cacheFile);

        foreach ($files as $file) {
            $sourceFile = rtrim($sourceDir, DS) . DS . basename($file);
            if (file_exists($sourceFile) && filemtime($sourceFile) > $cacheTime) {
                return false; // Cache obsolète si un fichier source est plus récent
            }
        }

        return true;
    }

    /**
     * Génère le contenu minifié à partir des fichiers CSS.
     *
     * @param string[] $files Liste des fichiers CSS.
     * @return string Contenu CSS combiné et minifié.
     */
    private static function generateMinifiedContent(array $files): string
    {
        $content = '';

        foreach ($files as $file) {
            if (empty($file)) {
                continue; // Ignorer les fichiers vides
            }
            $content .= self::compressCSS($file); // Ajouter le contenu compressé
        }

        return $content;
    }
}