<?php
/*
 * PSX is an open source PHP framework to develop RESTful APIs.
 * For the current version and information visit <https://phpsx.org>
 *
 * Copyright 2010-2024 Christoph Kappestein <christoph.kappestein@gmail.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace PSX\Api\Generator\Server;

use PSX\Api\Generator\Server\Dto\Context;
use PSX\Api\Generator\Server\Dto\File;
use PSX\Api\Generator\Server\Dto\Folder;
use PSX\Api\OperationInterface;
use PSX\Api\SpecificationInterface;
use PSX\Schema\ContentType;
use PSX\Schema\Generator;
use PSX\Schema\GeneratorInterface as SchemaGeneratorInterface;
use PSX\Schema\TypeInterface;

/**
 * TypeScript
 *
 * @author  Christoph Kappestein <christoph.kappestein@gmail.com>
 * @license http://www.apache.org/licenses/LICENSE-2.0
 * @link    https://phpsx.org
 */
class TypeScript extends ServerAbstract
{
    protected function newGenerator(): SchemaGeneratorInterface
    {
        return new Generator\TypeScript();
    }

    protected function getControllerPath(): string
    {
        return 'controller';
    }

    protected function getModelPath(): string
    {
        return 'dto';
    }

    protected function getFileExtension(): string
    {
        return 'ts';
    }

    protected function getFileContent(string $code, string $identifier): string
    {
        $comment = '/**' . "\n";
        $comment.= ' * ' . $identifier . ' automatically generated by SDKgen please do not edit this file manually' . "\n";
        $comment.= ' * {@link https://sdkgen.app}' . "\n";
        $comment.= ' */' . "\n";

        return $comment . "\n" . $code;
    }

    protected function buildControllerFileName(string $name): string
    {
        return $name . '.controller';
    }

    protected function generateHeader(File $file, array $imports): string
    {
        $path = [];
        $folder = $file->getFolder();
        while ($folder->getName() !== '.') {
            $path[] = $this->normalizer->class($folder->getName());
            $folder = $folder->getParent();
        }

        $basePath = str_repeat('../', count($path)) . '../dto';

        $controllerClass = ucfirst($file->getName()) . 'Controller';

        $controller = 'import { Controller, Get, Post, Put, Patch, Delete, HttpCode, Param, Query, Headers, Body } from \'@nestjs/common\'' . "\n";

        foreach ($imports as $className) {
            $controller.= 'import { ' . $className . ' } from "' . $basePath . '/' . $className . '";' . "\n";
        }

        $controller.= "\n";
        $controller.= '@Controller()' . "\n";
        $controller.= 'export class ' . $controllerClass . ' {' . "\n";

        return $controller;
    }

    protected function generateFooter(File $file): string
    {
        $controller = '}' . "\n";

        return $controller;
    }

    protected function generateArgumentPath(string $rawName, string $variableName, string $type, TypeInterface|ContentType $argumentType): string
    {
        return '@Param(\'' . $rawName . '\') ' . $variableName . ': ' . $type;
    }

    protected function generateArgumentQuery(string $rawName, string $variableName, string $type, TypeInterface|ContentType $argumentType): string
    {
        return '@Query(\'' . $rawName . '\') ' . $variableName . ': ' . $type;
    }

    protected function generateArgumentHeader(string $rawName, string $variableName, string $type, TypeInterface|ContentType $argumentType): string
    {
        return '@Headers(\'' . $rawName . '\') ' . $variableName . ': ' . $type;
    }

    protected function generateArgumentBody(string $variableName, string $type, TypeInterface|ContentType $argumentType): string
    {
        return '@Body() ' . $variableName . ': ' . $type;
    }

    protected function generateMethod(string $operationName, OperationInterface $operation, array $arguments, string $type, TypeInterface|ContentType $returnType): string
    {
        $methodName = ucfirst(strtolower($operation->getMethod()));

        $method = '  @' . $methodName . '(\'' . $operation->getPath() . '\')' . "\n";
        $method.= '  @HttpCode(' . $operation->getReturn()->getCode() . ')' . "\n";
        $method.= '  ' . $operationName . '(' . implode(', ', $arguments) . '): ' . $type . ' {' . "\n";
        $method.= '    // @TODO implement method' . "\n";
        $method.= '    return {};' . "\n";
        $method.= '  }' . "\n";
        $method.= "\n";

        return $method;
    }

    protected function buildContext(SpecificationInterface $specification, Folder $folder): Context
    {
        $context = parent::buildContext($specification, $folder);
        $context['controllers'] = $this->fetchControllers($folder);

        return $context;
    }

    private function fetchControllers(Folder $folder, array $path = []): array
    {
        $controllers = [];

        $folders = $folder->getFolders();
        foreach ($folders as $child) {
            $controllers = array_merge($controllers, $this->fetchControllers($child, array_merge($path, [$child->getName()])));
        }

        $files = $folder->getFiles();
        foreach ($files as $file) {
            $controllers[(count($path) > 0 ? implode('/', $path) . '/' : '') . $file->getName()] = $this->normalizer->class($file->getName(), 'Controller');
        }

        return $controllers;
    }
}
