<?php

namespace go\modules\business\mollie;

use DateTime;
use Exception;
use go\core;
use go\core\ErrorHandler;
use go\core\exception\Forbidden;
use go\core\orm\exception\SaveException;
use go\modules\business\finance\model\FinanceDocument;
use go\modules\business\finance\model\Payment;
use go\modules\business\finance\model\PaymentProviderInterface;
use go\modules\business\finance\model\RecurringPaymentProviderInterface;
use go\modules\community\addressbook\model\Contact;
use go\modules\community\addressbook\model\EmailAddress;
use Mollie\Api\Exceptions\ApiException;
use Mollie\Api\MollieApiClient;


/**
 * @copyright (c) 2020, Intermesh BV https://www.intermesh.nl
 * @author Merijn Schering <mschering@intermesh.nl>
 * @license http://www.gnu.org/licenses/agpl-3.0.html AGPLv3
 */
class Module extends core\Module implements PaymentProviderInterface, RecurringPaymentProviderInterface
{
	use core\event\EventEmitterTrait;

	public function getAuthor() : string
	{
		return "Intermesh BV <info@intermesh.nl>";
	}

    /**
     * The development status of this module
     * @return string
     */
    public function getStatus() : string{
        return self::STATUS_STABLE;
    }

	public function getDependencies() : array
	{
		return ['business/finance'];
	}

	public function requiredLicense(): ?string
	{
		return "billing";
	}


	public function getSettings()
	{
	 return model\Settings::get();
	}

	private $mollie;

	/**
	 * @throws ApiException
	 */
	private function getMollie(): MollieApiClient
	{
		//autoload composer libs for this module
		require_once __DIR__ . '/vendor/autoload.php';

		if(!isset($this->mollie)) {
			$this->mollie = new MollieApiClient();
			$this->mollie->setApiKey(Module::get()->getSettings()->apiKey);
		}

		return $this->mollie;
	}

	/**
	 * @throws Forbidden
	 */
	private function getDocument(int $documentId, string $token) : FinanceDocument {
		$document = FinanceDocument::findById($documentId);
		if ($document->token != $token) {
			throw new Forbidden();
		} else{
			go()->setAuthState(new core\auth\TemporaryState($document->createdBy));
		}

		return $document;
	}

	private function getCustomerId(FinanceDocument $document, bool $autoCreate = true) : ?string {

		$customerId = go()->getDbConnection()
			->selectSingleValue("customerId")
			->from("business_mollie_contact")
			->where("contactId", "=", $document->getCustomerId())
			->single();

		if($customerId) {
			return $customerId;
		}

		if(!$autoCreate) {
			return null;
		}

		$contact = Contact::findById($document->getCustomerId());

		$mollie = $this->getMollie();
		$customer = $mollie->customers->create([
			"name" => $contact->name,
			"email" => $contact->findEmailByType(EmailAddress::TYPE_BILLING)->email
		]);

		go()->getDbConnection()
			->insert("business_mollie_contact", [
				"contactId" => $document->getCustomerId(),
				"customerId" => $customer->id
			])
			->execute();

		return $customer->id;
	}

	/**
	 * @param int $documentId
	 * @param string $token
	 * @param bool $recurring
	 * @throws ApiException
	 * @throws Forbidden
	 */
	public function pagePayment(int $documentId, string $token, bool $recurring = false) {

		$document = $this->getDocument($documentId, $token);

		$mollie = $this->getMollie();

		$redirectUrl = go()->getAuthState()->getPageUrl() .	"/business/mollie/return/". $document->id . "/" . $document->token;

		if(!empty($_GET['returnUrl'])) {
			$redirectUrl .= '?returnUrl=' . rawurlencode($_GET['returnUrl']);
		}

		try {

			if($document->getTotalPrice() > Module::get()->getSettings()->maxAmount) {
				throw new Exception("Sorry, the amount is too high. Please pay by bank transfer.");
			}

			$data = [
				"amount" => [
					"currency" => "EUR",
					"value" => number_format($document->getTotalPrice(), 2, ".", "")
				],
				"description" => $document->number,
				"redirectUrl" => $redirectUrl,
				"cancelUrl" => $document->getCustomerUrl(),
				"webhookUrl"  => go()->getAuthState()->getPageUrl() . "/business/mollie/webhook/". $document->id . "/" . $document->token,
				"metadata" => [
					//Unused but might be useful in the future
					"documentId" => $document->id,
				],
			];

			if($recurring) {
				$data['customerId'] = $this->getCustomerId($document);
				$data['sequenceType'] = "first";
				$data['description'] .= " - " . go()->t("Direct debit", "business", "finance");
			}

			$payment = $mollie->payments->create($data);

			$document->paymentStatus = $payment->status;
			$document->paymentId = $payment->id;

			if(!$document->save()) {
				throw new SaveException($document);
			}

			//Make sure this page is not cached by the browser
			header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
			header("Cache-Control: post-check=0, pre-check=0", false);
			header("Pragma: no-cache");
			header("Location: " . $payment->getCheckoutUrl(), true, 303);
			exit();

		} catch (Exception $e) {
			$error = htmlspecialchars($e->getMessage());

			$returnUrl = $_GET['returnUrl'] ?? $document->getCustomerUrl();
			require(__DIR__ . '/views/web/error.php');
		}
	}

	/**
	 * @throws Forbidden
	 */
	public function pageReturn(int $documentId, string $token) {

		$document = $this->getDocument($documentId, $token);

		if(!$document->isPaid()) {
			$this->pageWebhook($documentId, $token);
		}

		$returnUrl = $_GET['returnUrl'] ?? $document->getCustomerUrl();

		require(__DIR__ . '/views/web/return.php');
	}


	public function backgroundPayment(FinanceDocument $document): void
	{
		if($document->isPaid()) {
			throw new Exception("Already paid");
		}

		if(!$this->hasMandate($document)) {
			throw new Exception("No mandate for customer");
		}

		$data = [
			"amount" => [
				"currency" => "EUR",
				"value" => number_format($document->getTotalPrice(), 2, ".", "")
			],
			"customerId" => $this->getCustomerId($document),
			"description" => $document->number . " (background payment)",
			"sequenceType" => "recurring",
			"webhookUrl"  => go()->getAuthState()->getPageUrl() . "/business/mollie/webhook/". $document->id . "/" . $document->token,
			"metadata" => [
				//Unused but might be useful in the future
				"documentId" => $document->id,
			],
		];

		$payment = $this->getMollie()->payments->create($data);

		$document->paymentStatus = $payment->status;
		$document->paymentId = $payment->id;
		$document->recurringPayment = true;

		if(!$document->save()) {
			throw new SaveException($document);
		}
	}

	public function hasMandate(FinanceDocument $document) :bool {
		$customerId = $this->getCustomerId($document, false);

		if(!$customerId) {
			return false;
		}
		$customer = $this->getMollie()->customers->get($customerId);

		foreach($customer->mandates() as $mandate) {
			if($mandate->status == "valid") {
				return true;
			}
		}

		return false;
	}

	/**
	 * @param int $documentId
	 * @param string $token
	 * @throws Forbidden
	 * @throws SaveException
	 * @throws Exception
	 */
	public function pageWebhook(int $documentId, string $token) {

		/*
		 * How to verify Mollie API Payments in a webhook.
		 *
		 * See: https://docs.mollie.com/guides/webhooks
		 */

		try {

			$document = $this->getDocument($documentId, $token);

			if(!$document->isPaid()) {
				$mollie = $this->getMollie();

				/*
				 * Retrieve the payment's current state.
				 */
				$payment = $mollie->payments->get($document->paymentId);

				$document->paymentStatus = $payment->status;
				if (!$document->save()) {
					throw new SaveException($document);
				}

				if ($payment->isPaid() && !$payment->hasRefunds() && !$payment->hasChargebacks()) {

					$p = (new Payment())->setValues([
						'businessId' => $document->findBook()->businessId,
						'documentId' => $document->id,
						'customerId' => $document->getCustomerId(),
						'reference' => 'mollie-' . $payment->id,
						'date' => new DateTime(),
						'amount' => $payment->amount->value,
						'description' => "Mollie transaction: " . $payment->description
					]);

					if (!$p->save()) {
						throw new SaveException($p);
					}

				}

				if($document->recurringPayment) {
					$contract = $document->getContract();
					if($contract) {
						$contract->sendInvoice($document);
					}
				}
			}
		} catch (Exception $e) {
			echo "API call failed: " . htmlspecialchars($e->getMessage());
			ErrorHandler::logException($e);
		}
	}

	public function getPaymentProviderTitle(): string
	{
		return go()->t("Pay via Mollie");
	}
}
