/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { FunctionUtil } from "../util/FunctionUtil.js";
/**
 * Observable
 *
 * Adds event listener functionality
 */
export class Observable {
    /**
     * Add a listener
     *
     * @param eventName
     * @param listener
     * @param options
     */
    on(eventName, listener, options) {
        //store original listener for the un() method. Because options may change the function
        const unbindkey = listener;
        if (options) {
            if (options.buffer !== undefined) {
                listener = FunctionUtil.buffer(options.buffer, listener);
            }
            if (options.once) {
                listener = this.once(eventName, listener);
            }
            if (options.delay) {
                listener = FunctionUtil.delay(options.delay, listener);
            }
        }
        this.lisnrs = this.lisnrs || {};
        if (!this.lisnrs[eventName]) {
            this.lisnrs[eventName] = [];
            this.onFirstListenerAdded(eventName);
        }
        if (options === null || options === void 0 ? void 0 : options.unshift) {
            this.lisnrs[eventName].unshift({ listener: listener, options: options, unbindkey: unbindkey });
        }
        else {
            this.lisnrs[eventName].push({ listener: listener, options: options, unbindkey: unbindkey });
        }
        return unbindkey;
    }
    /**
     * Override this function to lazily initialize events
     * @param eventName
     * @protected
     */
    onFirstListenerAdded(eventName) {
    }
    once(eventName, listener) {
        const newfn = (...args) => {
            listener.apply(null, args);
            // use set timeout so for .. of loop will continue with next listeners
            setTimeout(() => {
                this.un(eventName, listener);
            });
        };
        return newfn;
    }
    /**
     * Remove listener
     *
     * @param eventName
     * @param listener
     */
    un(eventName, listener) {
        if (!this.lisnrs || !this.lisnrs[eventName]) {
            return false;
        }
        for (let i = 0, l = this.lisnrs[eventName].length; i < l; i++) {
            const l = this.lisnrs[eventName][i];
            if (l.unbindkey === listener) {
                this.lisnrs[eventName].splice(i, 1);
                return true;
            }
        }
        return false;
    }
    /**
     * Fire an event
     *
     * When a listener returns false this function will return false too.
     *
     * @param eventName
     * @param args
     */
    fire(eventName, ...args) {
        // console.log(eventName, ...args);
        if (!this.lisnrs || !this.lisnrs[eventName]) {
            return true;
        }
        let ret = true;
        for (let l of this.lisnrs[eventName]) {
            if (l.listener.apply(null, args) === false) {
                ret = false;
            }
        }
        return ret;
    }
    relayEvent(comp, type) {
        //@ts-ignore
        comp.on(type, (...args) => {
            //@ts-ignore
            this.fire(type, ...args);
        });
    }
}
//# sourceMappingURL=Observable.js.map