import { List } from "./List";
import { store } from "../data";
import { Component, createComponent } from "./Component";
import { E } from "../util";
import { dragData } from "../DragData";
import { checkbox } from "./form";
export const TreeRowRenderer = (record, row, me, storeIndex) => {
    const node = E("div").cls("node"), caret = E("span").cls("caret"), icon = E("i").cls("icon"), label = E("a");
    icon.innerText = record.icon || "folder";
    label.innerText = record.text;
    if (record.href) {
        label.href = record.href;
    }
    if (record.children && record.children.length == 0) {
        row.cls("+no-children");
    }
    if (record.cls) {
        row.cls("+" + record.cls);
    }
    node.append(caret, icon);
    if (record.check != undefined) {
        const c = checkbox({
            value: record.check,
            listeners: {
                change: (field, newValue, oldValue) => {
                    record.check = newValue;
                    const top = me.findTopTree();
                    top.fire("checkchange", top, this, record, storeIndex, newValue);
                }
            }
        });
        c.render(node);
    }
    node.append(label);
    row.append(node);
};
export class Tree extends List {
    constructor(renderer = TreeRowRenderer) {
        super(store(), renderer);
        this.renderer = renderer;
        this.subTrees = {};
        this.expandedIds = {};
        this.parentStoreIndex = -1;
        this.baseCls = "goui goui-tree";
        this.on("rowclick", (list, storeIndex, row, ev) => {
            if (list == this) {
                void this._expand(row);
            }
        });
        this.store.on("remove", (collection, item, index) => {
            delete this.subTrees[index];
        });
        if (!(this.parent instanceof Tree)) {
            this.on("beforerender", () => {
                this.fire("beforeexpand", this, this, undefined, -1);
            });
            this.on("render", () => {
                this.fire("expand", this, this, undefined, -1);
            });
        }
        this.emptyStateHtml = "";
    }
    set data(records) {
        this.store.loadData(records);
    }
    /**
     * The full hierarchical store data of the tree
     */
    get data() {
        return this.store.data;
    }
    /**
     * Reload the tree if loaded via the "expand" event.
     */
    reload() {
        this.store.clear();
        this.fireExpand();
        for (let id in this.subTrees) {
            this.subTrees[id].reload();
        }
    }
    fireExpand() {
        if (!(this.parent instanceof Tree)) {
            // top level reload
            this.fire("beforeexpand", this, this, undefined, -1);
            this.fire("expand", this, this, undefined, -1);
        }
        else {
            const record = this.parent.store.get(this.parentStoreIndex), top = this.findTopTree();
            top.fire("beforeexpand", top, this, record, this.parentStoreIndex);
            top.fire("expand", top, this, record, this.parentStoreIndex);
        }
    }
    /**
     * Find the first menu in the tree of submenu's
     */
    findTopTree() {
        if (this.parent instanceof Tree) {
            return this.parent.findTopTree();
        }
        else {
            return this;
        }
    }
    /**
     * Expand this tree or a child node when index is given
     *
     * @param index
     */
    expand(index) {
        if (index === undefined || index == -1) {
            if (this.parent instanceof Tree) {
                this.parent.expand(this.parentStoreIndex);
            }
            return;
        }
        this._expand(this.getRowElements()[index]);
    }
    /**
     * Collapse this tree or a child node when index is given
     *
     * @param index
     */
    collapse(index) {
        if (!index || index == -1) {
            if (this.parent instanceof Tree) {
                this.parent.collapse(this.parentStoreIndex);
            }
            return;
        }
        this._collapse(this.getRowElements()[index]);
    }
    _expand(row) {
        const storeIndex = this.getRowElements().indexOf(row);
        const record = this.store.get(storeIndex);
        if (!record) {
            debugger;
            throw "Record not found for index: " + storeIndex;
        }
        const tree = this.subTrees[storeIndex] ? this.subTrees[storeIndex] : this.renderSubTree(row, record, storeIndex);
        const top = this.findTopTree();
        if (top.fire("beforeexpand", top, tree, record, storeIndex) === false) {
            return tree;
        }
        row.cls("+expanded");
        //we set the height so we can use an animation
        tree.el.style.height = tree.el.scrollHeight + "px";
        tree.el.addEventListener("transitionend", () => {
            //check if it's still expanded to handle fast clickers
            if (row.has(".expanded")) {
                tree.el.style.height = "auto";
            }
        }, { once: true });
        top.fire("expand", top, tree, record, storeIndex);
        return tree;
    }
    renderSubTree(row, record, storeIndex) {
        if (!record.id) {
            if (this.parentStoreIndex > -1) {
                record.id = this.parent.store.get(this.parentStoreIndex).id + "-" + storeIndex;
            }
            else {
                record.id = storeIndex + "";
            }
        }
        this.findTopTree().expandedIds[record.id] = true;
        const sub = new Tree(this.renderer);
        sub.dropOn = this.dropOn;
        sub.dropBetween = this.dropBetween;
        sub.draggable = this.draggable;
        sub.parent = this;
        sub.parentStoreIndex = storeIndex;
        if (record.children) {
            sub.data = record.children;
        }
        // set the data on parent if store is loaded by an expand event.
        // this way the data can be retrieved in full using the top tree's data get accessor. eg. tree.data
        sub.store.on("load", (store1, records, append) => {
            record.children = store1.data;
        });
        ["rowclick", "rowdblclick", "rowcontextmenu", "rowmousedown"].forEach((e) => {
            this.relayEvent(sub, e);
        });
        this.subTrees[storeIndex] = sub;
        //wrapper is needed for css transition transform to hide overflow
        const wrap = document.createElement("div");
        wrap.classList.add("wrap");
        row.append(wrap);
        sub.render(wrap);
        return sub;
    }
    _collapse(row) {
        row.cls("-expanded");
        const storeIndex = this.getRowElements().indexOf(row);
        //we set height 0 for animation
        this.subTrees[storeIndex].el.style.height = this.subTrees[storeIndex].el.offsetHeight + "px";
        requestAnimationFrame(() => {
            this.subTrees[storeIndex].el.style.height = "0";
        });
        const record = this.store.get(storeIndex);
        if (!record) {
            throw "Record not found";
        }
        const top = this.findTopTree();
        top.expandedIds[record.id] = true;
        top.fire("collapse", top, this, record, storeIndex);
    }
    renderRow(record, storeIndex) {
        let row = super.renderRow(record, storeIndex);
        row.cls("+data");
        row.id = Component.uniqueID();
        if (this.draggable) {
            row.draggable = true;
            row.ondragstart = this.onNodeDragStart.bind(this);
        }
        row.addEventListener("contextmenu", (e) => {
            e.stopPropagation();
            e.preventDefault();
            const sub = this._expand(row);
            sub.reload();
        });
        this.bindDropEvents(row);
        row.getElementsByClassName("caret")[0].on("click", (e) => {
            row.has(".expanded") ? this._collapse(row) : this._expand(row);
            e.preventDefault();
            e.stopPropagation();
        });
        return row;
    }
    onRowAppend(row, record, index) {
        if (this.findTopTree().expandedIds[record.id]) {
            this._expand(row);
        }
    }
    onNodeDragEnterAllowed(e, dropRow) {
        clearTimeout(this.dragOverTimeout);
        setTimeout(() => {
            //expand tree node if dragging over for 1 second
            this.dragOverTimeout = setTimeout(() => {
                this._expand(dropRow);
            }, 1000);
        });
    }
    onNodeDragLeaveAllowed(e, dropRow) {
        clearTimeout(this.dragOverTimeout);
    }
    dropAllowed(e, dropRow) {
        const topTree = this.findTopTree();
        return (dragData.row && !dragData.row.contains(dropRow)); // && topTree.fire("dropallowed", topTree, e, dropRow, dragData);
    }
    onNodeDrop(e) {
        const dropPos = this.getDropPosition(e);
        if (!dropPos) {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        const dropRow = this.findDropRow(e), dropIndex = this.getRowElements().indexOf(dropRow);
        this.clearOverClasses(dropRow);
        clearTimeout(this.dragOverTimeout);
        const dropTree = this._expand(dropRow);
        dragData.dropTree = this;
        dragData.childrenTree = dropTree;
        const topTree = this.findTopTree();
        topTree.fire("drop", topTree, e, dropRow, dropIndex, dropPos, dragData);
        return false;
    }
}
/**
 * Shorthand function to create {@see Table}
 *
 * @param config
 */
export const tree = (config) => createComponent(new Tree(), config);
//# sourceMappingURL=Tree.js.map