/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Store } from "../data/Store.js";
import { ObjectUtil } from "../util/index.js";
import { createComponent } from "../component/index.js";
/**
 * DataSourceStore class
 *
 * Uses an {@see AbstractDataSource} to present data in the view.
 * @category Data
 */
export class DataSourceStore extends Store {
    // private bufferedLoad?: (...args:any[]) => Promise<StoreRecord[]>;
    constructor(dataSource) {
        super();
        this.dataSource = dataSource;
        this.queryParams = {};
        this.hasMore = false;
        // public properties?: string[] = [];
        /**
         * Reload when the datasource changes
         */
        this.monitorChanges = true;
        /**
         * Builds record from entity
         * @param entity
         */
        this.buildRecord = async (entity) => entity;
        this.filters = {};
        this.dataSource.on('change', (dataSource, changes) => {
            void this.onChange(dataSource, changes);
        });
    }
    /**
     * Reloads the store when the datasource changes
     *
     * @protected
     */
    async onChange(DataSource, changes) {
        if (this.loaded && this.monitorChanges && !this.loading) {
            void this.reload();
        }
    }
    async internalLoad(append = false) {
        const queryParams = structuredClone(this.queryParams);
        queryParams.sort = this.sort;
        if (queryParams.limit) {
            //to see if the remote has more data we query one ID more.
            queryParams.limit++;
        }
        const queryResponse = await this.dataSource.query(queryParams);
        if (this.queryParams.limit) {
            // check if the server has more data.
            this.hasMore = queryResponse.ids.length > this.queryParams.limit;
            if (this.hasMore) {
                queryResponse.ids.pop();
            }
        }
        const getResponse = await this.dataSource.get(queryResponse.ids);
        const entities = await this.fetchRelations(getResponse.list), records = await Promise.all(entities.map(this.buildRecord));
        this.loadData(records, append);
        return records;
    }
    async fetchRelations(records) {
        if (!this.relations) {
            return records;
        }
        const promises = [];
        for (const relationName in this.relations) {
            const rel = this.relations[relationName];
            let id;
            for (let i = 0, l = records.length; i < l; i++) {
                id = ObjectUtil.path(records[i], rel.path);
                if (id) {
                    promises.push(rel.dataSource.single(id).then((e) => {
                        if (e) {
                            records[i][relationName] = e;
                        }
                    }));
                }
            }
        }
        return Promise.all(promises).then(() => records);
    }
    reload() {
        this.queryParams.position = 0;
        return super.reload();
    }
    loadNext(append = false) {
        if (!this.queryParams.limit) {
            throw new Error("Limit must be set for pagination");
        }
        this.queryParams.position = this.queryParams.position || 0;
        this.queryParams.position += this.queryParams.limit;
        return this.load(append);
    }
    loadPrevious() {
        if (!this.queryParams.limit) {
            throw new Error("Limit and position must be set!");
        }
        this.queryParams.position = this.queryParams.position || 0;
        this.queryParams.position -= this.queryParams.limit;
        return this.load(false);
    }
    hasNext() {
        return this.hasMore;
    }
    hasPrevious() {
        return this.queryParams.position > 0;
    }
    /**
     * Load more data when this element is scrolled down
     *
     * @param el
     */
    addScrollLoader(el) {
        const onScroll = () => {
            const pixelsLeft = el.scrollHeight - el.scrollTop - el.offsetHeight;
            if (pixelsLeft < 100) {
                if (!this.loading && this.hasNext()) {
                    void this.loadNext(true);
                }
            }
        };
        el.addEventListener("scroll", onScroll, { passive: true });
        // this will fill the empty space on firt load.
        this.on("load", (store, records, append) => {
            // use set timeout otherwise this.loading is still true
            setTimeout(() => {
                onScroll();
            });
        });
        return onScroll;
    }
    patchFilter(ref, filter) {
        var _a;
        const f = (_a = this.getFilter(ref)) !== null && _a !== void 0 ? _a : {};
        return this.setFilter(ref, Object.assign(f, filter));
    }
    setFilter(ref, filter) {
        if (filter === undefined) {
            delete this.filters[ref];
        }
        else {
            this.filters[ref] = filter;
        }
        const conditions = [];
        for (const k in this.filters) {
            conditions.push(this.filters[k]);
        }
        switch (conditions.length) {
            case 0:
                delete this.queryParams.filter;
                break;
            case 1:
                this.queryParams.filter = conditions[0];
                break;
            default:
                this.queryParams.filter = {
                    operator: "AND",
                    conditions: conditions
                };
                break;
        }
        return this;
    }
    clearFilter(...names) {
        names.forEach(n => this.setFilter(n, undefined));
    }
    getFilter(name) {
        return this.filters[name];
    }
}
/**
 * Shorthand function to create a {@see DataSourceStore}
 *
 * @param config
 */
export const datasourcestore = (config) => {
    let f;
    if (config.filters) {
        f = config.filters;
        delete config.filters;
    }
    const store = createComponent(new DataSourceStore(config.dataSource), config);
    if (f) {
        for (const filterName in f) {
            store.setFilter(filterName, f[filterName]);
        }
    }
    return store;
};
//# sourceMappingURL=DataSourceStore.js.map