/**
* Copyright Intermesh
*
* This file is part of Group-Office. You should have received a copy of the
* Group-Office license along with Group-Office. See the file /LICENSE.TXT
*
* If you have questions write an e-mail to info@intermesh.nl
*
* @copyright Copyright Intermesh
* @author {authorName} <{email}>
*/

go.modules.business.catalog.MainPanel = Ext.extend(go.modules.ModulePanel,
{
	id: "catalog",
	title: t("Catalog"),
	layout: 'responsive',
	layoutConfig: {
		triggerWidth: 1000
	},
	frontendConfig: {},

	initComponent: function () {
		this.frontendConfig = go.modules.business.catalog.ModuleConfig.frontendOptions;
		this.createArticleGrid();

		this.ArticleDetail = new go.modules.business.catalog.ArticleDetail({
			region: 'center',
			split: true,
			tbar: [{
					cls: 'go-narrow', //will only show on small devices
					iconCls: "ic-arrow-back",
					tooltip: t("Catalog"),
					handler: function () {
						this.westPanel.show();
						go.Router.goto("catalog");
					},
					scope: this
				}]
		});

		// route to main grid when details resets / is deleted for mobile view
		this.ArticleDetail.on("reset", () => {
			go.Router.goto("catalog");
		});

		this.sidePanel = new Ext.Panel({
			layout: "anchor",
			defaults: {
				anchor: "100%"
			},
			width: dp(300),
			cls: 'go-sidenav',
			region: "west",
			split: true,
			hidden: true,
			autoScroll: true,
			items: [
				this.createBusinessPanel(),
				this.createCategoryPanel(),
				this.createFilterPanel()
			]
		});

		this.westPanel = new Ext.Panel({
			region: "west",
			layout: "responsive",
			stateId: "go-catalog-west",
			split: true,
			width: dp(700),
			narrowWidth: dp(400), //this will only work for panels inside another panel with layout=responsive. Not ideal but at the moment the only way I could make it work
			items: [
				this.ArticleGrid, //first is default in narrow mode
				this.sidePanel
			]
		});

		this.items = [
			this.westPanel //first is default in narrow mode
		];
		if(this.frontendConfig.showDetailPanel) {
			this.items.push(this.ArticleDetail);
		}
		if(this.frontendConfig.showFilter) {
			this.sidePanel.show()
		}

		go.modules.business.catalog.MainPanel.superclass.initComponent.call(this);

		//use viewready so load mask can show
		this.ArticleGrid.on("viewready", this.runModule, this);
	},
	runModule: function () {
		var me = this;

		this.categoryPanel.getRootNode().expand();

		this.documentBusinessGrid.store.load().then(function() {
			me.documentBusinessGrid.getSelectionModel().selectRow(0);
		});
	},

	createCategoryPanel : function() {
		this.categoryPanel = new go.modules.business.catalog.CategoryTree({
			ddGroup: "articles",
			ddAppendOnly: true,
			enableDrop: true,
			enableDrag: true,
			tbar: [
				{
					xtype: "tbtitle",
					text: t("Categories")
				},
				"->",
				{
					xtype: "button",
					iconCls: "ic-add",
					handler: function() {
						const d = new go.modules.business.catalog.CategoryDialog();
						d.show();
					}
				}
			],
		});


		this.categoryPanel.selModel.on("selectionchange",  (sm, nodes) => {
			this.ArticleGrid.store.setFilter("categories", {categories: nodes.map(n => n.attributes.data.id)});
			this.ArticleGrid.store.load();
		});


		this.categoryPanel.on("beforenodedrop", this.onCategoryDrop, this);

		return this.categoryPanel;
	},


	onCategoryDrop: function (e) {
		const updates = {};

		if(e.source.dragData.node) {
			return this.onCategoryCategoryDrop(e);
		}

		go.Db.store("Article").get(e.source.dragData.selections.map(function (r) {
			return r.id
		})).then(function (result) {
			result.entities.forEach(function (article) {

				if(article.categories.indexOf(e.target.attributes.data.id) > -1) {
					return;
				}
				article.categories.push(e.target.attributes.data.id);
				updates[article.id] = {categories: article.categories};
			});

			go.Db.store("Article").set({
				update: updates
			}).then(function (response) {
				if (!go.util.empty(response.notUpdated)) {
					Ext.MessageBox.alert(t("Error"), t("Failed to add articles to the category"));
				}
			})
		});

	},

	onCategoryCategoryDrop: function(e) {
		go.Db.store("CatalogCategory").single(e.source.dragData.node.attributes.id).then(function (sourceCategory) {


				updates[sourceCategory.id] = {parentId: target.attributes.data.ids};


			go.Db.store("CatalogCategory").set({
				update: updates
			}).then(function (response) {
				if (!go.util.empty(response.notUpdated)) {
					Ext.MessageBox.alert(t("Error"), t("Failed to add category to the category"));
				}
			})
		});
	},

	createBusinessPanel : function() {
		this.documentBusinessGrid = new go.modules.business.catalog.BusinessGrid ({
			region: "north",
			autoHeight: true,
			title: t("Business")
		});
		this.documentBusinessGrid.getSelectionModel().on('selectionchange', this.onFinanceBusinessSelectionChange, this, {buffer: 1}); //add buffer because it clears selection first

		return this.documentBusinessGrid;
	},

	onFinanceBusinessSelectionChange : function (sm) {

		var selected = sm.getSelected();
		if(!selected) {
			Ext.MessageBox.alert(t("No business", "There are no businesses defined yet in the system settings"));
			this.setDisabled(true);
		}

		go.modules.business.finance.business = selected.data;

		this.businessId = selected.id;
		this.addButton.setDisabled(!this.businessId);
		this.ArticleGrid.store.setFilter("business", {businessId: this.businessId});
		this.ArticleGrid.store.load();
	},

	createFilterPanel: function () {
		return new Ext.Panel({
			region: "center",
			minHeight: dp(200),
			autoScroll: true,
			tbar: [{
				xtype: 'tbtitle',
				text: t("Filters")
			},
			'->',
			{
				xtype: 'filteraddbutton',
				entity: 'Article'
			},{
				cls: 'go-narrow',
				iconCls: "ic-arrow-forward",
				tooltip: t("Catalog"),
				handler: function () {
					this.ArticleGrid.show();
					go.Router.goto("catalog");
				},
				scope: this
			}],
			items: [
				this.filterGrid = new go.filter.FilterGrid({
					filterStore: this.ArticleGrid.store,
					entity: "Article"
				}),{
					xtype: 'variablefilterpanel',
					filterStore: this.ArticleGrid.store,
					entity: "Article"
				}
			]
		});
	},
	createArticleGrid: function () {
		this.ArticleGrid = new go.modules.business.catalog.ArticleGrid({
		region: 'center',
		enableDragDrop: true, //for dragging contacts to address books or groups in the tree
		ddGroup: "articles",
		multiSelectToolbarItems: [
			{
				hidden: go.customfields.CustomFields.getFieldSets('Article').length == 0,
				iconCls: 'ic-edit',
				tooltip: t("Batch edit"),
				handler: function() {
					var dlg = new go.form.BatchEditDialog({
						entityStore: "Article"
					});
					dlg.setIds(this.ArticleGrid.getSelectionModel().getSelections().column('id')).show();
				},
				scope: this
			}
		],
		tbar: [{
			cls: 'go-narrow', //Shows on mobile only
			iconCls: "ic-menu",
			handler: function () {
			this.sidePanel.show();
		},
		scope: this
	    },
	    '->',
	    {
		    xtype: 'tbsearch',
		    hidden: !this.frontendConfig.showSearchBar
		},
		this.addButton = new Ext.Button({
			iconCls: 'ic-add',
			tooltip: t('Add'),
			cls: "primary",
			handler: function (btn) {
				var ArticleForm = new go.modules.business.catalog.ArticleDialog();
				ArticleForm.setValues({
					businessId: this.businessId
				})
				ArticleForm.show();
			},
			scope: this
	    }),
	    this.moreBtn = new Ext.Button({
				iconCls: 'ic-more-vert',
				menu: []
			})],
			listeners: {
				rowdblclick: this.onGridDblClick,
				scope: this,
				keypress: this.onGridKeyPress
			}
		});

		this.ArticleGrid.on('navigate', function (grid, rowIndex, record) {
			go.Router.goto("Article/" + record.id);
		}, this);

		this.moreBtn.menu.add({
			iconCls: 'ic-cloud-upload',
			text: t("Import"),
			handler: function() {
				go.util.importFile(
					'Article',
					'.csv,.xlsx',
					{businessId: this.businessId},
					{
						// These fields can be selected to update contacts if ID or e-mail matches
						lookupFields: {'id' : "ID", 'number': t("Number")}
					}
				);
			},
			scope: this
		},{
			iconCls: 'ic-cloud-download',
			text: t("Export"),

			menu: [
				{
					text: 'Microsoft Excel',
					iconCls: 'filetype filetype-xls',
					handler: function() {
						go.util.exportToFile(
							'Article',
							Object.assign(go.util.clone(this.ArticleGrid.store.baseParams), this.ArticleGrid.store.lastOptions.params, {limit: 0, position: 0}),
							'xlsx');
					},
					scope: this
				},{
					text: 'Comma Separated Values',
					iconCls: 'filetype filetype-csv',
					handler: function() {
						go.util.exportToFile(
							'Article',
							Object.assign(go.util.clone(this.ArticleGrid.store.baseParams), this.ArticleGrid.store.lastOptions.params, {limit: 0, position: 0}),
							'csv');
					},
					scope: this
				},{
					text: t("Web page") + " (HTML)",
					iconCls: 'filetype filetype-html',
					handler: function() {
						go.util.exportToFile(
							'Article',
							Object.assign(go.util.clone(this.ArticleGrid.store.baseParams), this.ArticleGrid.store.lastOptions.params, {limit: 0, position: 0}),
							'html');
					},
					scope: this
				},
				{
					iconCls: 'filetype filetype-json',
					text: 'JSON',
					handler: function() {
						go.util.exportToFile(
							'Article',
							Object.assign(go.util.clone(this.ArticleGrid.store.baseParams), this.ArticleGrid.store.lastOptions.params, {limit: 0, position: 0}),
							'json');
					},
					scope: this
				},
			],
			scope: this
		});


			return this.ArticleGrid;
		},
		onGridDblClick: function (grid, rowIndex, e) {
			var record = grid.getStore().getAt(rowIndex);
			if (record.get('permissionLevel') < go.permissionLevels.write) {
				return;
			}

		    var dlg = new go.modules.business.catalog.ArticleDialog();
		    dlg.load(record.id).show();
		},

		onGridKeyPress: function (e) {
		if (e.keyCode != e.ENTER) {
			return;
		}
		var record = this.ArticleGrid.getSelectionModel().getSelected();
		if (!record) {
			return;
		}

		if (record.get('permissionLevel') < go.permissionLevels.write) {
			return;
		}

		var dlg = new go.modules.business.catalog.ArticleDialog();
		dlg.load(record.id).show();
	}
});
