/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Component, ComponentEventMap } from "./Component.js";
import { Config, Listener, ObservableListenerOpts } from "./Observable.js";
/**
 * Data available to draggable item listeners
 */
interface DragData {
    /**
     * x when dragging started
     */
    startX: number;
    /**
     * y when dragging started
     */
    startY: number;
    /**
     * Current x
     */
    x: number;
    /**
     * Current y
     */
    y: number;
    /**
     * offsetLeft of dragged element when dragging started
     */
    startOffsetLeft: number;
    /**
     * offsetTop of dragged element when dragging started
     */
    startOffsetTop: number;
    /**
     * The left offset from the element offsetLeft where the user grabbed the element
     */
    grabOffsetLeft: number;
    /**
     * The top offset from the element offsetLeft where the user grabbed the element
     */
    grabOffsetTop: number;
    data: any;
}
interface ConstrainBox {
    left: number;
    right: number;
    bottom: number;
    top: number;
}
/**
 * @inheritDoc
 */
export interface DraggableComponentEventMap<Type> extends ComponentEventMap<Type> {
    /**
     * Fires when the component is dropped
     *
     * @param comp
     */
    drop: (comp: Type, dragData: DragData, e: MouseEvent) => void;
    /**
     * Fires contanty while the component is being dragged
     * @param comp
     * @param dragData
     * @param e
     */
    drag: (comp: Type, dragData: DragData, e: MouseEvent) => void;
    /**
     * Return false to prevent drag
     *
     * @param comp
     * @param e
     */
    dragstart: (comp: Type, dragData: DragData, e: MouseEvent) => false | void;
}
export interface DraggableComponent {
    on<K extends keyof DraggableComponentEventMap<this>, L extends Listener>(eventName: K, listener: Partial<DraggableComponentEventMap<this>>[K], options?: ObservableListenerOpts): L;
    fire<K extends keyof DraggableComponentEventMap<this>>(eventName: K, ...args: Parameters<DraggableComponentEventMap<any>[K]>): boolean;
}
export declare class DraggableComponent extends Component {
    protected dragData?: DragData;
    private constrainBox?;
    private _dragConstrainTo;
    private _dragConstrainPad?;
    /**
     * Update left and top css properties when dragging
     */
    setPosition?: boolean;
    /**
     * Enable dragging
     */
    private _draggable;
    constructor(tagName?: keyof HTMLElementTagNameMap);
    /**
     * Enable or disable dragging
     */
    set draggable(draggable: boolean);
    private toggleDraggable;
    get draggable(): boolean;
    private onDragHandleClick;
    private onDragHandleMouseDown;
    /**
     * Returns the DOM element that can be grabbed to drag the component
     * @protected
     */
    protected getDragHandle(): HTMLElement;
    /**
     * Constrain dragging to this element
     * @param el
     * @param pad Supply paddings
     */
    dragConstrainTo(el: HTMLElement | Window, pad?: Partial<ConstrainBox>): void;
    private calcConstrainBox;
    private onDragStart;
    private onDrag;
    private constrainCoords;
    /**
     * Constrain the component to the given element
     *
     * @param el
     * @param pad
     */
    constrainTo(el: HTMLElement | Window, pad?: Partial<ConstrainBox>): void;
}
/**
 * Shorthand function to create {@see DraggableComponent}
 *
 * @param config
 * @param items
 */
export declare const draggable: (config?: Config<DraggableComponent, DraggableComponentEventMap<DraggableComponent>>, ...items: Component[]) => DraggableComponent;
export {};
//# sourceMappingURL=DraggableComponent.d.ts.map