/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Michael de Hart <mdhart@intermesh.nl>
 */
import { E } from "../../util/Element.js";
import { Component, createComponent } from "../Component.js";
import { Field } from "./Field.js";
/**
 * Radio field
 *
 * @example
 * ```
 * radio({
 * 		type: "button",
 * 		value: "option1",
 * 	  name: "radio",
 * 		options: [
 * 			{text: "Option 1", value: "option1"},
 * 			{text: "Option 2", value: "option2"},
 * 			{text: "Option 3", value: "option3"}
 * 		]}
 * 	)
 * ```
 */
export class RadioField extends Field {
    constructor(type = 'box') {
        super('div');
        this.inputs = {};
        this.baseCls = 'goui-form-field radiogroup';
        this.type = type;
        this.el.cls(type, true);
    }
    createLabel() {
        return;
    }
    createControl() {
        if (!this.name && !this.itemId) {
            this.itemId = "radio-" + Component.uniqueID();
        }
        this._labelEl = E("h3").cls("legend");
        this._labelEl.innerText = this.label;
        this.el.insertBefore(this._labelEl, this.wrap);
        const radio = E('div').cls('radio');
        if (this.invalidMsg) {
            this.applyInvalidMsg();
        }
        return radio;
    }
    set options(options) {
        if (this._options) {
            this.control.empty();
            this.inputs = {};
        }
        options.forEach((o) => {
            const btn = E('input').on("change", () => {
                this.fireChange();
            });
            btn.type = "radio";
            btn.name = this.name || (this.itemId + "");
            btn.readOnly = this.readOnly;
            if (!o.value) {
                o.value = "";
            }
            btn.value = o.value;
            if (this._value == o.value) {
                btn.checked = true;
            }
            this.inputs[o.value] = btn;
            const lbl = E('span').cls('box-label');
            if (o.icon) {
                lbl.append(E("i", o.icon).cls("icon"));
            }
            lbl.append(o.text);
            this.control.append(E('label', btn, lbl).cls('control'));
        });
    }
    get options() {
        var _a;
        return (_a = this._options) !== null && _a !== void 0 ? _a : [];
    }
    internalGetValue() {
        if (this.rendered) {
            for (let v in this.inputs) {
                if (this.inputs[v].checked) {
                    return v;
                }
            }
        }
        return this._value;
    }
    internalSetValue(v) {
        if (v && v in this.inputs)
            this.inputs[v].checked = true;
    }
}
/**
 * Shorthand function to create {@see RadioField}
 *
 * @param config
 */
export const radio = (config) => createComponent(new RadioField((config === null || config === void 0 ? void 0 : config.type) || 'box'), config);
//# sourceMappingURL=RadioField.js.map