<?php

namespace go\modules\business\finance;

use Exception;
use Faker\Generator;
use go\core;
use go\core\cron\GarbageCollection;
use go\core\fs\File;
use go\core\model;
use go\core\model\Link;
use go\core\orm\exception\SaveException;
use go\core\orm\Mapping;
use go\core\util\ClassFinder;
use go\modules\business\business\model\Business;
use go\modules\business\catalog\model\Article;
use go\modules\business\finance\model\ContactBusiness;
use go\modules\business\finance\model\Debtor;
use go\modules\business\finance\model\FinanceBook;
use go\modules\business\finance\model\FinanceBusiness;
use go\modules\business\finance\model\FinanceDocument;
use go\modules\business\finance\model\FinanceDocumentItem;
use go\modules\business\finance\model\FinanceDocumentItemGroup;
use go\modules\business\finance\model\Payment;
use go\modules\business\finance\model\PaymentProviderInterface;
use go\modules\business\finance\model\RecurringPaymentProviderInterface;
use go\modules\business\finance\model\Settings;
use go\modules\business\finance\model\UblInvoice;
use go\modules\business\finance\model\ZUGFeRDInvoice;
use go\modules\community\addressbook\model\Contact;


/**
 * @copyright (c) 2020, Intermesh BV https://www.intermesh.nl
 * @author System Administrat <admin@intermesh.localhost>
 * @license http://www.gnu.org/licenses/agpl-3.0.html AGPLv3
 */
class Module extends core\Module
{
	use core\event\EventEmitterTrait;

	const EVENT_RENDER_DOCUMENT_TOOLBAR = "renderdocumenttoolbar";

	public function getAuthor() : string
	{
		return "Intermesh BV <info@intermesh.nl>";
	}

    /**
     * The development status of this module
     * @return string
     */
    public function getStatus() : string{
        return self::STATUS_STABLE;
    }

	public function autoInstall(): bool
	{
		return true;
	}

	public function getDependencies() : array
	{
		return ['business/business', 'community/addressbook'];
	}

	public function requiredLicense(): ?string
	{
		return "billing";
	}

	/**
	 * @throws SaveException
	 */
	protected function afterInstall(model\Module $model) : bool
	{
		//DebtorMailer::install("0 11 * * *");

		$pos = 0;

		foreach(Business::find() as $business) {
			//for creating defaults in FinanceBusiness
			if(!$business->save()) {
				throw new core\orm\exception\SaveException($business);
			}

			$types = [
				FinanceBook::TYPE_QUOTE,
				FinanceBook::TYPE_SALES_ORDER ,
				FinanceBook::TYPE_SALES_INVOICE,
				FinanceBook::TYPE_PURCHASE_ORDER,
				FinanceBook::TYPE_PURCHASE_INVOICE
			];

			$lang = go()->t("types", "business", "finance");

			foreach($types as $type) {
				$book = new FinanceBook();
				$book->businessId = $business->id;
				$book->name = $lang[$type] . (($pos > 0) ? ' (' . $pos . ')' : '');
				$book->type = $type;

//				if($type == FinanceBook::TYPE_QUOTE) {
//					$book->defaultGreeting = "{{customer.salutation}}";
//					$book->defaultClosing = "Best regards,<br><br>{{creator.displayName}}<br>{{business.name}}<br>";
//				}

				if(!$book->save()) {
					throw new SaveException($book);
				}
			}

			$filter = new model\EntityFilter();
			$filter->type = "variable";
			$filter->name = "date";
			$filter->setEntity("FinanceDocument");
			$filter->setAcl([
				model\Group::ID_EVERYONE => model\Acl::LEVEL_READ
			]);
			if(!$filter->save()) {
				throw new SaveException($filter);
			}
			$pos++;
		}

		return parent::afterInstall($model);
	}




	public function defineListeners() {
		parent::defineListeners();

		Business::on(core\orm\Property::EVENT_MAPPING, static::class, 'onBusinessMap');
		Contact::on(core\orm\Property::EVENT_MAPPING, static::class, 'onContactMap');
		Link::on(Link::EVENT_SAVE, static::class, 'onLinkSave');
		GarbageCollection::on(GarbageCollection::EVENT_RUN, static::class, 'onGarbageCollection');
	}

	public static function onGarbageCollection() {


		$moduleId = static::get()->getModel()->id;

		model\PdfTemplate::delete(
			(new core\orm\Query())
				->where(['moduleId' => $moduleId])
				->where('`key` LIKE "fb-def-%"')
				->andWhere('SUBSTRING(`key`, 8)  NOT IN (SELECT id FROM business_finance_book)')
		);

		model\PdfTemplate::delete(
			(new core\orm\Query())
				->where(['moduleId' => $moduleId])
				->where('`key` LIKE "fb-xtr-%"')
				->andWhere('SUBSTRING(`key`, 8)  NOT IN (SELECT id FROM business_finance_book)')
		);

		model\PdfTemplate::delete(
			(new core\orm\Query())
				->where(['moduleId' => $moduleId])
				->where('`key` LIKE "statement-%"')
				->andWhere('SUBSTRING(`key`, 11)  NOT IN (SELECT id FROM business_business)')
		);

		//cleanup templates from DebtorProfileAction
		model\EmailTemplate::delete(
			(new core\orm\Query())
				->where(['moduleId' => $moduleId])
				->where('`key` LIKE "debtor-action-%"')
				->andWhere('SUBSTRING(`key`, 15) NOT IN (SELECT id FROM business_finance_business_debtor_profile_action)')
		);

		model\EmailTemplate::delete(
			(new core\orm\Query())
				->where(['moduleId' => $moduleId])
				->where('`key` LIKE "fb-def-%"')
				->andWhere('SUBSTRING(`key`, 8)  NOT IN (SELECT id FROM business_finance_book)')
		);

		model\EmailTemplate::delete(
			(new core\orm\Query())
				->where(['moduleId' => $moduleId])
				->where('`key` LIKE "fb-xtr-%"')
				->andWhere('SUBSTRING(`key`, 8)  NOT IN (SELECT id FROM business_finance_book)')
		);
	}

	/**
	 * Because we've implemented the getter method "getorganizationIds" the contact
	 * modSeq must be incremented when a link between two contacts is deleted or
	 * created.
	 *
	 * @param Link $link
	 * @throws Exception
	 */
	public static function onLinkSave(Link $link) {
		if(!$link->isBetween("LinkedEmail", "FinanceDocument") && !$link->isBetween("FinanceDocument", "FinanceDocument") ) {
			return;
		}

		$requiredProps = ['id', 'sentAt', 'businessId', 'type', 'completedAt', 'customerId', 'number', 'date', 'cancelled', 'itemGroups'];

		if($link->getToEntity() == "FinanceDocument") {
			$fd = FinanceDocument::findById($link->toId, $requiredProps);
		} else{
			$fd = FinanceDocument::findById($link->fromId, $requiredProps);
		}

		if($link->isBetween("LinkedEmail", "FinanceDocument")) {
			if ($fd->sentAt == null) {
				$fd->sentAt = new core\util\DateTime();
			}
		} else {

			switch($fd->findBook()->type) {

				case FinanceBook::TYPE_QUOTE:
				case FinanceBook::TYPE_SALES_ORDER:
					$fd2 = FinanceDocument::findById($link->fromId,$requiredProps);

					switch($fd2->findBook()->type) {
						case  FinanceBook::TYPE_PURCHASE_ORDER:
							//this link will affect the getPurchasedAmount() prop so trigger a change
							$fd2->change(true);
						break;

						case FinanceBook::TYPE_SALES_ORDER:

							if($fd->findOrderedAmount() >= $fd->getTotalPrice()) {
								$fd->completedAt = new \DateTime();
								$fd->cancelled = false;
								$fd->save();
							}

							break;

						case FinanceBook::TYPE_SALES_INVOICE:
							if($fd->findInvoicedAmount() >= $fd->getTotalPrice()) {
								$fd->completedAt = new \DateTime();
								$fd->cancelled = false;
								$fd->save();
							}
							break;
					}

					break;

				case FinanceBook::TYPE_PURCHASE_ORDER:
					$fd2 = FinanceDocument::findById($link->fromId, $requiredProps);

					if($fd2->findBook()->type == FinanceBook::TYPE_PURCHASE_INVOICE){
						//this link will affect the getPurchasedAmount() prop so trigger a change
						$fd2->change(true);

						//save will check status
						$fd->save();
					}
					break;


			}
		}


		if($fd->isModified() && !$fd->save()) {
			go()->error("Could not save FinanceDocument after sent email id: ". $fd->id);
		}

	}


	public static function onBusinessMap(Mapping $mapping) {
		$mapping->addHasOne('finance', FinanceBusiness::class, ['id' => 'businessId'], true);
	}

	public static function onContactMap(Mapping $mapping) {
		$mapping->addMap('businesses', ContactBusiness::class, ['id' => 'contactId']);
	}

	public function downloadStatement($contactId, $businessId)
	{
		$debtor = Debtor::findById($businessId .'-'.$contactId);
		$customer =  $debtor->findCustomer();

		$filename = $customer->name . '-statement.pdf';

		$debtor->toPdf()->render()->Output($filename, "I");
	}


	public function downloadPdf($documentId, $inline = 1)
	{
		$document = FinanceDocument::findById($documentId);
		if (!$document->getPermissionLevel()) {
			throw new core\exception\Forbidden();
		}

		$disp = $inline ? 'inline' : 'attachment';

		$pdf = $document->toPdfString();
		header("Content-Type: application/pdf");
		header("Content-Disposition: ".$disp."; filename=". File::stripInvalidChars($document->title(), '-') . '.pdf');

		echo $pdf;
	}

	public function pagePdf($documentId, $token, $inline = 1) {

		$document = FinanceDocument::findById($documentId);
		if ($document->token != $token) {
			throw new core\exception\Forbidden();
		} else{
			go()->setAuthState(new core\auth\TemporaryState($document->createdBy));
		}
		$disp = $inline ? 'inline' : 'attachment';

		$pdf = $document->toPdfString();

		header('Cache-Control: no-cache, must-revalidate, post-check=0, pre-check=0'); //prevent caching
		header('Expires: Mon, 26 Jul 1997 05:00:00 GMT'); //resolves problem with IE GET requests
		header('Pragma: no-cache');
		header("Content-Type: application/pdf");
		header("Content-Disposition: ".$disp."; filename=". File::stripInvalidChars($document->title(), '-') . '.pdf');

		echo $pdf;
	}

	public function pageZugferd($documentId, $token, $inline = 1) {

		$document = FinanceDocument::findById($documentId);
		if ($document->token != $token) {
			throw new core\exception\Forbidden();
		} else{
			go()->setAuthState(new core\auth\TemporaryState($document->createdBy));
		}
		header('Cache-Control: no-cache, must-revalidate, post-check=0, pre-check=0'); //prevent caching
		header('Expires: Mon, 26 Jul 1997 05:00:00 GMT'); //resolves problem with IE GET requests
		header('Pragma: no-cache');
		header("Content-Type: application/xml");
		header("Content-Disposition: inline; filename=". File::stripInvalidChars($document->title(), '-') . '-cii.xml');
		$zugferd = new ZUGFeRDInvoice($document);
		echo $zugferd->getXML();
	}

	public function pageUbl($documentId, $token, $inline = 1) {

		$document = FinanceDocument::findById($documentId);
		if ($document->token != $token) {
			throw new core\exception\Forbidden();
		} else{
			go()->setAuthState(new core\auth\TemporaryState($document->createdBy));
		}
		header('Cache-Control: no-cache, must-revalidate, post-check=0, pre-check=0'); //prevent caching
		header('Expires: Mon, 26 Jul 1997 05:00:00 GMT'); //resolves problem with IE GET requests
		header('Pragma: no-cache');
		header("Content-Type: application/xml");
		header("Content-Disposition: inline; filename=". File::stripInvalidChars($document->title(), '-') . '-ubl.xml');
		$zugferd = new UblInvoice($document);
		echo $zugferd->getXML();
	}


	public function pageDoc($documentId, $token) {

		$document = FinanceDocument::findById($documentId);
		if ($document->token != $token) {
			throw new core\exception\Forbidden();
		} else{
			go()->setAuthState(new core\auth\TemporaryState($document->createdBy));
		}

		require(__DIR__ . '/views/web/financeDocument.php');
	}

	public static $bookTypes = [
		'salesinvoice' => 'SI',
		'salesorder' => 'SO',
		'quote' => 'Q',
		'purchaseorder' => 'PO',
		'purchaseinvoice' => 'PI',
		'contract' => 'C'
	];


	/**
	 * Takes an int and number format and returns the next sequence number string
	 *
	 * @param int $sequenceNumber
	 * @param string $format eg. Q%Y-00000;
	 * @param string $type
	 * @return string
	 */
	public static function createNextSequence(int $sequenceNumber, string $format, string $type): string
	{
		$zeros = preg_match_all('/0+/', $format, $matches);


		$number = str_replace(
			['%y','%Y', '%m', '%d', '%T'],
			[date('y'), date('Y'), date('m'), date('d'), self::$bookTypes[$type]], $format);

		/** @noinspection DuplicatedCode */
		$match = array_pop($matches[0]);

		if(!$zeros) {
			return $number.$sequenceNumber;
		}else
		{
			//build 00001 for example
			$replacement = '';
			$zeroRepeat = strlen($match) - strlen($sequenceNumber);
			if ($zeroRepeat > 0) {
				$replacement = str_repeat('0', $zeroRepeat);
			}
			$replacement .= $sequenceNumber;

			return str_replace($match, $replacement, $number);
		}
	}

	public function demo(Generator $faker)
	{
		\go\modules\business\catalog\Module::get()->demo($faker);

		$articles = Article::find()
			->selectSingleValue('id')
			->limit(0)
			->offset(10)
			->orderBy(['id' => 'DESC'])
			->all();

		$months = 36;

		foreach (FinanceBook::find() as $book) {
			for($month = -$months; $month <0; $month++) {
				for ($n = 0; $n < 2; $n++) {
					$doc = new FinanceDocument();
					$doc->createNumber = true;
					$doc->bookId = $book->id;
					$doc->date = $faker->dateTimeBetween($month. " months", ($month +1) ." months");
					$doc->setContactId(\go\modules\community\addressbook\Module::get()->demoContact($faker)->id);
					$doc->setCustomerId(\go\modules\community\addressbook\Module::get()->demoCompany($faker)->id);

					$itemGroup = new FinanceDocumentItemGroup($doc);

					for ($i = 0; $i < $faker->numberBetween(1, 10); $i++) {
						$item = FinanceDocumentItem::fromArticleId($doc, $articles[$faker->numberBetween(0, 9)]);
						$item->quantity = $faker->numberBetween(1, 10);
						$itemGroup->items[] = $item;
					}

					$doc->itemGroups[] = $itemGroup;

					if (!$doc->save()) {
						throw new SaveException($doc);
					}

					if ($faker->boolean) {
						$payment = new Payment();
						$payment->businessId = $book->businessId;
						$payment->documentId = $doc->id;
						$payment->customerId = $doc->getCustomerId();
						$payment->amount = $doc->getTotalPrice();
						$payment->description = $doc->number;
						$payment->date = (clone $doc->date)->add(new \DateInterval("P14D"));
						if (!$payment->save()) {
							throw new SaveException($payment);
						}
					}
				}
			}
		}
	}


	/**
	 * @return PaymentProviderInterface[]
	 */
	public function findPaymentProviders():array {
		$cache = go()->getCache()->get("finance-payment-providers");
		if(isset($cache)) {
			return $cache;
		}

		$classFinder = new ClassFinder();
		$classes = $classFinder->findByParent(PaymentProviderInterface::class);
		go()->getCache()->set("finance-payment-providers", $classes);

		return $classes;
	}

	/**
	 * @return RecurringPaymentProviderInterface[]
	 */
	public function findRecurringPaymentProviders():array {
		$cache = go()->getCache()->get("finance-recurring-payment-providers");
		if(isset($cache)) {
			return $cache;
		}

		$classFinder = new ClassFinder();
		$classes = $classFinder->findByParent(RecurringPaymentProviderInterface::class);
		go()->getCache()->set("finance-recurring-payment-providers", $classes);

		return $classes;
	}

	public function getSettings()
	{
		return Settings::get();
	}

}
