/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Store, StoreConfig, StoreEventMap, StoreRecord } from "../data/Store.js";
import { AbstractDataSource, BaseEntity, Changes, dataSourceEntityType, DefaultEntity, QueryParams } from "./AbstractDataSource.js";
import { ObservableListener } from "../component/index.js";
type Relation<EntityType extends BaseEntity> = Partial<Record<keyof EntityType, {
    dataSource: AbstractDataSource<EntityType>;
    path: string;
}>>;
type RecordBuilder<EntityType, StoreRecord> = (entity: EntityType) => Promise<StoreRecord>;
/**
 * DataSourceStore class
 *
 * Uses an {@see AbstractDataSource} to present data in the view.
 * @category Data
 */
export declare class DataSourceStore<DataSource extends AbstractDataSource = AbstractDataSource, StoreRecord = dataSourceEntityType<DataSource>> extends Store<StoreRecord> {
    dataSource: DataSource;
    queryParams: Omit<QueryParams, "sort">;
    hasMore: boolean;
    relations?: Relation<DefaultEntity>;
    /**
     * Reload when the datasource changes
     */
    monitorChanges: boolean;
    /**
     * Builds record from entity
     * @param entity
     */
    buildRecord: RecordBuilder<dataSourceEntityType<DataSource>, StoreRecord>;
    constructor(dataSource: DataSource);
    /**
     * Reloads the store when the datasource changes
     *
     * @protected
     */
    protected onChange(DataSource: DataSource, changes: Changes): Promise<void>;
    protected internalLoad(append?: boolean): Promise<Awaited<StoreRecord>[]>;
    private fetchRelations;
    reload(): Promise<StoreRecord[]>;
    loadNext(append?: boolean): Promise<StoreRecord[]>;
    loadPrevious(): Promise<StoreRecord[]>;
    hasNext(): boolean;
    hasPrevious(): boolean;
    /**
     * Load more data when this element is scrolled down
     *
     * @param el
     */
    addScrollLoader(el: HTMLElement): () => void;
    patchFilter(ref: string, filter: any | undefined): this;
    private filters;
    setFilter(ref: string, filter: Record<string, any> | undefined): this;
    clearFilter(...names: string[]): void;
    getFilter(name: string): any;
}
export type DataSourceStoreConfig<DataSource extends AbstractDataSource, RecordType extends StoreRecord> = Omit<StoreConfig<RecordType>, "listeners"> & {
    queryParams?: QueryParams;
    dataSource: DataSource;
    buildRecord?: RecordBuilder<dataSourceEntityType<DataSource>, RecordType>;
    relations?: Relation<DefaultEntity>;
    listeners?: ObservableListener<StoreEventMap<DataSourceStore<DataSource, RecordType>, RecordType>>;
    filters?: Record<string, Record<string, any>>;
};
/**
 * Shorthand function to create a {@see DataSourceStore}
 *
 * @param config
 */
export declare const datasourcestore: <DataSource extends AbstractDataSource = AbstractDataSource<DefaultEntity>, RecordType extends StoreRecord = dataSourceEntityType<DataSource>>(config: DataSourceStoreConfig<DataSource, RecordType>) => DataSourceStore<DataSource, RecordType>;
export {};
//# sourceMappingURL=DataSourceStore.d.ts.map