<?php

namespace go\modules\business\studio;

use Exception;
use go\core;
use go\core\model\Acl;
use go\core\model\Module as CoreModuleModel;
use go\core\orm\Entity;
use go\modules\business\studio\model\Settings;
use go\modules\business\studio\model\Studio;
use PDO;

/**
 * @copyright (c) 2020, Intermesh BV http://www.intermesh.nl
 * @author Merijn Schering <mschering@intermesh.nl>
 * @license http://www.gnu.org/licenses/agpl-3.0.html AGPLv3
 */
class Module extends core\Module
{
	/**
	 * The development status of this module
	 * @return string
	 */
	public function getStatus() : string{
		return self::STATUS_STABLE;
	}

	public function getAuthor(): string
	{
		return "Joachim van de Haterd - Intermesh BV <jvdhaterd@intermesh.nl>";
	}

	public function requiredLicense(): ?string
	{
		return 'groupoffice-pro';
	}

	public function getSettings()
	{
		return Settings::get();
	}

	public function defineListeners()
	{
		parent::defineListeners();

		CoreModuleModel::on(Entity::EVENT_SAVE, static::class, 'onModuleSave');
	}

	public static function onModuleSave(CoreModuleModel $module) {
		if($module->isModified(['permissions'])) {

			$studioModel = Studio::find(['id','moduleAclId'])->where(['name' => $module->package .'-'.$module->name ])->single();
			if(!$studioModel) {
				return;
			}

			$acl = Acl::findById($studioModel->moduleAclId);
			$acl->groups = [];
			foreach ($module->permissions as $groupId => $p) {
				$acl->addGroup($groupId, !empty($p->getRights()->mayManage) ? Acl::LEVEL_MANAGE : Acl::LEVEL_READ);
			}
			$acl->save();
		}
	}

	/**
	 * Patches the code to comply with 6.6 standards
	 *
	 */
	public static function patch65to66()
	{
		$cli = PHP_SAPI=='cli';
		$conn = go()->getDbConnection();
		$stmt = $conn->query("select m.name,m.package from core_module m inner join studio_studio s on s.moduleId = m.id");

		try {
			while ($mod = $stmt->fetch(PDO::FETCH_ASSOC)) {
				if($cli) {
					echo 'Attempting to patch package ' . $mod['package'] . ', module ' . $mod['name'] . '...'. PHP_EOL;
				}
				$folder = go()->getEnvironment()->getInstallFolder()->getFolder("go/modules/" . $mod['package'] . "/" . $mod['name']);
				if (!$folder->exists()) {
					if($cli) {
						echo 'Module folder does not exist. No further action needed.'. PHP_EOL;
					}
					continue;
				}
				if($cli) {
					echo 'Updating module class.' . PHP_EOL;
				}

				$moduleFile = $folder->getFile("Module.php");
				$replaced = str_replace(
					["public function getAuthor()\n"],
					["public function getAuthor(): string\n"],
					$moduleFile->getContents()
				, $count);
				if($count) {
					$moduleFile->putContents(
						$replaced
					);
				}

				if($cli) {
					echo 'Updating model classes.' . PHP_EOL;
				}

				$modelFiles = $folder->getFolder("model")->getFiles();


				foreach ($modelFiles as $file) {
					$replaced = str_replace(
						[
							"protected static function defineMapping()\n",
							"public function title()\n",
							"public function getPermissionLevel()\n",
							"protected function canCreate()\n",
							"protected static function defineFilters()\n",
							"public static function converters() {\n",
							"protected function getSearchDescription()\n",
							"public static function converters()\n"
						],
						[
							"protected static function defineMapping(): \go\core\orm\Mapping\n",
							"public function title(): string\n",
							"public function getPermissionLevel(): int\n",
							"protected function canCreate(): bool\n",
							"protected static function defineFilters(): \go\core\orm\Filters\n",
							"public static function converters():array\n\t{\n",
							"protected function getSearchDescription(): string \n",
							"public static function converters(): array\n"

						],
						$file->getContents(), $count);
					if($count) {
						$file->putContents($replaced);
					}
				}

				if($cli) {
					echo 'Updating controller classes.' . PHP_EOL;
				}
				$ctrlFiles = $folder->getFolder("controller")->getFiles();

				foreach ($ctrlFiles as $ctrl) {
					$replaced = str_replace(
						[
							"protected function entityClass() {\n",

						],
						[
							"protected function entityClass():string {\n",
						],
						$ctrl->getContents(), $count);
					if($count) {
						$ctrl->putContents($replaced);
					}
				}

				if($cli) {
					echo 'Attempt successful!' . PHP_EOL;
				}
			}
		} catch( Exception $e) {
			return "WARNING: Could not patch studio modules. Please make the folder writable or fix the code manually. Exception: " . $e->getMessage();

		}
	}


	/**
	 * Patches the code to comply with 6.6 standards
	 *
	 */
	public static function patch66to67()
	{
		$cli = PHP_SAPI=='cli';
		$conn = go()->getDbConnection();
		$stmt = $conn->query("select m.name,m.package from core_module m inner join studio_studio s on s.moduleId = m.id");

		try {
			while ($mod = $stmt->fetch(PDO::FETCH_ASSOC)) {
				if ($cli) {
					echo 'Attempting to patch package ' . $mod['package'] . ', module ' . $mod['name'] . '...' . PHP_EOL;
				}
				$folder = go()->getEnvironment()->getInstallFolder()->getFolder("go/modules/" . $mod['package'] . "/" . $mod['name']);
				if (!$folder->exists()) {
					if ($cli) {
						echo 'Module folder does not exist. No further action needed.' . PHP_EOL;
					}
					continue;
				}


				if($cli) {
					echo 'Updating controller classes.' . PHP_EOL;
				}
				$ctrlFiles = $folder->getFolder("controller")->getFiles();

				foreach ($ctrlFiles as $ctrl) {
					$replaced = str_replace(
						[
							"use go\core\jmap\Entity;\n",
						    "* @return array",
							": array"
						],
						[
							"use go\core\util\ArrayObject;\n",
							"* @return ArrayObject",
							": ArrayObject"
						],
						$ctrl->getContents(), $count);
					if($count) {
						$ctrl->putContents($replaced);
					}
				}

				if($cli) {
					echo 'Attempt successful!' . PHP_EOL;
				}

			}
		} catch( Exception $e) {
			return "WARNING: Could not patch studio modules to the 6.7 standard. Please make the folder writable or fix the code manually. Exception: " . $e->getMessage();
		}
	}

}