<?php

namespace GO\Projects2;

use Faker\Generator;
use GO;
use go\core\db\Criteria;
use go\core\model\EntityFilter;
use go\core\orm\Filters;
use go\core\orm\Mapping;
use go\core\orm\Query;
use go\modules\community\addressbook\model\Contact;
use go\modules\community\addressbook\Module;
use go\modules\community\tasks\model\TaskList;
use GO\Projects2\Model\Status;
use GO\Projects2\Model\Template;
use GO\Projects2\Model\Type;
use PDO;
use go\core\model\User;
use go\core\model\UserDisplay;
use go\core\orm\Property;
use GO\Projects2\Model\UserSettings;

class Projects2Module extends \GO\Professional\Module
{

	const FINANCE_PERMISSIONS = 45;

	public function appCenter()
	{
		return true;
	}

	public function depends()
	{
		return ["business/business"];
	}

	public function getRights()
	{
		return ['mayFinance' => 1, 'mayManage' => 2, 'mayBudget' => 4];
	}

	/**
	 * Add the project manager notify email job to the cron controller
	 */
	public static function initListeners()
	{

		if (GO::modules()->isInstalled('files') && class_exists('\GO\Files\Controller\FolderController')) {
			$folderController = new \GO\Files\Controller\FolderController();
			$folderController->addListener('checkmodelfolder', "GO\Projects2\Projects2Module", "createParentFolders");
		}
		\GO\Base\Model\User::model()->addListener('delete', "GO\Projects2\Projects2Module", "deleteUser");

	}

	public static function defineListeners()
	{
		// Add contact filter
		Contact::on(Contact::EVENT_FILTER, static::class, 'onContactFilter');
		User::on(Property::EVENT_MAPPING, static::class, 'onMap');
		UserDisplay::on(UserDisplay::EVENT_FILTER, static::class, 'onUserFilter');
		User::on(User::EVENT_FILTER, static::class, 'onUserFilter');
	}

	public static function onMap(Mapping $mapping)
	{
		$mapping->addHasOne('projectsSettings', UserSettings::class, ['id' => 'userId'], true);
		return true;
	}

	public static function onContactFilter(Filters $filters)
	{
		$filters->add('projectfilterid', function (Criteria $criteria, $value, Query $query) {
			$entityFilter = EntityFilter::findById($value);
			$query = \GO\Projects2\Model\ProjectEntity::find(['id'])
				->selectSingleValue('p.contact_id')
				->filter($entityFilter->getFilter());

			$criteria->Where('c.id', 'IN', $query);
		});
	}

	public static function onUserFilter(Filters $filters)
	{

		$filters->add('projectId', function(Criteria $criteria, $value, Query $query){
			if(!$query->isJoined('pr2_resources', 'resource')) {
				$query->join('pr2_resources', 'resource', 'u.id=resource.user_id')
					->andWhere('resource.project_id = ' . $value);
			}
		});

		$filters->add('resourceForProject', function (Criteria $criteria, $value, Query $query){
			if($value) {
				if(!$query->isJoined('pr2_resources', 'resource')) {
					$query->join('pr2_resources', 'resource', 'u.id=resource.user_id')
						->join('pr2_projects', 'prjres', 'resource.project_id=prjres.id');
				}
				$query->andWhere('prjres.id = ' . $value)->distinct();
			}
		});

	}
	/**
	 * When a folder is created for a project we must attach the parent folders 
	 * to the projects as well.
	 * 
	 * @param \GO\Projects2\Model\Folder  $model
	 * @param \GO\Files\Model\Folder $folder
	 */
	public static function createParentFolders($model, $folder, $newFolder){
		
		if($newFolder && is_a($model, "GO\Projects2\Model\Project")){
			if($project = $model->parent()){
				GO::debug("Checking folder: ".$project->path);
				
				$folderController = new \GO\Files\Controller\FolderController();				
				$folderController->checkModelFolder($project, true, true);
			}
		}
		
	}
	

	public function autoInstall() {
		return true;
	}

	public function install() {
		
		// Install the notification cron for income contracts
		\GO\Projects2\Projects2Module::createDefaultIncomeContractNotificationCron();

		parent::install();

		$defaultType = new Type();
		$defaultType->name = GO::t("Default");
		$defaultType->save();

		$defaultStatus = new Status();
		$defaultStatus->name = GO::t("Ongoing", "projects2");
		$defaultStatus->show_in_tree=true;
		$defaultStatus->save();

		$noneStatus = new Status();
		$noneStatus->name = GO::t("None", "projects2");
		$noneStatus->show_in_tree=true;
		$noneStatus->filterable=true;
		$noneStatus->save();

		$status = new Status();
		$status->name = GO::t("Complete", "projects2");
		$status->complete=true;
		$status->show_in_tree=false;
		$status->save();


		$folder = new \GO\Base\Fs\Folder(GO::config()->file_storage_path.'projects2/template-icons');
		$folder->create();

		if(!$folder->child('folder.png')){
			$file = new \GO\Base\Fs\File(GO::modules()->projects2->path . 'install/images/folder.png');
			$file->copy($folder);
		}


		if(!$folder->child('project.png')){
			$file = new \GO\Base\Fs\File(GO::modules()->projects2->path . 'install/images/project.png');
			$file->copy($folder);
		}

		$template = new Template();
		$template->name = GO::t("Projects folder", "projects2");
		$template->name_template = "";
		$template->default_status_id = $noneStatus->id;
		$template->default_type_id = $defaultType->id;
		$template->icon=$folder->stripFileStoragePath().'/folder.png';
		$template->project_type=Template::PROJECT_TYPE_CONTAINER;
    $template->show_in_tree=true;
		$template->save();

		$template->acl->addGroup(GO::config()->group_everyone);


		$template = new Template();
		$template->name = GO::t("Standard project", "projects2");
		$template->name_template = "";
		$template->default_status_id = $defaultStatus->id;
		$template->default_type_id = $defaultType->id;
		$template->project_type=Template::PROJECT_TYPE_PROJECT;
		$template->fields = 'responsible_user_id,status_date,customer,budget_fees,contact,expenses';
		$template->icon=$folder->stripFileStoragePath().'/project.png';
		$template->save();

		$template->acl->addGroup(GO::config()->group_everyone);
	}

	/**
	 * Get the user this time entry is for based on a session that is set
	 * If the logged in user is no manger always return itself
	 * 
	 * @return int
	 */
	public static function getActiveTimeregistrationUserId() {

		if (GO::user()->getModulePermissionLevel('timeregistration2') == \GO\Base\Model\Acl::MANAGE_PERMISSION && !empty(GO::session()->values['tr_active_user']))
			return GO::session()->values['tr_active_user'];
		else
			return GO::user()->id;
	}

	/**
	 * Create the default notification cronjob for income contracts
	 */
	public static function createDefaultIncomeContractNotificationCron(){
		
		$cron = new \GO\Base\Cron\CronJob();

		$cron->name = GO::t("Contract Expiry Notification Cron", "projects2");
		$cron->active = true;
		$cron->runonce = false;
		$cron->minutes = '2';
		$cron->hours = '7';
		$cron->monthdays = '*';
		$cron->months = '*';
		$cron->weekdays = '*';
		$cron->job = 'GO\Projects2\Cron\IncomeNotification';

		return $cron->save();
	}
	
	
	public static function deleteUser($user){

		Model\Resource::model()->deleteByAttribute('user_id', $user->id);
		Model\DefaultResource::model()->deleteByAttribute('user_id', $user->id);
	}


	public function demo(Generator $faker)
	{
		$templates = Template::model()->find();

		$folderTemplate = $templates->fetch();
		$projectTemplate = $templates->fetch();

		$status = Status::model()->findSingle();

		$type = Type::model()->findSingleByAttribute('name', 'Demo');
		if(!$type){
			$type = new Type();
			$type->name='Demo';
			if(!$type->save())
			{
				var_dump($type->getValidationErrors());
				exit();
			}
			$type->acl->addGroup(\GO::config()->group_internal, \GO\Base\Model\Acl::WRITE_PERMISSION);
		}


		$folderProject1 = \GO\Projects2\Model\Project::model()->findSingleByAttribute('name','1. Webdesign');
		if(!$folderProject1){
			$folderProject1 = new \GO\Projects2\Model\Project();
			$folderProject1->name='1. Webdesign';
			$folderProject1->start_time=time();
			$folderProject1->description='Just a placeholder for sub projects.';
			$folderProject1->template_id=$folderTemplate->id;
			$folderProject1->type_id=$type->id;
			$folderProject1->status_id=$status->id;
			if(!$folderProject1->save()){
				var_dump($folderProject1->getValidationErrors());
				exit();
			}
		}

		$folderProject2 = \GO\Projects2\Model\Project::model()->findSingleByAttribute('name','2. Apps');
		if(!$folderProject2){
			$folderProject2 = new \GO\Projects2\Model\Project();
			$folderProject2->name='2. Apps';
			$folderProject2->start_time=time();
			$folderProject2->description='Just a placeholder for sub projects.';
			$folderProject2->template_id=$folderTemplate->id;
			$folderProject2->type_id=$type->id;
			$folderProject2->status_id=$status->id;
			if(!$folderProject2->save()){
				var_dump($folderProject2->getValidationErrors());
				exit();
			}
		}

		$users = User::find(['id', 'displayName', 'email'])->limit(3)->all();


		for($i = 0; $i < 10; $i++) {


				$rocketProject = new \GO\Projects2\Model\Project();
				$rocketProject->type_id = $type->id;
				$rocketProject->status_id = $status->id;
				$rocketProject->name = $faker->text(50);
				$rocketProject->description = $faker->text(100);
				$rocketProject->template_id = $projectTemplate->id;
				$rocketProject->parent_project_id = $i % 2 ? $folderProject1->id : $folderProject2->id;
				$rocketProject->start_time = time();
				$rocketProject->due_time = \GO\Base\Util\Date::date_add(time(), 0, 1);
				$rocketProject->company_id = Module::get()->demoCompany($faker)->id;
				$rocketProject->contact_id = Module::get()->demoContact($faker)->id;
//				$rocketProject->budget=20000;
				if(!$rocketProject->save()) {
					var_dump($rocketProject->getValidationErrors());
					exit();
				}


				$resource = new \GO\Projects2\Model\Resource();
				$resource->project_id = $rocketProject->id;
				$resource->user_id = $users[0]->id;
				$resource->budgeted_units = 100;
				$resource->external_fee = 80;
				$resource->internal_fee = 40;
				$resource->save();

				$resource = new \GO\Projects2\Model\Resource();
				$resource->project_id = $rocketProject->id;
				$resource->user_id = $users[1]->id;
				$resource->budgeted_units = 16;
				$resource->external_fee = 120;
				$resource->internal_fee = 60;
				$resource->save();

				$resource = new \GO\Projects2\Model\Resource();
				$resource->project_id = $rocketProject->id;
				$resource->user_id = $users[2]->id;
				$resource->budgeted_units = 16;
				$resource->external_fee = 90;
				$resource->internal_fee = 45;
				$resource->save();



				$expenseBudget = new \GO\Projects2\Model\ExpenseBudget();
				$expenseBudget->description = 'Machinery';
				$expenseBudget->nett = 10000;
				$expenseBudget->project_id = $rocketProject->id;
				$expenseBudget->save();

				$expense = new \GO\Projects2\Model\Expense();
				$expense->description = 'Rocket fuel';
				$expense->project_id = $rocketProject->id;
				$expense->invoice_id = "1234";
				$expense->nett = 3000;
				$expense->save();


				$expense = new \GO\Projects2\Model\Expense();
				$expense->expense_budget_id = $expenseBudget->id;
				$expense->description = 'Fuse machine';
				$expense->invoice_id = "1235";
				$expense->project_id = $rocketProject->id;
				$expense->nett = 2000;
				$expense->save();

			if(\go\core\model\Module::isInstalled("community", "tasks")) {
				$tasklist = TaskList::createForProject($rocketProject->id);

				\go\modules\community\tasks\Module::get()->demoTasks($faker, $tasklist, false);
			}


				if(\go\core\model\Module::isInstalled("legacy", "savemailas")) {
					\GO\Savemailas\SavemailasModule::get()->demoMail($faker, $rocketProject);
				}

				echo ".";
			}


	}
}
