/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Field } from "./Field.js";
import { createComponent } from "../Component.js";
/**
 * Field to return an array with objects
 *
 * @see Form
 */
export class ArrayField extends Field {
    /**
     *
     * @param buildField Function that returns a new form field for an array item
     */
    constructor(buildField) {
        super("div");
        this.buildField = buildField;
        this.enableChangeEvent = true;
        this.baseCls = "flow";
        this.items.on("datachanged", () => {
            if (this.enableChangeEvent) {
                this.fireChange();
            }
        });
    }
    renderControl() {
        // empty
    }
    set value(v) {
        super.value = v;
        this.enableChangeEvent = false;
        this.items.clear();
        if (v) {
            v.forEach((item) => {
                this.internalAddValue(item);
            });
        }
        this.enableChangeEvent = true;
    }
    get value() {
        const v = [];
        this.items.forEach((item) => {
            if (item instanceof Field) {
                v.push(item.value);
            }
        });
        return v;
    }
    /**
     * Add value to the values array. Also fires change event
     *
     * @param value
     */
    addValue(value) {
        if (!this.valueOnFocus) {
            this.captureValueForChange();
        }
        return this.internalAddValue(value);
    }
    internalAddValue(value) {
        const field = this.buildField(value);
        field.value = value;
        this.items.add(field);
        return field;
    }
    reset() {
        super.reset();
        if (this.items.count()) {
            this.enableChangeEvent = false;
            this.items.clear();
            this.enableChangeEvent = true;
            this.fireChange();
        }
    }
    isEmpty() {
        return this.items.count() > 0;
    }
}
/**
 * Shorthand function to create {@see ArrayField}
 *
 * @see https://github.com/Intermesh/goui-docs/blob/main/script/form/ArrayFieldPage.ts
 *
 * @param config
 * @param items
 */
export const arrayfield = (config, ...items) => createComponent(new ArrayField(config.buildField), config, items);
//# sourceMappingURL=ArrayField.js.map