/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Field } from "./Field.js";
import { createComponent } from "../Component.js";
import { DateTime, Format } from "../../util";
const defaultDisplayFieldRenderer = (v, field) => v !== null && v !== void 0 ? v : "";
/**
 * Display field
 *
 * Component used to display data only.
 *
 * A {@see Form} can be used to edit data but also to present data using display fields
 */
export class DisplayField extends Field {
    /**
     *
     * @param renderer Renderer function for the value of the field
     */
    constructor(renderer = defaultDisplayFieldRenderer) {
        super();
        this.renderer = renderer;
        this.baseCls = 'goui-display-field';
        /**
         * Escape value HTML
         *
         * {@link Format.escapeHTML}
         */
        this.escapeValue = true;
        /**
         * Hide this field when the value is empty
         */
        this.hideWhenEmpty = true;
    }
    createControl() {
        return document.createElement("div");
    }
    internalSetValue(v) {
        if (this.control) {
            const setFn = (str) => {
                if (this.escapeValue) {
                    str = Format.escapeHTML(str);
                }
                this.control.innerHTML = str;
                if (this.hideWhenEmpty)
                    this.hidden = str == "";
            }, str = this.renderer(v, this);
            str instanceof Promise ? str.then(setFn) : setFn(str);
        }
    }
}
/**
 * Shortcut function to create a {@see DisplayField}
 *
 * @param config
 * @param items
 */
export const displayfield = (config, ...items) => { var _a; return createComponent(new DisplayField((_a = config === null || config === void 0 ? void 0 : config.renderer) !== null && _a !== void 0 ? _a : defaultDisplayFieldRenderer), config, items); };
/**
 * Create display field with date icon and renderer
 *
 * @param config
 * @param items
 */
export const displaydatefield = (config, ...items) => {
    var _a;
    if (!config.icon)
        config.icon = "today";
    if (!config.renderer)
        config.renderer = (v) => v ? (new DateTime(v)).format(Format.dateFormat) : "";
    return createComponent(new DisplayField((_a = config === null || config === void 0 ? void 0 : config.renderer) !== null && _a !== void 0 ? _a : defaultDisplayFieldRenderer), config, items);
};
//# sourceMappingURL=DisplayField.js.map