/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { DateInterval } from "./DateInterval";
const SystemTimeZone = Intl.DateTimeFormat().resolvedOptions().timeZone.toLowerCase();
function pad(n) {
    return n.toString().padStart(2, "0");
}
/**
 * DateTime object
 *
 * Adds formatting, parsing and timezone support to a standard Date object
 *
 * @category Utility
 */
export class DateTime {
    static staticInit(lang) {
        const locale = new Intl.Locale(lang), dayList = ['su', 'mo', 'tu', 'we', 'th', 'fr', 'sa'];
        if ('weekInfo' in locale) {
            // @ts-ignore
            DateTime.firstWeekDay = locale.weekInfo.firstDay; // weekInfo might not be supported in all browsers
        }
        let tmp = new Date('1970-01-01'), intlDays = new Intl.DateTimeFormat(lang, { weekday: 'long' });
        for (let i = 0; i < 7; i++) { // monday
            tmp.setDate(i + 4 + DateTime.firstWeekDay);
            const d = dayList[tmp.getDay()];
            DateTime.dayMap.push(d);
            DateTime.dayNames[d] = intlDays.format(tmp);
        }
        let intlMonth = new Intl.DateTimeFormat(lang, { month: 'long' });
        for (let i = 0; i < 12; i++) {
            tmp.setMonth(i);
            DateTime.monthNames[i] = intlMonth.format(tmp);
        }
    }
    /**
     * Constructor
     *
     * @param date Can be a date object, a unix timestamp or date string (see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Date#date_time_string_format)
     */
    constructor(date) {
        /**
         * The timezone of the date
         */
        this.timezone = SystemTimeZone;
        this.date = (date instanceof Date) ?
            date :
            (date ? new Date(date) : new Date());
    }
    /**
     * User timezone offset in minutes
     *
     * eg.
     *
     * ```
     * const browser = new Date("2021-10-21T16:00:00+00:00");
     *
     * const amsterdam = utc.toTimezone("europe/amsterdam");
     *
     * console.log(amsterdam.getUserTimezoneOffset()); // 120 (in daylight savings time)
     * ```
     */
    getTimezoneOffset() {
        //JS dates always have system timezone
        const systemTZOffset = this.date.getTimezoneOffset();
        return systemTZOffset + this.getSystemTimezoneDiff();
    }
    /**
     * Convert date to timezone
     *
     * @example On computer with Amsterdam timezone
     *
     * ```
     * const browser = new Date("2021-10-21T16:00:00+00:00");
     * console.log(browser.format("c"), browser.getTimeZone());
     * const riga = browser.toTimezone("europe/riga");
     *
     * console.log(riga.format("c"), riga.getTimeZone());
     *
     * const utc = riga.toUTC();
     *
     * console.log(utc.format("c"), utc.getTimeZone());
     *
     * ```
     *
     * Output:
     * 2021-10-21T18:00:00+02:00 Europe/Amsterdam
     * 2021-10-21T19:00:00+03:00 europe/riga
     * 2021-10-21T16:00:00+00:00 UTC
     *
     *
     * @param timezone eg. europe/amsterdam
     */
    toTimezone(timezone) {
        if (this.timezone == timezone) {
            return this.clone();
        }
        const offset = this.getTimezoneOffset();
        // get the difference in timezone
        const d = this.clone();
        d.timezone = timezone;
        const newOffset = d.getTimezoneOffset();
        d.setMinutes(d.getMinutes() - newOffset + offset);
        return d;
    }
    /**
     * Calculate difference between this and the given date
     *
     * @param end
     */
    diff(end) {
        const di = new DateInterval();
        di.setFromDates(this, end);
        return di;
    }
    /**
     * Create a copy of this object without reference
     */
    clone() {
        return new DateTime(new Date(this.date));
    }
    /**
     * Convert to UTC timezone
     */
    toUTC() {
        return this.toTimezone("UTC");
    }
    static getFormatter(timezone) {
        if (!DateTime.cache[timezone]) {
            DateTime.cache[timezone] = new Intl.DateTimeFormat('en-US', {
                timeZone: timezone,
                dateStyle: 'short',
                timeStyle: 'short'
            });
        }
        return DateTime.cache[timezone];
    }
    /**
     * Get the offset between the system timezone and the date timezone in minutes.
     *
     * For example if the computer has europe/amsterdam and the date UTF it's 60 minutes in winter time.
     *
     * @private
     */
    getSystemTimezoneDiff() {
        if (this.timezone == SystemTimeZone) {
            return 0;
        }
        //calculate diff in minutes when changing to given timezone.
        const local = DateTime.getFormatter(this.timezone).format(this.date);
        // const local = this.date.toLocaleString("en-US", {timeZone: this.timezone});
        const d = new Date(local);
        // we don't care about milliseconds and seconds
        const u = Math.floor(this.date.getTime() / 1000 / 60), localU = Math.floor(d.getTime() / 1000 / 60);
        return u - localU;
    }
    /**
     * The ISO-8601 week number of year (weeks starting on Monday)
     */
    getWeekOfYear() {
        const ms1d = 864e5, // milliseconds in a day
        ms7d = 7 * ms1d; // milliseconds in a week
        // adapted from http://www.merlyn.demon.co.uk/weekcalc.htm
        const DC3 = Date.UTC(this.getYear(), this.getMonth() - 1, this.getMonthDay() + 3) / ms1d, // an Absolute Day Number
        AWN = Math.floor(DC3 / 7), // an Absolute Week Number
        Wyr = new Date(AWN * ms7d).getUTCFullYear();
        return AWN - Math.floor(Date.UTC(Wyr, 0, 7) / ms7d) + 1;
    }
    /**
     * Get the numeric day number of the year
     */
    getDayOfYear() {
        let num = 0, d = new DateTime(this.date.getTime()), m = this.getMonth(), i;
        for (i = 1, d.setMonthDay(1), d.setMonth(1); i < m; d.setMonth(++i)) {
            num += d.getDaysInMonth();
        }
        return num + this.getMonthDay() - 1;
    }
    /**
     * Get the year
     */
    getYear() {
        return this.date.getFullYear();
    }
    /** Gets the month. Unlike JS it's 1 - 12 */
    getMonth() {
        return this.date.getMonth() + 1;
    }
    getDate() {
        return this.date.getDate();
    }
    getMonthDay() {
        return this.getDate();
    }
    /** 0 for sunday, 1 for monday, 2 for tuesday */
    getDay() {
        return this.date.getDay();
    }
    /**
     * Like getDay but take firstWeekDay of the week into account
     * 0 = first day of the week, 6 = last day
     */
    getWeekDay() {
        if (DateTime.firstWeekDay == 1) { // monday
            return (this.date.getDay() || 7) - 1;
        }
        // sunday
        return this.date.getDay();
    }
    getHours() {
        return this.date.getHours();
    }
    getMinutes() {
        return this.date.getMinutes();
    }
    getSeconds() {
        return this.date.getSeconds();
    }
    getMilliseconds() {
        return this.date.getMilliseconds();
    }
    /** Gets the time value in milliseconds. */
    getTime() {
        return this.date.getTime();
    }
    getMinuteOfDay() {
        return this.date.getHours() * 60 + this.date.getMinutes();
    }
    /**
     * Sets the hour value in the Date object
     * @param hours A numeric value equal to the hours value.
     * @params min A numeric value equal to the minutes value.
     * @param sec A numeric value equal to the seconds value.
     */
    setHours(hours, min = this.date.getMinutes(), sec = this.date.getSeconds(), ms = this.date.getMilliseconds()) {
        this.date.setHours(hours, min, sec, ms);
        return this;
    }
    setMinutes(min) {
        this.date.setMinutes(min);
        return this;
    }
    setSeconds(s) {
        this.date.setSeconds(s);
        return this;
    }
    setMilliseconds(ms) {
        this.date.setMilliseconds(ms);
        return this;
    }
    setYear(year) {
        this.date.setFullYear(year);
        return this;
    }
    setMonth(month) {
        this.date.setMonth(month - 1);
        return this;
    }
    setDate(date) {
        this.date.setDate(date);
        return this;
    }
    setMonthDay(date) {
        return this.setDate(date);
    }
    setWeekDay(day) {
        this.date.setDate(this.date.getDate() - this.getWeekDay() + day);
        return this;
    }
    /** Jump to day in current week */
    setDay(day) {
        this.date.setDate(this.date.getDate() - this.date.getDay() + day);
        return this;
    }
    addYears(years) {
        this.date.setFullYear(this.date.getFullYear() + years);
        return this;
    }
    addMonths(months) {
        this.date.setMonth(this.date.getMonth() + months);
        return this;
    }
    addDays(days) {
        this.date.setDate(this.date.getDate() + days);
        return this;
    }
    addHours(hours) {
        this.date.setHours(this.date.getHours() + hours);
        return this;
    }
    addMinutes(minutes) {
        this.date.setMinutes(this.date.getMinutes() + minutes);
        return this;
    }
    addSeconds(seconds) {
        this.date.setSeconds(this.date.getSeconds() + seconds);
        return this;
    }
    /**
     * Add a date interval
     *
     * @param dateInterval
     */
    add(dateInterval) {
        const inv = dateInterval.invert ? -1 : 1;
        this.setYear(this.getYear() + dateInterval.years * inv);
        this.setMonth(this.getMonth() + dateInterval.months * inv);
        this.setDay(this.getDay() + dateInterval.days * inv);
        this.setHours(this.getHours() + dateInterval.hours * inv);
        this.setMinutes(this.getMinutes() + dateInterval.minutes * inv);
        this.setSeconds(this.getSeconds() + dateInterval.seconds * inv);
        return this;
    }
    /**
     * Check if current date is in a leap year
     */
    isLeapYear() {
        const year = this.getYear();
        return !!((year & 3) == 0 && (year % 100 || (year % 400 == 0 && year)));
    }
    /**
     * Get the number of days in the current month
     */
    getDaysInMonth() {
        const daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
        const m = this.getMonth() - 1;
        return m == 1 && this.isLeapYear() ? 29 : daysInMonth[m];
    }
    getGMTOffset(colon = ":") {
        const tzo = this.getTimezoneOffset();
        return (tzo > 0 ? "-" : "+") + pad(Math.floor(Math.abs(tzo) / 60)) + colon + pad(Math.abs(tzo % 60));
    }
    /**
     * Format the date into a string
     *
     * You can use the following characters. You can escape a character with a \ to output it as given.
     *
     * d - The day of the month (from 01 to 31)
     * D - A textual representation of a day (three letters)
     * j - The day of the month without leading zeros (1 to 31)
     * l (lowercase 'L') - A full textual representation of a day
     * N - The ISO-8601 numeric representation of a day (1 for Monday, 7 for Sunday)
     * S - The English ordinal suffix for the day of the month (2 characters st, nd, rd or th. Works well with j)
     * w - A numeric representation of the day (0 for Sunday, 6 for Saturday)
     * z - The day of the year (from 0 through 365)
     * W - The ISO-8601 week number of year (weeks starting on Monday)
     * F - A full textual representation of a month (January through December)
     * m - A numeric representation of a month (from 01 to 12)
     * M - A short textual representation of a month (three letters)
     * n - A numeric representation of a month, without leading zeros (1 to 12)
     * t - The number of days in the given month
     * L - Whether it's a leap year (1 if it is a leap year, 0 otherwise)
     * o - The ISO-8601 year number
     * Y - A four digit representation of a year
     * y - A two digit representation of a year
     * a - Lowercase am or pm
     * A - Uppercase AM or PM
     * B - Swatch Internet time (000 to 999)
     * g - 12-hour format of an hour (1 to 12)
     * G - 24-hour format of an hour (0 to 23)
     * h - 12-hour format of an hour (01 to 12)
     * H - 24-hour format of an hour (00 to 23)
     * i - Minutes with leading zeros (00 to 59)
     * s - Seconds, with leading zeros (00 to 59)
     * u - Microseconds (added in PHP 5.2.2)
     * e - The timezone identifier (Examples: UTC, GMT, Atlantic/Azores)
     * I  (capital i) - Whether the date is in daylights savings time (1 if Daylight Savings Time, 0 otherwise)
     * O - Difference to Greenwich time (GMT) in hours (Example: +0100)
     * P - Difference to Greenwich time (GMT) in hours:minutes (added in PHP 5.1.3)
     * T - Timezone abbreviations (Examples: EST, MDT)
     * Z - Timezone offset in seconds. The offset for timezones west of UTC is negative (-43200 to 50400)
     * c - The ISO-8601 date (e.g. 2013-05-05T16:34:42+00:00)
     * r - The RFC 2822 formatted date (e.g. Fri, 12 Apr 2013 12:01:05 +0200)
     * U - The seconds since the Unix Epoch (January 1 1970 00:00:00 GMT)
     *
     */
    format(format) {
        const chars = format.split("");
        let output = "";
        for (let i = 0, l = chars.length; i < l; i++) {
            let char = chars[i];
            if (char == '\\') {
                i++;
                if (chars.length > i + 1) {
                    char += chars[i];
                }
            }
            else if (char in DateTime.converters) {
                char = DateTime.converters[char](this) + "";
            }
            output += char;
        }
        return output;
    }
    static createFormatRegex(format) {
        const chars = format.split("");
        let output = "";
        for (let i = 0, l = chars.length; i < l; i++) {
            let char = chars[i];
            switch (char) {
                case 'Y':
                    char = "(?<Y>\\d{4})";
                    break;
                case 's':
                case 'i':
                case 'y':
                    char = "(?<" + char + ">\\d{2})";
                    break;
                case 'G':
                case 'H':
                case 'm':
                case 'd':
                case 'g':
                case 'h':
                case 'j':
                case 'n':
                    char = "(?<" + char + ">\\d{1,2})";
                    break;
                case 'a':
                    char = "(?<a>am|pm)";
                    break;
                case 'A':
                    char = "(?<a>AM|PM)";
                    break;
                case 'P':
                    char = "(?<P>[\+\-](\\d{2}):(\\d{2}))";
                    break;
                case 'c':
                    char = DateTime.createFormatRegex("Y-m-d\TH:i:sP");
                    break;
                // case '\\':
                // 	i++;
                // 	if (chars.length > i + 1) {
                // 		char += chars[i];
                // 	}
                // 	break;
                default:
                    //do nothing
                    break;
            }
            output += char;
        }
        return output;
    }
    /**
     * Create date by given format. See {@link DateTime.format}.
     * Supports:
     * Y, y, m, n, d, j, H, h, G, g, i, s, a, A
     *
     * @example
     * ```
     * const date = Date.createFromFormat("2021-10-21 21:09", "Y-m-d H:i"));
     *
     * const date = console.log(Date.createFromFormat("10/12/2021 9:09am", "m/d/Y g:ia", "America/New_York));
     * ```
     */
    static createFromFormat(dateStr, format = "c", timezone) {
        const regex = new RegExp(DateTime.createFormatRegex(format), 'u');
        const result = regex.exec(dateStr);
        if (!result) {
            return undefined;
        }
        const date = new DateTime('1970-01-01'); // we want this to always work the same
        date.setHours(0, 0, 0, 0);
        if (timezone) {
            date.timezone = timezone;
        }
        // Set year and month first...
        if (result.groups["Y"]) {
            date.setYear(parseInt(result.groups["Y"]));
            delete result.groups["Y"];
        }
        else if (result.groups["y"]) {
            date.setYear(parseInt(2000 + result.groups["y"]));
            delete result.groups["y"];
        }
        for (let key of ["n", "m"]) {
            if (result.groups[key]) {
                date.setMonth(parseInt(result.groups[key]));
                delete result.groups[key];
            }
        }
        // ...then do the rest.
        for (let key in result.groups) {
            switch (key) {
                case 'j':
                case 'd':
                    date.setMonthDay(parseInt(result.groups[key]));
                    break;
                case 'G':
                case 'H':
                    date.setHours(parseInt(result.groups[key]));
                    break;
                case 'h':
                case 'g':
                    const pm = result.groups.a ? result.groups.a == 'pm' : result.groups.A ? result.groups.A == 'PM' : false;
                    const h = pm ? parseInt(result.groups[key]) : parseInt(result.groups[key]) + 12;
                    date.setHours(h);
                    break;
                case 'i':
                    date.setMinutes(parseInt(result.groups[key]));
                    break;
                case 's':
                    date.setSeconds(parseInt(result.groups[key]));
                    break;
            }
        }
        return date;
    }
    /**
     * Compare with given date
     *
     * Returns:
     *
     * - -1 if this interval is smaller than the other
     * - 0 if intervals are equal
     * - 1 if this interval is greater than the other
     *
     * @param date
     * @return number
     */
    compare(date) {
        return (this.date > date.date ? 1 : 0) - (this.date < date.date ? 1 : 0);
    }
}
DateTime.firstWeekDay = 1; // 1 = monday, 7 = sunday
DateTime.dayNames = {}; // {mo: t('Monday'), tu: t('Tuesday'), ...}
DateTime.dayMap = []; // ['mo','tu',...] if week starts on monday else index 0 = 'su'
DateTime.monthNames = [];
DateTime.cache = {};
DateTime.converters = {
    'd': date => pad(date.getMonthDay()),
    'D': date => DateTime.dayNames[DateTime.dayMap[date.getWeekDay()]].substring(0, 3),
    'j': date => date.getMonthDay().toString(),
    'l': date => DateTime.dayNames[DateTime.dayMap[date.getWeekDay()]],
    'S': date => ["st", "nd", "rd"][((date.getMonthDay() + 90) % 100 - 10) % 10 - 1] || "th",
    'w': date => date.getDay().toString(),
    'z': date => date.getDayOfYear().toString(),
    'W': date => date.getWeekOfYear().toString(),
    'F': date => DateTime.monthNames[date.getMonth() - 1],
    'm': date => pad(date.getMonth()),
    'M': date => DateTime.monthNames[date.getMonth() - 1].substring(0, 3),
    'n': date => date.getMonth().toString(),
    'Y': date => date.getYear().toString(),
    'y': date => (date.getYear() + "").substr(-2),
    'a': date => date.getHours() > 12 ? 'pm' : 'am',
    'A': date => date.getHours() > 12 ? 'PM' : 'AM',
    'g': date => (date.getHours() % 12).toString(),
    'G': date => date.getHours().toString(),
    'h': date => pad(date.getHours() % 12),
    'H': date => pad(date.getHours()),
    'i': date => pad(date.getMinutes()),
    's': date => pad(date.getSeconds()),
    'O': date => date.getGMTOffset(""),
    'P': date => date.getGMTOffset(),
    'U': date => Math.floor(date.getTime() / 1000).toString(),
    'c': date => date.format("Y-m-d\TH:i:sP")
};
DateTime.staticInit(navigator.language);
//# sourceMappingURL=DateTime.js.map