/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { client } from "./Client.js";
import { AbstractDataSource } from "@intermesh/goui";
import { createComponent } from "@intermesh/goui";
var andOrNot;
(function (andOrNot) {
    andOrNot[andOrNot["AND"] = 0] = "AND";
    andOrNot[andOrNot["OR"] = 1] = "OR";
    andOrNot[andOrNot["NOT"] = 2] = "NOT";
})(andOrNot || (andOrNot = {}));
/**
 * JMAP Data source
 *
 * Single Source Of Truth for JMAP entities
 *
 */
export class JmapDataSource extends AbstractDataSource {
    constructor() {
        super(...arguments);
        /**
         * The ID to use when committing
         */
        this._nextCallId = 1;
    }
    internalQuery(params) {
        var _a;
        return client.jmap(((_a = this.controllerRoute) !== null && _a !== void 0 ? _a : this.id) + "/query", params, this.useCallId());
    }
    /**
     * The call ID of the next JMAP method call. Useful for result references.
     */
    get nextCallId() {
        return this.id + "_" + this._nextCallId;
    }
    useCallId() {
        const callId = this.nextCallId;
        this._nextCallId++;
        return callId;
    }
    async internalCommit(params) {
        var _a;
        try {
            return await client.jmap(((_a = this.controllerRoute) !== null && _a !== void 0 ? _a : this.id) + "/set", params, this.useCallId());
        }
        catch (error) {
            //automatic retry when statemismatch occurs
            if (error.type && error.type == 'stateMismatch') {
                console.warn("statemismatch, we'll update and auto retry the JMAP set request");
                await this.updateFromServer();
                params.ifInState = await this.getState();
                return this.internalCommit(params);
            }
            throw error;
        }
    }
    /**
     * This function makes sure the store is up to date. Should not be necessary but we ran into problems where tasks
     * were out of date when viewed. This should always prevent that.
     * @return {Promise<self>}
     */
    async validateState() {
        var _a;
        const r = await client.jmap(((_a = this.controllerRoute) !== null && _a !== void 0 ? _a : this.id) + "/get", {
            ids: []
        });
        await this.checkState(r.state, r);
        return this;
    }
    internalGet(ids) {
        var _a;
        return client.jmap(((_a = this.controllerRoute) !== null && _a !== void 0 ? _a : this.id) + '/get', {
            ids: ids
        }, this.useCallId());
    }
    async internalRemoteChanges(state) {
        var _a;
        return client.jmap(((_a = this.controllerRoute) !== null && _a !== void 0 ? _a : this.id) + "/changes", {
            sinceState: state
        }, this.useCallId());
    }
}
const stores = {};
/**
 * Get a single instance of a store by ID
 *
 * @param storeId
 */
export const jmapds = (storeId, config) => {
    if (!stores[storeId]) {
        stores[storeId] = createComponent(new JmapDataSource(storeId), config);
    }
    return stores[storeId];
};
//# sourceMappingURL=JmapDataSource.js.map