<?php
namespace go\modules\business\contracts;

use Exception;
use Faker\Generator;
use go\core;
use go\core\model;
use go\core\orm\exception\SaveException;
use go\core\orm\Mapping;
use go\core\orm\Property;
use go\modules\business\business\model\Business;
use go\modules\business\catalog\model\Article;
use go\modules\business\contracts\cron\CreateInvoices;
use go\modules\business\contracts\model\Contract;
use go\modules\business\contracts\model\ContractBook;
use go\modules\business\contracts\model\ContractItem;
use go\modules\business\contracts\model\ContractsBusiness;
use go\modules\business\contracts\model\Settings;
use go\modules\business\finance\model\FinanceBook;
use go\modules\business\finance\model\FinanceDocument;
use go\modules\business\finance\model\FinanceDocumentItem;
use go\modules\business\finance\model\FinanceDocumentItemGroup;
use go\modules\business\finance\model\Payment;

/**
 * @copyright (c) 2020, Intermesh BV https://www.intermesh.nl
 * @author System Administrat <admin@intermesh.localhost>
 * @license http://www.gnu.org/licenses/agpl-3.0.html AGPLv3
 */
class Module extends core\Module
{
	public function getAuthor() : string
	{
		return "Intermesh BV <info@intermesh.nl>";
	}

	public function getDependencies() : array
	{
		return ["business/business", "business/finance", "community/addressbook"];
	}

	public function requiredLicense(): ?string
	{
		return "billing";
	}

    /**
     * The development status of this module
     * @return string
     */
    public function getStatus() : string{
        return self::STATUS_STABLE;
    }

	public function autoInstall(): bool
	{
		return true;
	}

	public function defineListeners() {
		parent::defineListeners();

		model\EmailTemplate::on(model\EmailTemplate::EVENT_PERMISSION_LEVEL, static::class, 'onTemplatePermissionLevel');
		model\PdfTemplate::on(model\PdfTemplate::EVENT_PERMISSION_LEVEL, static::class, 'onTemplatePermissionLevel');

	}

	public static function onTemplatePermissionLevel($emailTemplate) {
		if($emailTemplate->moduleId == static::get()->getModel()->id) {
			return \go\modules\business\business\Module::get()->getModel()->getPermissionLevel();
		}
	}

	/**
	 * @throws SaveException
	 * @throws Exception
	 */
	protected function afterInstall(model\Module $model) : bool
	{
		CreateInvoices::install("0 6 * * *");

		foreach(Business::find() as $business) {
			//for creating defaults in ContractsBusiness
			if(!$business->save()) {
				throw new SaveException($business);
			}

			$financeBook = FinanceBook::find()->where(['businessId' => $business->id])->single();

			if($financeBook) {
				$contractBook = new ContractBook();
				$contractBook->businessId = $business->id;
				$contractBook->name = $business->name;
				$contractBook->targetBookId = $financeBook->id;
				$contractBook->save();
			}

		}

		return parent::afterInstall($model);
	}

	public static function onBusinessMap(Mapping $mapping) {
		$mapping->addHasOne('contracts', ContractsBusiness::class, ['id' => 'businessId'], true);
	}

	public static function onGarbageCollection() {

		//clean up templates starting with "<businessId>-"
		$moduleId = static::get()->getModel()->id;


		//cleanup templates from DebtorProfileAction
		model\EmailTemplate::delete(
			(new core\db\Query())
				->where(['moduleId' => $moduleId])
				->where('`key` LIKE "contract-book-%"')
				->andWhere('SUBSTRING(`key`, 15) NOT IN (SELECT id FROM business_contracts_book)')
		);


	}


	public function demo(Generator $faker)
	{
		\go\modules\business\catalog\Module::get()->demo($faker);

		$articles = Article::find()
			->selectSingleValue('id')
			->limit(0)
			->offset(10)
			->orderBy(['id' => 'DESC'])
			->all();

		$now = new core\util\DateTime("-14 days");

		foreach(ContractBook::find() as $book) {

			for($n = 0; $n < 10; $n++) {
				$doc = new Contract();
				$doc->reference = $faker->numberBetween(1000, 9999);
				$doc->description = $faker->text(50);
				$doc->bookId = $book->id;
				$doc->startsAt = new core\util\DateTime("@".$faker->dateTimeBetween("-2 years", "now")->format("U"));
				$doc->contactId = \go\modules\community\addressbook\Module::get()->demoContact($faker)->id;
				$doc->setCustomerId(\go\modules\community\addressbook\Module::get()->demoCompany($faker)->id);

				$doc->intervalMonths = $n % 2 ? 1 : 12;


				for ($i = 0; $i < $faker->numberBetween(1, 10); $i++) {
					$item = ContractItem::fromArticleId($doc, $articles[$faker->numberBetween(0, 9)]);
					$item->quantity = $faker->numberBetween(1, 10);
					$doc->items[] = $item;
				}


				if (!$doc->save()) {
					throw new SaveException($doc);
				}

				while($doc->nextAt < $now) {
					$invoice = $doc->createNext();

					$payment = new Payment();
					$payment->businessId = $book->businessId;
					$payment->documentId = $invoice->id;
					$payment->customerId = $invoice->getCustomerId();
					$payment->amount = $invoice->getTotalPrice();
					$payment->description = $invoice->number;
					$payment->date = (clone $invoice->date)->add(new \DateInterval("P14D"));
					if (!$payment->save()) {
						throw new SaveException($payment);
					}
				}
			}

		}
	}

	public function getSettings()
	{
		return Settings::get();
	}


}
