<?php

namespace go\modules\business\support\install;

use Exception;
use GO\Base\Db\FindCriteria;
use GO\Base\Db\FindParams;
use GO\Base\Model\Acl;
use go\core\ErrorHandler;
use go\core\fs\Blob;
use go\core\fs\File;
use go\core\jmap\Entity;
use go\core\model\Alert;
use go\core\model\Alert as CoreAlert;
use go\core\model\Field;
use go\core\model\Link;
use go\core\model\User;
use go\core\orm\EntityType;
use go\core\orm\exception\SaveException;
use go\core\util\DateTime;
use go\core\util\StringUtil;
use go\modules\business\support\Module;
use go\modules\community\addressbook\model\Contact;
use go\modules\community\comments\model\Comment;
use go\modules\community\comments\model\CommentAttachment;
use go\modules\community\tasks\model\Category;
use go\modules\community\tasks\model\Task;
use go\modules\community\tasks\model\TaskList;
use go\modules\community\tasks\model\TaskListGrouping;
use GO\Tickets\Model\Message;
use GO\Tickets\Model\Status;
use GO\Tickets\Model\Ticket;
use GO\Tickets\Model\Type;
use GO\Tickets\Model\TypeGroup;

class MigrateTickets
{
	/**
	 * @var CoreAlert
	 */
	private $alert;

	public function run(bool $createUsers = true)
	{

		$this->alert = new Alert();
		$this->alert->setEntity(Module::get()->getModel());
		$this->alert->userId = go()->getUserId();
		$this->alert->triggerAt = new DateTime();
		$this->alert->setData([
				'title' => go()->t("Tickets migration", "business", "support"),
				'body' => go()->t("Your tickets migration process has started in the background.", "business", "finance"),
				'progress' => 0
			]
		);
		$this->alert->tag = "migratetickets";

		if (!$this->alert->save()) {
			throw new SaveException($this->alert);
		}

		// Speed things up.
		Entity::$trackChanges = false;
		\go\modules\community\history\Module::$enabled = false;

		Module::$sendMailOnComment = false;

		CoreAlert::$enabled = false;

		// Speed things up.
		Entity::$trackChanges = false;
		\go\modules\community\history\Module::$enabled = false;

		// we need this for now
		go()->getSettings()->allowRegistration = true;

		try {
			$this->migrateTypes();
			$this->migrateStatuses();
			$this->migrateTickets($createUsers);

			CoreAlert::$enabled = true;

			$this->alert->setData([
					'title' => go()->t("Tickets migration", "business", "support"),
					'body' => go()->t("Your tickets have been migrated to the support module.", "business", "finance"),
					'progress' => 100
				]
			);
			if (!$this->alert->save()) {
				throw new SaveException($this->alert);
			}

		} catch (\Throwable $e) {
			$error = ErrorHandler::logException($e);

			echo $error . "\n";

			Entity::$trackChanges = true;
			CoreAlert::$enabled = true;
			$this->alert->setData([
					'title' => go()->t("Tickets migration", "business", "support"),
					'body' => go()->t("The migration failed: ", "business", "support") . $e->getMessage(),
					'progress' => 100
				]
			);
			if (!$this->alert->save()) {
				throw new SaveException($this->alert);
			}
		}

	}


	private function progress(int $progress)
	{

		$this->alert->setData([
				'progress' => $progress
			]
		);

		Entity::$trackChanges = true;
		CoreAlert::$enabled = true;
		if (!$this->alert->save()) {
			throw new SaveException($this->alert);
		}
		CoreAlert::$enabled = false;
		EntityType::push();
		Entity::$trackChanges = false;
	}


	private function migrateTickets(bool $createUsers = true)
	{

		echo "Tickets\n";

		Field::create("Task", "Support", "oldTicketNo", "Old ticket no.", "Text");

		go()->rebuildCache();

		echo "Creating index\n";
		go()->getDbConnection()->exec("create index if not exists tasks_task_custom_fields_oldTicketNo_index
    on tasks_task_custom_fields (oldTicketNo);");

		echo "Finding tickets to migrate\n";
		$fp = FindParams::newInstance()
//			->order('ctime', 'desc')
			->ignoreAcl()
			->join('tasks_task_custom_fields', FindCriteria::newInstance()
				->addRawCondition('cf.oldTicketNo', 't.ticket_number'),
				'cf', 'LEFT');
		$fp->getCriteria()->addRawCondition('cf.oldTicketNo IS NULL');

//		$fp->limit(10);

		$tickets = Ticket::model()->find($fp);

		echo "Start migrate of tickets\n";

		$total = $tickets->rowCount();
		$i = 0;
		foreach ($tickets as $ticket) {

			try {

				echo "Ticket: " . $ticket->ticket_number . "\n";

				/** @var Ticket $ticket */

//			$task = Task::find()
//				->joinCustomFields()
//				->where('customFields.oldTicketNo', '=', $ticket->ticket_number)
//				->single();
//			if(!$task) {
				$task = new Task();
//			} else{
//				//remove comments
//				Comment::delete(Comment::findFor($task));
//			}


				/*
* - @property string $contactName The full name of the contact
* - @property int $group_id
* x @property int $unseen
* @property int $files_folder_id
* x @property int $mtime
* x @property int $ctime
* x @property string $subject
* - @property string $phone
* - @property string $email
* - @property string $last_name
* - @property string $middle_name
* - @property string $first_name
* x @property int $company_id
* - @property string $company
* x @property int $contact_id
* x @property int $agent_id
* x @property int $user_id
* x @property int $type_id
* x @property int $status_id
* x @property int $priority
* - @property int $ticket_verifier
* x @property string $ticket_number
* x @property int $id
* - @property string $country
* - @property string $state
* - @property string $city
* x @property int $due_date
* - @property bool $due_reminder_sent true when reminder mail was sent in due date
* x @property string $zip
* x @property string $address_no
* x @property string $address
* x @property string $salutation
* x @property int $last_response_time
* x @property string $cc_addresses
				 */


				$task->setCustomField('oldTicketNo', $ticket->ticket_number);
				$task->title = $ticket->subject;

				try {
					if ($createUsers) {
						$user = User::findOrCreateByUsername($ticket->email, $ticket->email, trim($ticket->first_name . " " . $ticket->middle_name) . " " . $ticket->last_name, ['enabled' => false], ['id', 'username', 'email', 'displayName']);
					} else {
						$user = User::find()->where(['email' => $ticket->email])->single();
					}
					$task->createdBy = $user ? $user->id : null;
				} catch (Exception $e) {
					echo "Exception while creating user : " . $ticket->email . " : " . $e->getMessage() . "\n";
					$task->createdBy = 1;
				}


				$task->createdAt = DateTime::createFromFormat("U", $ticket->mtime);
				$task->modifiedBy = \go\core\model\User::exists($ticket->muser_id) ? $ticket->muser_id : 1;
				$task->modifiedAt = DateTime::createFromFormat("U", max($ticket->mtime, $ticket->last_response_time));
				//for a second run we want to make sure it's modified otherwise it's set on save. We'll add one second.
				if (!$task->isModified('modifiedAt')) {
					$task->modifiedAt->add(new \DateInterval("PT1S"));
				}
				$task->responsibleUserId = User::exists($ticket->agent_id) ? $ticket->agent_id : null;
				$task->description = "";
				$task->tasklistId = isset($this->typesMap[$ticket->type_id]) ? $this->typesMap[$ticket->type_id] : current($this->typesMap);
				$task->filesFolderId = $ticket->files_folder_id;

				if ($ticket->status_id == -1) {
					$task->setProgress("completed");
				} else {
					$task->setProgress($ticket->unseen || !$task->responsibleUserId ? "needs-action" : "in-progress");

					if ($ticket->status_id != 0) {
						$task->categories = [$this->statusMap[$ticket->status_id]];
					}
				}

//			if(!empty($ticket->due_date)) {
				$task->due = $task->modifiedAt->add(new \DateInterval("P1D"));
//			}

				switch ($ticket->priority) {
					case Ticket::PRIORITY_HIGH:
						$task->priority = Task::PRIORITY_HIGH;
						break;
					case Ticket::PRIORITY_LOW:
						$task->priority = Task::PRIORITY_LOW;
						break;
				}


				// create description from note type messages
//			$messages = Message::model()->findByAttribute('ticket_id', $ticket->id, FindParams::newInstance()->select('*'))->fetchAll();
//			foreach($messages as $message) {
//				if(!empty(empty($message->content)) && $message->is_note) {
//					$createdAt = DateTime::createFromFormat("U", $message->ctime);
//					$user = UserDisplay::findById($message->user_id);
//					$task->description .=  "\n\n" . $user->displayName . " @ " . $createdAt->toUserFormat(true) . ":\n"
//						. $message->content . "\n----------\n\n";
//				}
//			}

				if (!$task->save()) {
					throw new SaveException($task);
				}

				if (!empty($ticket->contact_id)) {
					$contact = Contact::findById($ticket->contact_id);
					if ($contact) {
						Link::create($task, $contact);
					}
				}

				if (!empty($ticket->company_id)) {
					$contact = Contact::findById($ticket->company_id);
					if ($contact) {
						Link::create($task, $contact);
					}
				}

				$ticket->copyLinks($task);


				$messages = Message::model()->findByAttribute('ticket_id', $ticket->id, FindParams::newInstance()->select('*'))->fetchAll();
				$this->copyMessages($messages, $task);
			} catch (Exception $e) {
				ErrorHandler::logException($e, "Failed to migrate ticket ID: " . $ticket->id);
			}

			$i++;

			$this->progress( floor(($i * 100) / $total) );

		}
	}

	private function copyMessages(array $messages, Task $task)
	{

		foreach ($messages as $message) {

			if (empty($message->content)) {
				continue;
			}

			/** @var Message $message */

			/*
			* x @property string $rate_name
			* x @property string $rate_hours
			* x @property string $rate_amount
			* x @property string $rate_cost_code
			* x @property int $rate_id
			* x @property int $mtime
			* x @property int $ctime
			* x @property int $user_id
			* x @property boolean $is_note
			* @property string $attachments
			* x @property string $content
			* - @property boolean $has_status
			* - @property boolean $has_type
			* - @property int $status_id
			* - @property int $type_id
			* x @property int $ticket_id
			* - @property int $template_id
			* - @property int $id
				*/


			try {
				$comment = new Comment();
				$comment->setEntity($task);
				$comment->section = $message->is_note ? "private" : null;
				$comment->text = StringUtil::textToHtml($message->content);
				$comment->modifiedBy = User::exists($message->user_id) ? $message->user_id : 1;
				$comment->createdBy = User::exists($message->user_id) ? $message->user_id : ($message->user_id == 0 ? $task->createdBy : 1);
				$comment->createdAt = DateTime::createFromFormat("U", $message->mtime);
				$comment->modifiedAt = DateTime::createFromFormat("U", $message->mtime);
				$comment->date = DateTime::createFromFormat("U", $message->mtime);

				foreach ($message->getFiles() as $file) {
					if ($file->fsFile->exists()) {
						$att = new CommentAttachment($comment);
						$att->name = $file->name;
						$blob = Blob::fromFile(new File($file->fsFile->path()));
						$blob->save();
						$att->blobId = $blob->id;
						$comment->attachments[] = $att;
						//$file->delete();
					}
				}

				if (!$comment->save()) {
					throw new SaveException($comment);
				}
			} catch (Exception $e) {
				ErrorHandler::logException($e, "Failed to save comment for ticket ID: " . $task->id);
				go()->debug($comment->text);
			}

		}
	}


	private $typesMap = [];
	private $statusMap = [];


	/**
	 * Ticket types become task lists with role = 'support'
	 *
	 * @return void
	 * @throws SaveException
	 * @throws \go\core\http\Exception
	 */
	private function migrateTypes()
	{

		echo "Type groups\n";

		$typeGroups = TypeGroup::model()->find();
		foreach ($typeGroups as $typeGroup) {
			echo "Type group: " . $typeGroup->name . "\n";

			$tlg = TaskListGrouping::find()->where('name', '=', $typeGroup->name)->single();
			if (!$tlg) {
				$tlg = new TaskListGrouping();
				$tlg->name = $typeGroup->name;
				$tlg->save();
			}

			$typeGroupMap[$typeGroup->id] = $tlg->id;
		}


		echo "Types\n";
		$types = Type::model()->find();
		foreach ($types as $type) {
			echo "Type: " . $type->name . "\n";
			$tasklist = TaskList::find()->where(['role' => TaskList::Support, 'name' => $type->name])->single();
			if (!$tasklist) {
				$tasklist = new TaskList();
				$tasklist->name = $type->name;
				$tasklist->setRole('support');


				if ($type->type_group_id && isset($typeGroupMap[$type->type_group_id])) {
					$tasklist->groupingId = $typeGroupMap[$type->type_group_id];
				}

				if (!$tasklist->save()) {
					throw new SaveException($tasklist);
				}

				//copy permissions
				$sourceAcl = \go\core\model\Acl::findById($type->acl_id);
				/** @var Acl $acl */
				$targetAcl = $tasklist->findAcl();
				$targetAcl->groups = array_map(function ($g) {
					return $g->copy();
				}, $sourceAcl->groups);
				if (!$targetAcl->save()) {
					throw new SaveException($targetAcl);
				}
			}
			$this->typesMap[$type->id] = $tasklist->id;
		}
	}

	private function migrateStatuses()
	{
		echo "Statuses\n";
		$statuses = Status::model()->find();
		foreach ($statuses as $status) {
			echo "Status: " . $status->name . "\n";
			$category = Category::find()->where(['name' => $status->name])->single();
			if (!$category) {
				$category = new Category();
				$category->name = $status->name;
				if (!$category->save()) {
					throw new SaveException($category);
				}
			}

			$this->statusMap[$status->id] = $category->id;
		}
	}
}