import {commenteditor} from "./CommentEditor.js";
import {
	browser,
	btn,
	comp,
	DateTime,
	fieldset,
	Form,
	form,
	router,
	t,
	tbar,
	textfield,
	Window
} from "@intermesh/goui";
import {tasklistcombo} from "./TasklistCombo.js";
import {client, FilterCondition, JmapDataSource, jmapds} from "@intermesh/groupoffice-core";


export class SupportCreateWindow extends Window {
	constructor() {
		super();

		Object.assign(this, {
				title: t("Add"),
				width: 700,
				height: 500,
				modal: true,
				resizable: true,
				maximizable: true
			}
		);

		const taskListCombo = tasklistcombo({
			flex: 1
		});
		(taskListCombo.list.store.queryParams.filter as FilterCondition).forSupport = true;

		const commentEditor = commenteditor();

		this.items.add(
			form({
				flex: 1,
				cls: "vbox",
				handler: () => {
						this.send();
					}
			},

			fieldset({
					cls: "scroll",
					flex: 1
				},

				comp({cls: "hbox gap"},

					textfield({
						label: t("Subject"),
						name: "subject",
						required: true,
						flex: 2
					}),
					taskListCombo
					),

				commentEditor
			),

			tbar({},
				btn({
					type: "button",
					text: t("Attach files"),
					icon: "attach_file",
					handler: async () => {

						const files = await browser.pickLocalFiles(true);
						this.mask();
						const blobs = await client.uploadMultiple(files);
						this.unmask();


						const atts = commentEditor.attachments.value;
						for (let blob of blobs) {
							atts.push({
								name: blob.name,
								blobId: blob.id
							});
						}
						commentEditor.attachments.value = atts;

					}
				}),

				'->',

				btn({
					type: "submit",
					text: t("Send")
				})
			)))

			this.on("focus", () => this.items.get(0)!.focus());
	}

	private async send  () {
		const v = this.findChildByType(Form)!.value

		const now = new DateTime(), tomorrow = (new DateTime).setMonthDay(now.getMonthDay() + 1);

		const taskDS = jmapds("Task"),
			commentDS = jmapds("Comment");

		taskDS.commitBaseParams = {
			forSupport: true
		};

		try {
			const task = await taskDS.create({
				title: v.subject,
				start: now.format("c"),
				due: tomorrow.format("c"),
				tasklistId: v.tasklistId
			});

			delete v.subject;
			delete v.tasklistId;

			const comment = await commentDS.create(
				Object.assign({
					entity: "Task",
					entityId: task.id
				}, v)
			);

			void router.goto("supportclient/" + task.id);
			this.close();
		} catch(e:any) {
			if(e.type && e.type == 'stateMismatch') {
				Promise.all([
					taskDS.updateFromServer(),
					commentDS.updateFromServer()]
				).then(() => this.send());
			} else
			{
				console.error(e);
				Window.alert(t("Sorry, an error occured. Please try again."));
			}
		}


	};

}

