/**
 * Buffer or delay a function
 * @category Utility
 *
 */
export class BufferedFunction {
    /**
     * Constructor
     * @param delay Delay to execute function
     * @param fn Function to buffer
     */
    constructor(delay, fn) {
        this.delay = delay;
        this.fn = fn;
        this.delay = delay;
        this.fn = fn;
    }
    /**
     * Bugger the function with the delay set on this class
     *
     * @param args
     */
    buffer(args = []) {
        this.cancel();
        return new Promise(resolve => {
            this.id = window.setTimeout(() => {
                this.cancel();
                resolve(this.fn.apply(null, args));
            }, this.delay);
        });
    }
    /**
     * Cancel the function call
     */
    cancel() {
        if (this.id) {
            clearTimeout(this.id);
            this.id = undefined;
        }
    }
    /**
     * Check if it's still pending for execution
     */
    pending() {
        return this.id !== undefined;
    }
}
/**
 * Function utilities
 *
 * @category Utility
 */
export class FunctionUtil {
    /**
     * Buffer the function with this number of milliseconds.
     * When the function is called multiple times within the delay period only the last call will execute.
     *
     * @param delay
     * @param fn
     */
    static buffer(delay, fn) {
        const bf = new BufferedFunction(delay, fn);
        return (...args) => {
            return bf.buffer(args);
        };
    }
    /**
     *
     * Delay function execution
     *
     * @param delay
     * @param fn
     */
    static delay(delay, fn) {
        return (...args) => {
            const bf = new BufferedFunction(delay, fn);
            bf.buffer(args);
        };
    }
    /**
     * Execute on the next repaint
     *
     * The onRepaint method tells the browser that you wish to perform an animation and requests
     * that the browser calls a specified function to update an animation before the next repaint.
     *
     * @link https://developer.mozilla.org/en-US/docs/Web/API/window/requestAnimationFrame
     * @param fn
     */
    static onRepaint(fn) {
        let frame = -1;
        return (...args) => {
            if (frame > -1) {
                cancelAnimationFrame(frame);
            }
            frame = window.requestAnimationFrame(() => {
                fn.apply(null, args);
            });
        };
    }
    /**
     * Will combine the given functions into one.
     *
     * The newFn function will be called with the return value of origFn function + the parameters of origFn
     * and function will return the value of newFn
     *
     * @param origFn
     * @param newFn
     *
     * @example
     * ```
     * function test(a:number, b:string) : string {
     * 	return b;
     * }
     *
     * FunctionUtil.createSequence(test, function(r, a, b) {
     * 	return r + "b";
     * })
     * ```
     *
     */
    static createSequence(origFn, newFn) {
        return function (...args) {
            const r = origFn.apply(this, args);
            return newFn.call(this, r, ...args);
        };
    }
    /**
     * Create a combined function of an orignal and new function. The new function will be called
     * before the original,
     *
     * @param origFn
     * @param newFn
     *
     * @example
     * ```
     * export function playgroundTableOverride() {
     * 	PlaygroundTable.prototype.render = FunctionUtil.createInterceptor(
     * 		PlaygroundTable.prototype.render,
     * 		function(this:PlaygroundTable) {
     * 			this.el.classList.add("cls-added-by-override");
     * 		}
     * 	)
     * }
     * ```
     */
    static createInterceptor(origFn, newFn) {
        return function (...args) {
            newFn.apply(this, args);
            return origFn.apply(this, args);
        };
    }
}
//# sourceMappingURL=FunctionUtil.js.map