/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { DefaultEntity, Listener, Observable, ObservableEventMap } from "@intermesh/goui";
export interface LoginData {
    action?: "login";
    username?: string;
    password?: string;
    loginToken?: string;
    authenticators?: {
        otpauthenticator?: {
            code: string;
        } | undefined;
    };
}
type User = DefaultEntity;
export interface RegisterData {
    action: "register";
    user: Partial<User>;
}
export interface ForgottenData {
    action: "forgotten";
    email: String;
}
interface ClientEventMap<Type extends Observable> extends ObservableEventMap<Type> {
    authenticated: <Sender extends Type>(client: Sender, session: any) => void;
    logout: <Sender extends Type>(client: Sender) => void;
}
export interface Client {
    on<K extends keyof ClientEventMap<this>, L extends Listener>(eventName: K, listener: Partial<ClientEventMap<this>>[K]): L;
    un<K extends keyof ClientEventMap<this>>(eventName: K, listener: Partial<ClientEventMap<this>>[K]): boolean;
    fire<K extends keyof ClientEventMap<Client>>(eventName: K, ...args: Parameters<NonNullable<ClientEventMap<any>[K]>>): boolean;
}
type UploadResponse = {
    id: string;
    size: number;
    type: string;
    name: string;
    file: File;
    subfolder: string | undefined;
};
/**
 * Result reference
 *
 * @link https://jmap.io/spec-core.html#references-to-previous-method-results
 */
export interface ResultReference {
    /**
     * The method call id (see Section 3.1.1) of a previous method call in the current request.
     */
    resultOf: string;
    /**
     * The required name of a response to that method call.
     */
    name: string;
    /**
     * A pointer into the arguments of the response selected via the name and resultOf properties. This is a JSON
     * Pointer [@!RFC6901], except it also allows the use of * to map through an array (see the description below).
     *
     */
    path: string;
}
export declare class Client<UserType extends User = User> extends Observable {
    private _lastCallCounter;
    private _lastCallId?;
    private _requests;
    private _requestData;
    private _session;
    private timeout?;
    private debugParam;
    private user;
    uri: string;
    private CSRFToken;
    /**
     * Either a cookie + CSRFToken are used when the API is on the same site. If it's not then an access token can be used
     *
     * @private
     */
    private accessToken;
    private delayedJmap;
    private SSEABortController?;
    private pollInterval?;
    private SSEEventsRegistered;
    private SSELastEntities?;
    constructor();
    set session(session: any);
    /**
     * this should be firing on set session() but in GO we first have to load custom fields and modules before this fires.
     */
    fireAuth(): void;
    get session(): any;
    /**
     * The ID of the last JMAP method call
     */
    get lastCallId(): string | undefined;
    isLoggedIn(): Promise<User | false>;
    private request;
    logout(): Promise<void>;
    private static blobCache;
    getBlobURL(blobId: string): Promise<any>;
    downloadBlobId(blobId: string, filename: string): Promise<void>;
    auth(data: LoginData | RegisterData | ForgottenData): Promise<Response>;
    /**
     * Get the logged-in user.
     */
    getUser(): Promise<UserType | undefined>;
    downloadUrl(blobId: string): string;
    pageUrl(path: string): string;
    private getDefaultHeaders;
    private buildHeaders;
    /**
     * Upload a file to the API
     *
     * @todo Progress. Not possible ATM with fetch() so we probably need XMLHttpRequest()
     * @param file
     */
    upload(file: File): Promise<UploadResponse>;
    /**
     * Upload multiple files to the API
     *
     * @example
     * ```
     * btn({
     * 	type: "button",
     * 	text: t("Attach files"),
     * 	icon: "attach_file",
     * 	handler: async () => {
     *
     * 		const files = await browser.pickLocalFiles(true);
     * 		this.mask();
     * 		const blobs = await client.uploadMultiple(files);
     * 		this.unmask();
     * 	  console.warn(blobs);
     *
     * 	}
     * })
     * ```
     * @param files
     */
    uploadMultiple(files: File[]): Promise<UploadResponse[]>;
    /**
     * Execute JMAP method
     *
     * Multiple calls will be joined together in a single call on the next event loop
     *
     * @param method
     * @param params
     */
    jmap(method: string, params?: Object, callId?: string | undefined): Promise<any>;
    /**
     * Performs the requests queued in the jmap() method
     *
     * @private
     */
    private doJmap;
    /**
     * When SSE is disabled we'll poll the server for changes every 2 minutes.
     * This also keeps the token alive. Which expires in 30M.
     */
    updateAllDataSources(entities: string[]): void;
    startPolling(entities: string[]): void;
    stopSSE(): void;
    /**
     * Initializes Server Sent Events via EventSource. This function is called in MainLayout.onAuthenticated()
     *
     * Note: disable this if you want to use xdebug because it will crash if you use SSE.
     *
     * @returns {Boolean}
     */
    startSSE(entities: string[]): Promise<false | undefined>;
    private registerSSEEvents;
}
export declare const client: Client<DefaultEntity>;
export {};
//# sourceMappingURL=Client.d.ts.map