/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Component, ComponentEventMap } from "../Component.js";
import { Listener, ObservableListenerOpts } from "../Observable.js";
import { Button } from "../Button.js";
import { MaterialIcon } from "../MaterialIcon";
import { Menu } from "../menu";
/**
 * @inheritDoc
 */
export interface FieldEventMap<Type> extends ComponentEventMap<Type> {
    /**
     * Fires when the field changes. It fires on blur.
     *
     * Note: this event does not fire on {@see ContainerField} and {@see Form}
     *
     * @param field
     */
    change: (field: Type, newValue: any, oldValue: any) => void;
    /**
     * Fires when setValue() is called
     *
     * @param field
     * @param newValue
     * @param oldValue
     */
    setvalue: (field: Type, newValue: any, oldValue: any) => void;
    /**
     * Fires when field is reset
     *
     * @param field
     * @param newValue
     * @param oldValue
     */
    reset: (field: Type, newValue: any, oldValue: any) => void;
    /**
     * Fires when validated
     *
     * Use {@see setInvalid()} to mark field invalid
     *
     * @param field
     */
    validate: (field: Type) => void;
    /**
     * Fires when the field is invalid
     *
     * Use {@see setInvalid()} to mark field invalid
     *
     * @param field
     */
    invalid: (field: Type) => void;
}
export interface Field extends Component {
    on<K extends keyof FieldEventMap<this>, L extends Listener>(eventName: K, listener: Partial<FieldEventMap<this>>[K], options?: ObservableListenerOpts): L;
    un<K extends keyof FieldEventMap<this>>(eventName: K, listener: Partial<FieldEventMap<this>>[K]): boolean;
    fire<K extends keyof FieldEventMap<this>>(eventName: K, ...args: Parameters<FieldEventMap<any>[K]>): boolean;
}
/**
 * Base class for a form field
 *
 * Field components should at least implement "createControl" and "internalSetValue".
 */
export declare abstract class Field extends Component {
    private _buttons?;
    private toolbar?;
    private _wrap?;
    protected _labelEl?: HTMLElement;
    private _icon;
    private iconEl?;
    constructor(tagName?: keyof HTMLElementTagNameMap);
    readonly isFormField = true;
    /**
     * Adds standard style. You may want to remove this if you don't want the standard
     * look of a form field.
     *
     * @protected
     */
    protected baseCls: string;
    private _name?;
    private _required;
    private _readOnly;
    private _label;
    protected _value: any;
    protected control: HTMLElement | undefined;
    /**
     * The value this field resets to when a form is reset.
     * Changes when a form loads.
     * @protected
     */
    protected resetValue: any;
    /**
     * The value that was set before adding this component to a parent.
     *
     * @protected
     */
    protected defaultValue: any;
    invalidMsg: string;
    private _hint;
    private hintEl?;
    /**
     * Used for "change" event
     * @protected
     */
    protected valueOnFocus?: string;
    /**
     * Validate the field on blur
     */
    validateOnBlur: boolean;
    /**
     * Fires a change event if the field's value is different since it got focus
     * @protected
     */
    protected fireChangeOnBlur: boolean;
    protected onAdded: (comp: Field, index: number, parent: Component) => void;
    protected onFocusOut(e: FocusEvent): void;
    /**
     * Return true if the field was modified
     */
    protected isChangedSinceFocus(): boolean;
    protected onFocusIn(e: FocusEvent): void;
    protected captureValueForChange(): void;
    protected internalRender(): HTMLElement;
    isFocusable(): boolean;
    /**
     * A wrapper DIV element that contains input and toolbar for input buttons like an expand button for a drop down
     */
    get wrap(): HTMLDivElement;
    protected renderControl(): void;
    private renderButtons;
    /**
     * When buttons with menus are added it is handy to delay the validation on blur.
     * Because when the user will click something in the menu it will blur the field and you probably don't
     * want it to validate at that point. It's important that the menu will return focus to the field
     * and sets the value afterward.
     *
     * @param menu
     * @protected
     */
    protected setupMenu(menu: Menu): void;
    protected createControl(): HTMLElement | undefined;
    /**
     * Render buttons inside the text field
     *
     * @example
     * ```
     * buttons: [
     * 				 		btn({icon: "clear", handler:(btn) => (btn.parent!.parent! as Field).value = ""})
     * 					]
     * ```
     * @param buttons
     */
    set buttons(buttons: Button[] | undefined);
    get buttons(): Button[] | undefined;
    protected createHint(): HTMLDivElement | void;
    protected createLabel(): HTMLElement | void;
    private getLabelText;
    /**
     * Form element name which will be the key in values
     * If omitted the field won't be included in the form values.
     */
    get name(): string;
    /**
     * The field's name
     */
    set name(name: string);
    get required(): boolean;
    /**
     * Required or not
     */
    set required(required: boolean);
    get label(): string;
    /**
     * The field's label
     */
    set label(label: string);
    get icon(): MaterialIcon | "" | undefined;
    /**
     * The field's icon rendered at the left inside the field
     */
    set icon(icon: MaterialIcon | "" | undefined);
    get hint(): string;
    /**
     * The field's hint text
     */
    set hint(hint: string);
    get readOnly(): boolean;
    /**
     * Make the field read only
     */
    set readOnly(readOnly: boolean);
    /**
     * Check if the field was modified since create or when a form was loaded and @see trackReset() was called.
     */
    isModified(): boolean;
    /**
     * Copies the current value to the reset value. Typically happens when this component was added to a parent and
     * when the form it belongs too loads.
     *
     * @see Form in the trackModifications method
     */
    trackReset(): void;
    /**
     * Set the field value
     */
    set value(v: any);
    /**
     * Applies set value to the control.
     *
     * This is also called when the control is rendered. Note that this.rendered is still false when that happens.
     *
     * @param v
     * @protected
     */
    protected internalSetValue(v?: any): void;
    /**
     * Helper to fire "change" event. Child classes must implement this.
     */
    protected fireChange(): void;
    get value(): any;
    /**
     * Reset the field value to the value that was given to the field's constructor
     * @see setValue()
     */
    reset(): void;
    /**
     * Set the field as invalid and set a message
     *
     * @param msg
     */
    setInvalid(msg: string): void;
    /**
     * Check if the field is empty
     */
    isEmpty(): boolean;
    protected validate(): void;
    protected setValidityState(input: HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement): void;
    protected applyInvalidMsg(): void;
    /**
     * Clears the invalid state
     */
    clearInvalid(): void;
    /**
     * Checks if the field is valid
     */
    isValid(): boolean;
    private createIcon;
}
//# sourceMappingURL=Field.d.ts.map