/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Component, ComponentState } from "../Component.js";
import { Store, StoreRecord } from "../../data/Store.js";
import { TableColumn } from "./TableColumns.js";
import { List, ListEventMap } from "../List.js";
import { Config, Listener, ObservableListenerOpts } from "../Observable";
type GroupByRenderer = (groupBy: any, record: any, thEl: HTMLTableCellElement, table: Table) => string | Promise<string> | Component | Promise<Component>;
export interface Table<StoreType extends Store = Store> extends List<StoreType> {
    on<K extends keyof ListEventMap<this>, L extends Listener>(eventName: K, listener: Partial<ListEventMap<this>>[K], options?: ObservableListenerOpts): L;
    un<K extends keyof ListEventMap<this>>(eventName: K, listener: Partial<ListEventMap<this>>[K]): boolean;
    fire<K extends keyof ListEventMap<this>>(eventName: K, ...args: Parameters<ListEventMap<any>[K]>): boolean;
}
/**
 * Table component
 *
 * @example
 * ```
 * const records:StoreRecord[] = [];
 *
 * 	for(let i = 1; i <= 100; i++) {
 * 		records.push({number: i, description: "Test " + i, createdAt: (new DateTime()).format("c")});
 * 	}
 *
 * 	const table = table({
 * 		store: const store = jmapstore({
 * 			entity: "TaskList",
 * 			properties: ['id', 'name', 'support'],
 * 			queryParams: {
 * 				limit: 20,
 * 				filter: {
 * 					forSupport: true,
 * 					role: "support", //support tasklists
 * 				}
 * 			},
 * 			sort: [{property: "name", isAscending: true}]
 * 		}),
 * 		cls: "fit",
 * 		columns: [
 * 			{
 * 				header: "Index",
 * 				id: "id",
 * 				renderer: (value, record, td, table) => {
 * 					return table.getStore().findRecordIndex(r => record.number == r.number).toString();
 * 				},
 * 				resizable: true,
 * 				width: 60,
 * 				sortable: false
 * 			},
 * 			{
 * 				header: "Number",
 * 				id: "number",
 * 				sortable: true,
 * 				resizable: true,
 * 				width: 200
 * 			},
 * 			{
 * 				header: "Description",
 * 				id: "description",
 * 				sortable: true,
 * 				resizable: true,
 * 				width: 300
 * 			},
 * 			datecolumn({
 * 				header: "Created At",
 * 				id: "createdAt",
 * 				sortable: true
 * 			})
 * 			]
 * 	});
 *  ```
 */
export declare class Table<StoreType extends Store = Store> extends List<StoreType> {
    readonly store: StoreType;
    columns: TableColumn[];
    /**
     *
     * @param store Store to provide data
     * @param columns The table columns
     */
    constructor(store: StoreType, columns: TableColumn[]);
    /**
     * Make the table fits its container in width by setting min-width: 100%
     * Defaults to true
     */
    fitParent: boolean;
    /**
     * Show headers
     */
    headers: boolean;
    /**
     * Group the table by this property.
     */
    groupBy?: string;
    protected emptyStateTag: keyof HTMLElementTagNameMap;
    /**
     * Group renderer function
     */
    groupByRenderer: GroupByRenderer;
    private minCellWidth;
    protected baseCls: string;
    private headersRow?;
    protected itemTag: keyof HTMLElementTagNameMap;
    protected getRowElements(): HTMLElement[];
    protected internalRemove(): void;
    protected restoreState(state: ComponentState): void;
    /**
     * Find column by "property" property.
     *
     * It's the property path of the data linked to the column
     *
     * @param id
     */
    findColumnById(id: string): TableColumn | undefined;
    protected buildState(): ComponentState;
    protected renderBody(): void;
    private rerender;
    private columnMenu;
    private showColumnMenu;
    private createColumnSplitter;
    private renderColGroup;
    private renderHeaders;
    private onSort;
    private colsAreFixed;
    /**
     * When resizing columns we must calculate absolute pixel widths
     *
     * @private
     */
    private fixColumnWidths;
    /**
     * Returns the sum of column widths
     *
     * @param untilColumnIndex Calc width until this column
     */
    private calcTableWidth;
    private lastGroup?;
    private groupEl?;
    protected renderGroup(record: any): HTMLElement;
    protected onRecordRemove(collection: StoreType, item: StoreRecord, index: number): void;
    protected findDropRow(e: DragEvent): HTMLElement;
}
export type TableConfig<TableType extends Table = Table> = Omit<Config<TableType, ListEventMap<TableType>, "store" | "columns">, "rowSelection">;
/**
 * Shorthand function to create {@see Table}
 *
 * @param config
 */
export declare const table: <StoreType extends Store<StoreRecord> = Store<StoreRecord>>(config: TableConfig<Table<StoreType>>) => Table<StoreType>;
export {};
//# sourceMappingURL=Table.d.ts.map