/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Config, Listener, Observable, ObservableEventMap, ObservableListener, ObservableListenerOpts } from "./Observable.js";
import { Collection } from "../util";
/**
 * A component identifier by id, itemId, Component instance or custom function
 */
export type FindComponentPredicate = string | Component | ((comp: Component) => boolean | void);
type ClassTypeOf<T> = abstract new (...args: any[]) => T;
export declare const REM_UNIT_SIZE: number;
export interface ComponentEventMap<Type> extends ObservableEventMap<Type> {
    /**
     * Fires when the component renders and is added to the DOM
     *
     * @see Component.render()
     * @param comp
     */
    render: (comp: Type) => void;
    /**
     * Fires just before rendering
     *
     * @see Component.render()
     * @param comp
     */
    beforerender: (comp: Type) => void;
    /**
     * Fires before the element is removed. You can cancel the remove by returning false
     *
     * @see Component.remove()
     * @param comp
     */
    beforeremove: (comp: Type) => false | void;
    /**
     * Fires after the component has been removed
     *
     * @see Component.remove()
     * @param comp
     */
    remove: (comp: Type) => void;
    /**
     * Fires before show. You can cancel the show by returning false
     *
     * @see Component.show()
     * @param comp
     */
    beforeshow: (comp: Type) => false | void;
    /**
     * Fires after showing the component
     *
     * @see Component.show()
     * @param comp
     */
    show: (comp: Type) => void;
    /**
     * Fires before hide. You can cancel the hide by returning false.
     *
     * @see Component.hide()
     * @param comp
     */
    beforehide: (comp: Type) => false | void;
    /**
     * Fires after hiding the component
     *
     * @see Component.show()
     * @param comp
     */
    hide: (comp: Type) => void;
    /**
     * Fires on focus
     *
     * @param comp
     * @param o
     */
    focus: (comp: Type, o?: FocusOptions) => void;
    /**
     * Fires when this component is added to a parent but before rendering
     *
     * @param me
     * @param index the index in the parents' items
     */
    added: (comp: Type, index: number, parent: Component) => void;
}
export interface Component extends Observable {
    on<K extends keyof ComponentEventMap<Component>, L extends Listener>(eventName: K, listener: Partial<ComponentEventMap<Component>>[K], options?: ObservableListenerOpts): L;
    un<K extends keyof ComponentEventMap<this>>(eventName: K, listener: Partial<ComponentEventMap<this>>[K]): boolean;
    fire<K extends keyof ComponentEventMap<Component>>(eventName: K, ...args: Parameters<ComponentEventMap<any>[K]>): boolean;
}
/**
 * State object for saving and restoring state of component in browser storage for example
 */
export type ComponentState = Record<string, any>;
/**
 * Component
 *
 * A component in its simplest form.
 *
 * @example
 *
 * ```typescript
 * Component.create({
 *   tagName: "hr",
 *   cls: "special"
 * })
 * ```
 */
export declare class Component extends Observable {
    readonly tagName: keyof HTMLElementTagNameMap;
    protected _cls?: string;
    private maskTimeout?;
    /**
     * Component constructor
     *
     * @param tagName The tagname used for the root HTMLElement of this component
     */
    constructor(tagName?: keyof HTMLElementTagNameMap);
    /**
     * A base class not configurable. cls can be used to add extra classes leaving this class alone
     *
     * @protected
     */
    protected baseCls: string;
    /**
     * True when added to the DOM tree
     *
     * @private
     */
    private _rendered;
    /**
     * Component item ID that can be used to lookup the Component inside a Component with Component.findItem() and
     * Component.findItemIndex();
     *
     * if stateId is given it will also be used as itemId
     */
    private _itemId;
    /**
     * ID used for storing state of the component in the State storage.
     *
     * If stateId is given it will also be used as itemId
     *
     * If not set then the component won't store it's state.
     */
    stateId?: string;
    /**
     * When this item is added to a Component this is set to the parent Component
     */
    parent?: Component;
    /**
     * Normally components are rendered to its parent component's element. But in some cases like menu's it's desired
     * to render them to the root and position them absolute.
     */
    renderTo?: HTMLElement;
    private _items?;
    private _mask;
    /**
     * Set arbitrary data on a component.
     *
     * Should be used with caution as this data is not typed.
     */
    readonly dataSet: Record<string, any>;
    /**
     * Component item ID that can be used to lookup the Component inside a Component with Component.findItem() and
     * Component.findItemIndex();
     *
     * if stateId is given it will also be used as itemId
     */
    get itemId(): string;
    set itemId(itemId: string);
    private initItems;
    protected getState(): Record<string, any>;
    protected hasState(): boolean | "" | undefined;
    /**
     * Restore state of the component in this function. It's called before render in init().
     *
     * @see saveState();
     * @param state
     *
     * @protected
     */
    protected restoreState(state: ComponentState): void;
    /**
     * Call save start when something relevant to the state changes.
     * Implement buildState() to save relevant state properties and restore it in restoreState()
     *
     * stateId must be set on components to be stateful
     *
     * @protected
     */
    protected saveState(): void;
    /**
     * Build state for the component
     *
     * @see saveState();
     * @protected
     */
    protected buildState(): ComponentState;
    /**
     * Title of the dom element
     */
    set title(title: string);
    get title(): string;
    /**
     * Check if the component has been rendered and added to the DOM tree
     */
    get rendered(): boolean;
    private _el?;
    get el(): HTMLElement;
    /**
     * Class name to add to element
     *
     * Some common classes to add for layout:
     *
     * - hbox: Set's flex layout to horizontal boxes. Use flex: n to stretch columns
     * - vbox: As above but vertical
     * - fit: Fit the parent's size
     * - scroll: Set's autoscroll: true
     * - pad: Set common padding on the element
     * - border-(top|bottom|left|right) to add a border
     *
     * Other:
     *
     * - goui-fade-in: The component will fade in when show() is used.
     * - goui-fade-out: The component will fade out when hide is used.
     *
     */
    set cls(cls: string);
    get cls(): string;
    private initClassName;
    /**
     * Renders the component and it's children
     */
    protected internalRender(): HTMLElement;
    /**
     * The tabindex attribute specifies the tab order of an element (when the "tab" button is used for navigating).
     */
    set tabIndex(tabIndex: number);
    get tabIndex(): number;
    /**
     * CSS flex value
     */
    set flex(flex: number | string);
    /**
     * CSS flex value
     */
    get flex(): number | string;
    /**
     * Make it resizable
     */
    set resizable(resizable: boolean);
    get resizable(): boolean;
    /**
     * Render the component
     *
     * @param parentEl The element this componennt will render into
     * @param insertBefore If given, the element will be inserted before this child
     */
    render(parentEl?: Node, insertBefore?: Node): HTMLElement;
    /**
     * Finds the DOM node in the parent's children to insert before when rendering a child component
     *
     * @protected
     */
    protected getInsertBefore(): Node | undefined;
    /**
     * Remove component from the component tree
     */
    remove(): boolean;
    protected internalRemove(): void;
    /**
     * Hide element
     */
    set hidden(hidden: boolean);
    /**
     * Overridable method so that the (before)show/hide event fire before and after.
     *
     * @param hidden
     * @protected
     */
    protected internalSetHidden(hidden: boolean): void;
    get hidden(): boolean;
    /**
     * Hide this component
     *
     * This sets the "hidden" attribute on the DOM element which set's CSS display:none.
     * You can change this to fade with css class "goui-fade-in" and "goui-fade-out"
     *
     * If you want to override this function, please override internalSetHidden instead so the beforeshow and show event
     * fire at the right time.
     */
    hide(): boolean;
    /**
     * Show the component
     *
     * If you want to override this function, please override internalSetHidden instead so the beforeshow and show event
     * fire at the right time.
     */
    show(): boolean;
    /**
     * Disable component
     */
    set disabled(disabled: boolean);
    get disabled(): boolean;
    /**
     * Set the HTML contents of the component (innerHTML)
     */
    set html(html: string);
    get html(): string;
    /**
     * Set the innerText
     */
    set text(text: string);
    get text(): string;
    /**
     * Set the width in scalable pixels
     *
     * The width is applied in rem units divided by 10. Because the font-size of the html
     * element has a font-size of 62.5% this is equals the amount of pixels, but it can be
     * scaled easily for different themes.
     *
     */
    set width(width: number);
    get width(): number;
    /**
     * Set inline style
     */
    set style(style: Partial<CSSStyleDeclaration>);
    get style(): Partial<CSSStyleDeclaration>;
    computeZIndex(): number;
    /**
     * The height in scalable pixels
     *
     * @see width
     */
    set height(height: number);
    get height(): number;
    /**
     * Element ID
     */
    set id(id: string);
    get id(): string;
    /**
     * Focus the component
     *
     * @param o
     */
    focus(o?: FocusOptions): void;
    isFocusable(): boolean;
    /**
     * Get the component that's next to this one
     */
    nextSibling(): Component | undefined;
    /**
     * Get the component that's previous to this one
     */
    previousSibling(): Component | undefined;
    /**
     * Find ancestor
     *
     * The method traverses the Component's ancestors (heading toward the document root) until it finds
     * one where the given function returns true.
     *
     * @param fn When the function returns true the item will be returned. Otherwise it will move up to the next parent.
     */
    findAncestor(fn: (cmp: Component) => void | boolean): Component | undefined;
    /**
     * Find parent by instance type of the parent
     *
     * @example
     * ```
     * const form = textField.findAncestorByType(Form);
     * ```
     * @param cls
     */
    findAncestorByType<T extends ClassTypeOf<Component>>(cls: T): InstanceType<T> | undefined;
    /**
     * The child components of this component
     */
    get items(): Collection<Component>;
    protected renderItems(): void;
    protected get itemContainerEl(): HTMLElement;
    /**
     * Can be overriden to wrap the component
     *
     * @param item
     * @protected
     */
    protected renderItem(item: Component): void;
    /**
     * Find the item by element ID, itemId property, Component instance or custom function
     */
    findItemIndex(predicate: FindComponentPredicate): number;
    /**
     * Find the item by element ID, itemId property, Component instance or custom function.
     *
     * If you want to search the component tree hierarchy use {@see findChild()}
     *
     */
    findItem(predicate: FindComponentPredicate): Component | undefined;
    /**
     * Cascade down the component hierarchy
     *
     * @param fn When the function returns false then the cascading will be stopped. The current Component will be finished!
     */
    cascade(fn: (comp: Component) => boolean | void): this;
    private createFindPredicateFunction;
    /**
     * Find a child at any level by element ID, itemId property, Component instance or custom function.
     *
     * It cascades down the component hierarchy. See also {@see findChildByType}
     *
     */
    findChild(predicate: FindComponentPredicate): Component | undefined;
    /**
     * Find children at any level by element ID, itemId property, Component instance or custom function.
     *
     * It cascades down the component hierarchy. See also {@see findChildByType}
     *
     */
    findChildren(predicate: FindComponentPredicate): Component[];
    /**
     * Find child by instance type of the parent
     *
     * @example
     * ```
     * const form = textField.findAncestorByType(Form);
     * ```
     * @param cls
     */
    findChildByType<T extends Component>(cls: ClassTypeOf<T>): T | undefined;
    /**
     * Find children by instance type of the parent
     *
     * @example
     * ```
     * const form = textField.findAncestorByType(Form);
     * ```
     * @param cls
     */
    findChildrenByType<T extends Component>(cls: ClassTypeOf<T>): T[];
    /**
     * Set attributes of the DOM element
     *
     * @param attr
     */
    set attr(attr: Record<string, string>);
    /**
     * Mask the component to disable user interaction
     * It creates an absolute positioned Mask
     * component. This component should have a non-static position style for this to work.
     */
    mask(delay?: number): void;
    /**
     * Unmask the body
     */
    unmask(): void;
    private static _uniqueID;
    static uniqueID(): string;
    /**
     * Print this component. Everything else will be left out.
     */
    print(): void;
}
/**
 * Mask element
 *
 * Shows a mask over the entire (position:relative) element it's in.
 *
 * Used in {@see Body.mask()}
 */
export declare class Mask extends Component {
    protected baseCls: string;
    /**
     * Show loading spinner
     */
    set spinner(spinner: boolean);
}
/**
 * Shorthand function to create a {@see Mask} component
 *
 * @param config
 */
export declare const mask: (config?: Config<Mask>) => Mask;
/**
 * Shorthand function to create {@see Component}
 */
export declare const comp: (config?: Config<Component>, ...items: Component[]) => Component;
export declare const p: (config?: Config<Component> | string, ...items: Component[]) => Component;
export declare const small: (config?: Config<Component> | string, ...items: Component[]) => Component;
export declare const h1: (config?: Config<Component> | string, ...items: Component[]) => Component;
export declare const h2: (config?: Config<Component> | string, ...items: Component[]) => Component;
export declare const h3: (config?: Config<Component> | string, ...items: Component[]) => Component;
export declare const h4: (config?: Config<Component> | string, ...items: Component[]) => Component;
export declare const code: (config?: Config<Component> | string, ...items: Component[]) => Component;
export declare const section: (config?: Config<Component> | string, ...items: Component[]) => Component;
export declare const hr: (config?: Config<Component>) => Component;
export declare const img: (config: Config<Component> & {
    src: string;
    alt?: string;
}) => Component;
export declare const a: (config: Config<Component> & {
    href?: string;
    target?: string;
}, ...items: Component[]) => Component;
export declare const progress: (config?: Config<Component>) => Component;
export type BaseConfig = {
    listeners?: ObservableListener<any>;
};
export declare const createComponent: <T extends Observable, C extends BaseConfig>(comp: T, config?: C | undefined, items?: Component[]) => T;
export {};
//# sourceMappingURL=Component.d.ts.map