/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { TextField } from "./TextField.js";
import { FunctionUtil } from "../../util/FunctionUtil.js";
import { btn } from "../Button.js";
import { createComponent } from "../Component.js";
import { listpicker } from "../picker";
import { Menu, menu } from "../menu";
/**
 * Autocomplete field
 */
export class AutocompleteField extends TextField {
    /**
     *
     * @param list The table to use for suggestions
     * @param buffer Buffer typing in the input in ms
     */
    constructor(list, buffer = 300) {
        super();
        this.list = list;
        this.buffer = buffer;
        this.autocomplete = "off";
        this.baseCls += " autocomplete";
        this.picker = listpicker({
            list: list
        });
        this.picker.on("select", (tablePicker, record) => {
            tablePicker.list.findAncestorByType(Menu).hide();
            this.focus();
            //set value after focus for change event
            this.value = this.pickerRecordToValue(this, record);
            this.fire('select', this, record);
        });
        this.menu = menu({
            height: 300,
            cls: "scroll",
            listeners: {
                hide: (menu) => {
                    if (menu.rendered) {
                        const textfield = menu.findAncestorByType(TextField);
                        textfield.focus();
                    }
                }
            }
        }, this.picker);
        this.menuButton = btn({
            icon: "expand_more",
            type: "button",
            handler: () => {
                this.fire("autocomplete", this, "");
            },
            menu: this.menu
        });
    }
    /**
     * Method that transforms a record from the TablePicker store to a value for this field.
     * This is not necessarily a text value. In conjunction with {@see valueToTextField()} this
     * could also be an ID of an object for example.
     *
     * @param field
     * @param record
     */
    pickerRecordToValue(field, record) {
        return record.id;
    }
    /**
     * This method transforms the value in to a text representation for the input field
     *
     * @param field
     * @param value
     */
    async valueToTextField(field, value) {
        return "";
    }
    internalSetValue(v) {
        if (v == undefined) {
            return super.internalSetValue(v);
        }
        this.valueToTextField(this, v + "").then(v => {
            if (this.input) {
                super.internalSetValue(v);
            }
            else {
                this.on("render", () => {
                    super.internalSetValue(v);
                }, { once: true });
            }
        });
    }
    get value() {
        return this._value;
    }
    set value(v) {
        super.value = v;
    }
    internalRender() {
        this.buttons = this.buttons || [];
        this.buttons.push(this.menuButton);
        const el = super.internalRender();
        this.menu.alignTo = this.wrap;
        this.menu.alignToInheritWidth = true;
        this.input.addEventListener('input', FunctionUtil.buffer(this.buffer, this.onInput.bind(this)));
        this.input.addEventListener('keydown', (ev) => {
            switch (ev.key) {
                case 'Enter':
                    if (!this.menu.hidden) {
                        ev.preventDefault();
                        this.picker.onSelect();
                    }
                    break;
                case 'ArrowDown':
                    ev.preventDefault();
                    this.fire("autocomplete", this, this.input.value);
                    this.menuButton.showMenu();
                    this.list.focus();
                    break;
                case 'Escape':
                    if (!this.menu.hidden) {
                        this.menu.hide();
                        ev.preventDefault();
                        ev.stopPropagation();
                        this.focus();
                    }
                    break;
            }
        });
        return el;
    }
    onInput(ev) {
        this.menuButton.showMenu();
        this.fire("autocomplete", this, this.input.value);
    }
}
/**
 * Shorthand function to create an {@see AutocompleteField}
 *
 * @param config
 */
export const autocomplete = (config) => createComponent(new AutocompleteField(config.list), config);
//# sourceMappingURL=AutocompleteField.js.map