/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Form } from "./Form";
import { t } from "../../Translate";
import { createComponent } from "../Component";
import { Window } from "../Window";
export class DataSourceForm extends Form {
    constructor(dataSource) {
        super();
        this.dataSource = dataSource;
        this.handler = async (form1) => {
            var _a;
            try {
                let data, v = this.currentId ? this.modified : this.value;
                this.fire('beforesave', this, v);
                if (!this.isNew) {
                    data = await this.dataSource.update(this.currentId, v);
                }
                else {
                    data = await this.dataSource.create(v);
                }
                if (data) {
                    this.fire('save', this, data, this.isNew);
                }
            }
            catch (e) {
                if (e.type == "invalidProperties") {
                    this.handleServerValidation(e);
                    return;
                }
                console.error(t("Error"), e);
                if (this.fire('saveerror', this, e) !== false) {
                    void Window.error((_a = e.message) !== null && _a !== void 0 ? _a : t("Unknown error"));
                }
            }
            finally {
                this.unmask();
            }
        };
    }
    handleServerValidation(error) {
        for (const propertyName in error.validationErrors) {
            const field = this.findField(propertyName);
            if (!field) {
                continue;
            }
            field.setInvalid(error.validationErrors[propertyName].description);
        }
        const invalid = this.findFirstInvalid();
        if (invalid) {
            invalid.focus();
        }
        this.setInvalid(t('You have errors in your form. The invalid fields are marked.'));
    }
    create(data) {
        this.reset();
        //this.currentId = '_new_';
        this.fire('load', this, data);
        if (data) {
            this.value = data;
        }
    }
    reset() {
        super.reset();
        delete this.currentId;
    }
    get isNew() {
        return !this.currentId;
    }
    /**
     * Load an entity into the form
     *
     * @param id
     */
    async load(id) {
        this.mask();
        try {
            this.currentId = id;
            let entity = await this.dataSource.single(id);
            if (!entity) {
                throw "Failed to load entity with id " + id;
            }
            this.fire('load', this, entity);
            this.value = entity;
        }
        catch (e) {
            console.error(t("Error"), e);
            if (this.fire('loaderror', this, e) !== false) {
                void Window.error(e.message);
            }
        }
        finally {
            this.unmask();
        }
    }
}
/**
 * Shorthand function to create {@see DataSourceForm}
 *
 * @param config
 * @param items
 */
export const datasourceform = (config, ...items) => createComponent(new DataSourceForm(config.dataSource), config, items);
//# sourceMappingURL=DataSourceForm.js.map