/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Component } from "../Component.js";
import { tbar } from "../Toolbar.js";
import { t } from "../../Translate.js";
import { E } from "../../util/Element.js";
/**
 * Base class for a form field
 *
 * Field components should at least implement "createControl" and "internalSetValue".
 */
export class Field extends Component {
    constructor(tagName = "label") {
        super(tagName);
        this.isFormField = true;
        /**
         * Adds standard style. You may want to remove this if you don't want the standard
         * look of a form field.
         *
         * @protected
         */
        this.baseCls = "goui-form-field";
        this._required = false;
        this._readOnly = false;
        this._label = "";
        this.invalidMsg = "";
        this._hint = "";
        /**
         * Validate the field on blur
         */
        this.validateOnBlur = true;
        /**
         * Fires a change event if the field's value is different since it got focus
         * @protected
         */
        this.fireChangeOnBlur = true;
        this.onAdded = (comp, index, parent) => {
            this.trackReset();
            this.defaultValue = this.value;
        };
        this.on("added", this.onAdded, { once: true });
        this.control = this.createControl();
    }
    onFocusOut(e) {
        if (e.relatedTarget instanceof HTMLElement && this.el.contains(e.relatedTarget)) {
            //focus is still within this field
            return;
        }
        if (this.validateOnBlur) {
            this.validate();
        }
        // detect changed value. Handle objects by comparing JSON values
        if (this.fireChangeOnBlur && this.isChangedSinceFocus()) {
            this.fireChange();
        }
    }
    /**
     * Return true if the field was modified
     */
    isChangedSinceFocus() {
        // detect changed value. Handle objects by comparing JSON values
        const v = this.value;
        if (typeof (v) == 'object') {
            return JSON.stringify(this.valueOnFocus) != JSON.stringify(v);
        }
        else {
            return this.valueOnFocus != v;
        }
    }
    onFocusIn(e) {
        if (e.relatedTarget instanceof HTMLElement && this.el.contains(e.relatedTarget)) {
            //focus is still within this field
            return;
        }
        if (this.fireChangeOnBlur) {
            this.captureValueForChange();
        }
    }
    captureValueForChange() {
        const v = this.value;
        this.valueOnFocus = typeof (v) == 'object' ? structuredClone(v) : v;
    }
    internalRender() {
        const el = super.internalRender();
        this.renderControl();
        this.el.addEventListener("focusin", this.onFocusIn.bind(this));
        this.el.addEventListener("focusout", this.onFocusOut.bind(this));
        return el;
    }
    isFocusable() {
        return !this.hidden;
    }
    // get el(): HTMLElement {
    // 	const el = super.el;
    //
    // 	// if(!this.wrap) {
    // 	// 	el.append(this.wrap = E("div").cls('+wrap'));
    // 	// }
    // 	return el;
    // }
    /**
     * A wrapper DIV element that contains input and toolbar for input buttons like an expand button for a drop down
     */
    get wrap() {
        // wrap required to place buttons after element
        if (!this._wrap) {
            this.el.append(this._wrap = E("div").cls('+wrap'));
        }
        return this._wrap;
    }
    renderControl() {
        // label must follow input so we can make the transform transition with pure css with input::focus & input::placeholder-shown + label
        const label = this.createLabel();
        if (label) {
            this.wrap.append(label);
        }
        if (this.control) {
            this.wrap.append(this.control.cls('+control'));
            if (this.title) {
                this.control.title = this.title;
            }
        }
        this.renderButtons();
        const hint = this.createHint();
        if (hint) {
            this.el.appendChild(hint);
        }
        //		this.internalSetValue(this.value);
    }
    renderButtons() {
        if (this._buttons) {
            this.toolbar = tbar({}, ...this._buttons);
            this.toolbar.parent = this;
            this.toolbar.render(this.wrap);
            this._buttons.forEach((btn) => {
                if (btn.menu) {
                    this.setupMenu(btn.menu);
                }
            });
        }
        else {
            if (this.toolbar) {
                this.toolbar.remove();
            }
        }
    }
    /**
     * When buttons with menus are added it is handy to delay the validation on blur.
     * Because when the user will click something in the menu it will blur the field and you probably don't
     * want it to validate at that point. It's important that the menu will return focus to the field
     * and sets the value afterward.
     *
     * @param menu
     * @protected
     */
    setupMenu(menu) {
        let origValidateOnBlur = false;
        menu.on("beforeshow", () => {
            origValidateOnBlur = this.validateOnBlur;
            this.validateOnBlur = false;
        });
        menu.on("hide", () => {
            this.validateOnBlur = origValidateOnBlur;
            this.focus();
        });
    }
    createControl() {
        return undefined;
    }
    /**
     * Render buttons inside the text field
     *
     * @example
     * ```
     * buttons: [
     * 				 		btn({icon: "clear", handler:(btn) => (btn.parent!.parent! as Field).value = ""})
     * 					]
     * ```
     * @param buttons
     */
    set buttons(buttons) {
        this._buttons = buttons;
        if (this.rendered) {
            this.renderButtons();
        }
    }
    get buttons() {
        return this._buttons;
    }
    createHint() {
        this.hintEl = E('div', this._hint).cls('hint');
        return this.hintEl;
    }
    createLabel() {
        this._labelEl = E('div', this.getLabelText()).cls('label');
        return this._labelEl;
    }
    getLabelText() {
        if (!this._label) {
            return "";
        }
        let labelText = this._label;
        if (this._required) {
            labelText += ' *';
        }
        return labelText;
    }
    /**
     * Form element name which will be the key in values
     * If omitted the field won't be included in the form values.
     */
    get name() {
        return this._name || "";
    }
    /**
     * The field's name
     */
    set name(name) {
        this._name = name;
    }
    get required() {
        return this._required;
    }
    /**
     * Required or not
     */
    set required(required) {
        this._required = required;
        if (this._labelEl) {
            this._labelEl.innerHTML = this.getLabelText();
        }
        if (this.rendered) {
            this.clearInvalid();
        }
    }
    get label() {
        return this._label + "";
    }
    /**
     * The field's label
     */
    set label(label) {
        this._label = label;
        if (this._labelEl) {
            this._labelEl.innerHTML = this.getLabelText();
        }
    }
    get icon() {
        return this._icon;
    }
    /**
     * The field's icon rendered at the left inside the field
     */
    set icon(icon) {
        this._icon = icon;
        this.createIcon();
    }
    get hint() {
        return this._hint + "";
    }
    /**
     * The field's hint text
     */
    set hint(hint) {
        this._hint = hint;
        if (this.rendered) {
            //this sets hint if not invalid
            this.applyInvalidMsg();
        }
    }
    get readOnly() {
        return this._readOnly;
    }
    /**
     * Make the field read only
     */
    set readOnly(readOnly) {
        this.el.classList.toggle("readonly", readOnly);
        this._readOnly = readOnly;
    }
    /**
     * Check if the field was modified since create or when a form was loaded and @see trackReset() was called.
     */
    isModified() {
        // We use stringify to support object and array values
        // console.log("isModified()", JSON.stringify(this.resetValue), JSON.stringify(this.value))
        return JSON.stringify(this.resetValue) !== JSON.stringify(this.value);
    }
    /**
     * Copies the current value to the reset value. Typically happens when this component was added to a parent and
     * when the form it belongs too loads.
     *
     * @see Form in the trackModifications method
     */
    trackReset() {
        // use structuredclone to support arrays and objects
        this.resetValue = structuredClone(this.value);
    }
    /**
     * Set the field value
     */
    set value(v) {
        const old = this._value;
        this._value = v;
        this.internalSetValue(v);
        this.checkHasValue();
        this.fire("setvalue", this, this._value, old);
    }
    checkHasValue() {
        this.el.classList.toggle("has-value", !this.isEmpty());
    }
    /**
     * Applies set value to the control.
     *
     * This is also called when the control is rendered. Note that this.rendered is still false when that happens.
     *
     * @param v
     * @protected
     */
    internalSetValue(v) {
    }
    /**
     * Helper to fire "change" event. Child classes must implement this.
     */
    fireChange() {
        const v = this.value;
        this.fire("setvalue", this, v, this.valueOnFocus);
        this.fire("change", this, v, this.valueOnFocus);
        this.valueOnFocus = undefined;
        this.checkHasValue();
    }
    get value() {
        return this._value;
    }
    /**
     * Reset the field value to the value that was given to the field's constructor
     * @see setValue()
     */
    reset() {
        const old = this.value;
        this.value = this.resetValue;
        this.clearInvalid();
        this.fire("reset", this, this.resetValue, old);
        this.fire("setvalue", this, this.resetValue, old);
        this.fire("change", this, this.resetValue, old);
    }
    /**
     * Set the field as invalid and set a message
     *
     * @param msg
     */
    setInvalid(msg) {
        this.invalidMsg = msg;
        if (this.rendered) {
            this.applyInvalidMsg();
        }
        this.fire("invalid", this);
    }
    /**
     * Check if the field is empty
     */
    isEmpty() {
        const v = this.value;
        return v === undefined || v === null || v === "";
    }
    validate() {
        this.clearInvalid();
        if (this._required && this.isEmpty()) {
            this.setInvalid(t("This field is required"));
        }
        this.fire("validate", this);
    }
    /*

            badInput
:
false
customError
:
false
patternMismatch
:
false
rangeOverflow
:
false
rangeUnderflow
:
false
stepMismatch
:
false
tooLong
:
false
tooShort
:
false
typeMismatch
:
false

             */
    setValidityState(input) {
        const validity = input.validity;
        if (validity.valid) {
            return;
        }
        if (validity.typeMismatch) {
            switch (input.type) {
                case 'email':
                    this.setInvalid(t("Please enter a valid e-mail address"));
                    return;
                default:
                    this.setInvalid(t("Please enter a valid value"));
                    return;
            }
        }
        else if (validity.valueMissing) {
            this.setInvalid(t("This field is required"));
        }
        else if (validity.tooLong) {
            this.setInvalid(t("The maximum length for this field is {max}").replace("{max}", input.maxLength));
        }
        else if (validity.tooShort) {
            this.setInvalid(t("The minimum length for this field is {max}").replace("{max}", input.minLength));
        }
        else if (validity.patternMismatch) {
            this.setInvalid(t("Please match the format requested").replace("{max}", input.minLength));
        }
        else {
            console.warn("TODO: Implement translated message");
            this.setInvalid(input.validationMessage);
        }
    }
    applyInvalidMsg() {
        if (this.invalidMsg) {
            this.el.classList.add("invalid");
            if (this.hintEl)
                this.hintEl.innerHTML = this.invalidMsg;
        }
        else {
            this.el.classList.remove("invalid");
            if (this.hintEl)
                this.hintEl.innerHTML = this._hint || "";
        }
    }
    /**
     * Clears the invalid state
     */
    clearInvalid() {
        if (!this.invalidMsg) {
            return;
        }
        this.invalidMsg = "";
        if (this.rendered) {
            this.applyInvalidMsg();
        }
    }
    /**
     * Checks if the field is valid
     */
    isValid() {
        if (this.invalidMsg != "") {
            return false;
        }
        this.validate();
        if (this.invalidMsg != "") {
            console.warn("Field '" + this.name + "' is invalid: " + this.invalidMsg, this);
        }
        return this.invalidMsg == "";
    }
    createIcon() {
        if (this._icon) {
            if (!this.iconEl) {
                this.iconEl = E("i").cls("icon");
            }
            this.iconEl.innerText = this._icon;
            this.el.classList.add("with-icon");
            if (this.wrap) {
                this.wrap.insertBefore(this.iconEl, this.wrap.firstChild);
            }
            return this.iconEl;
        }
        else {
            if (this.iconEl) {
                this.iconEl.remove();
                this.el.classList.remove("with-icon");
            }
        }
    }
}
//# sourceMappingURL=Field.js.map