<?php
declare(strict_types = 1);

/**
 * Gestion des émojis.
 *
 * @license http://www.gnu.org/licenses/gpl.html
 * @link http://www.igalerie.org/
 */
class Emoji
{
	/**
	 * Liste pour sélecteur d'émojis.
	 * https://unicode.org/emoji/charts/full-emoji-list.html
	 *
	 * @var array
	 */
	const LIST =
	[
		'smileys-and-emotion' =>
		[
			'face-smiling' =>
			[
				'1F600', '1F603', '1F604', '1F601', '1F606', '1F605', '1F923',
				'1F602', '1F642', '1F643', '1F609', '1F60A', '1F607'
			],
			'face-affection' =>
			[
				'1F970', '1F60D', '1F929', '1F618', '1F617', '1F61A', '1F619'
			],
			'face-tongue' =>
			[
				'1F60B', '1F61B', '1F61C', '1F61D', '1F911'
			],
			'face-hand' =>
			[
				'1F917', '1F92D', '1F92B', '1F914'
			],
			'face-neutral-skeptical' =>
			[
				'1F910', '1F928', '1F610', '1F611', '1F636', '1F60F', '1F612',
				'1F644', '1F62C', '1F925'
			],
			'face-sleepy' =>
			[
				'1F60C', '1F614', '1F62A', '1F924', '1F634'
			],
			'face-unwell' =>
			[
				'1F637', '1F912', '1F915', '1F922', '1F92E', '1F927', '1F975',
				'1F976', '1F974', '1F635', '1F92F'
			],
			'face-hat' =>
			[
				'1F920', '1F973'
			],
			'face-glasses' =>
			[
				'1F60E', '1F913', '1F9D0'
			],
			'face-concerned' =>
			[
				'1F615', '1F61F', '1F641', '1F62E', '1F62F', '1F632', '1F633',
				'1F97A', '1F626', '1F627', '1F628', '1F630', '1F625', '1F622',
				'1F62D', '1F631', '1F616', '1F623', '1F61E', '1F613', '1F629',
				'1F62B', '1F971'
			],
			'face-negative' =>
			[
				'1F624', '1F621', '1F620', '1F92C', '1F608', '1F47F', '1F480',
				'2620 FE0F'
			],
			'face-costume' =>
			[
				'1F4A9', '1F921', '1F479', '1F47A', '1F47B', '1F47D', '1F47E',
				'1F916'
			],
			'cat-face' =>
			[
				'1F63A', '1F638', '1F639', '1F63B', '1F63C', '1F63D', '1F640',
				'1F63F', '1F63E'
			],
			'monkey-face' =>
			[
				'1F648', '1F649', '1F64A'
			],
			'heart' =>
			[
				'1F48C', '1F498', '1F49D', '1F496', '1F497', '1F493', '1F49E',
				'1F495', '1F49F', '2763 FE0F', '1F494', '2764 FE0F', '1F9E1',
				'1F49B', '1F49A', '1F499', '1F49C', '1F90E', '1F5A4', '1F90D'
			],
			'emotion' =>
			[
				'1F48B', '1F4AF', '1F4A2', '1F4A5', '1F4AB', '1F4A6', '1F4A8',
				'1F573', '1F4AC', '1F4AD', '1F4A4'
			]
		],
		'people-and-body' =>
		[
			'hand-fingers-open' =>
			[
				'1F44B', '1F91A', '1F590', '270B FE0F', '1F596'
			],
			'hand-fingers-partial' =>
			[
				'1F44C', '270C FE0F', '1F91E', '1F918', '1F919'
			],
			'hand-single-finger' =>
			[
				'1F448', '1F449', '1F446', '1F595', '1F447', '261D FE0F'
			],
			'hand-fingers-closed' =>
			[
				'1F44D', '1F44E', '270A FE0F', '1F44A', '1F91B', '1F91C'
			],
			'hands' =>
			[
				'1F44F', '1F64C', '1F450', '1F91D', '1F64F'
			],
			'hand-prop' =>
			[
				'270D FE0F', '1F485', '1F933'
			],
			'body-parts' =>
			[
				'1F4AA', '1F9BE', '1F9BF', '1F9B5', '1F9B6', '1F442', '1F9BB',
				'1F443', '1F9E0', '1F9B7', '1F9B4', '1F440', '1F441', '1F445',
				'1F444'
			],
			'person' =>
			[
				'1F476', '1F9D2', '1F466', '1F467', '1F9D1', '1F471', '1F468',
				'1F9D4', '1F469', '1F9D3', '1F474', '1F475'
			],
			'person-role' =>
			[
				'1F46E', '1F575', '1F482', '1F477', '1F934', '1F478', '1F473',
				'1F472', '1F9D5', '1F935', '1F470', '1F930'
			],
			'person-fantasy' =>
			[
				'1F47C', '1F385', '1F936', '1F9B8', '1F9B9', '1F9D9', '1F9DA',
				'1F9DB', '1F9DC', '1F9DD', '1F9DE', '1F9DF'
			],
			'person-activity' =>
			[
				'1F6B6', '1F9CD', '1F9CE', '1F3C3', '1F483', '1F57A', '1F574',
				'1F46F', '1F9D6', '1F9D7'
			],
			'person-sport' =>
			[
				'1F93A', '1F3C7', '26F7 FE0F', '1F3C2', '1F3CC', '1F3C4',
				'1F6A3', '1F3CA', '26F9 FE0F', '1F3CB', '1F6B4', '1F6B5',
				'1F938', '1F93C', '1F93D', '1F93E', '1F939'
			],
			'person-resting' =>
			[
				'1F9D8', '1F6C0', '1F6CC'
			],
			'family' =>
			[
				'1F46D', '1F46B', '1F46C', '1F48F', '1F491'
			],
			'person-symbol' =>
			[
				'1F5E3', '1F464', '1F465', '1F46A', '1F463'
			]
		],
		'animals-and-nature' =>
		[
			'animal-mammal' =>
			[
				'1F435', '1F412', '1F98D', '1F9A7', '1F436', '1F415', '1F9AE',
				'1F429', '1F43A', '1F98A', '1F99D', '1F431', '1F408', '1F981',
				'1F431', '1F42F', '1F405', '1F406', '1F434', '1F40E', '1F984',
				'1F993', '1F98C', '1F42E', '1F402', '1F403', '1F404', '1F437',
				'1F416', '1F417', '1F43D', '1F40F', '1F411', '1F410', '1F42A',
				'1F42B', '1F999', '1F992', '1F418', '1F98F', '1F99B', '1F42D',
				'1F401', '1F400', '1F439', '1F430', '1F407', '1F43F', '1F994',
				'1F987', '1F43B', '1F428', '1F43C', '1F9A5', '1F9A6', '1F9A8',
				'1F998', '1F9A1', '1F43E'
			],
			'animal-bird' =>
			[
				'1F983', '1F414', '1F413', '1F423', '1F424', '1F425', '1F426',
				'1F427', '1F54A', '1F985', '1F986', '1F9A2', '1F989', '1F9A9',
				'1F99A', '1F99C'
			],
			'animal-amphibian' =>
			[
				'1F438'
			],
			'animal-reptile' =>
			[
				'1F40A', '1F422', '1F98E', '1F40D', '1F432', '1F409', '1F995',
				'1F996'
			],
			'animal-marine' =>
			[
				'1F433', '1F40B', '1F42C', '1F41F', '1F420', '1F421', '1F988',
				'1F419', '1F41A'
			],
			'animal-bug' =>
			[
				'1F40C', '1F98B', '1F41B', '1F41C', '1F41D', '1F41E', '1F997',
				'1F577', '1F578', '1F982', '1F99F', '1F9A0'
			],
			'plant-flower' =>
			[
				'1F490', '1F338', '1F4AE', '1F3F5', '1F339', '1F940', '1F33A',
				'1F33B', '1F33C', '1F337'
			],
			'plant-other' =>
			[
				'1F331', '1F332', '1F333', '1F334', '1F335', '1F33E', '1F33F',
				'2618 FE0F', '1F340', '1F341', '1F342', '1F343', '1F344'
			]
		],
		'food-and-drink' =>
		[
			'food-fruit' =>
			[
				'1F347', '1F348', '1F349', '1F34A', '1F34B', '1F34C', '1F34D',
				'1F96D', '1F34E', '1F34F', '1F350', '1F351', '1F352', '1F353',
				'1F95D', '1F345', '1F965'
			],
			'food-vegetable' =>
			[
				'1F951', '1F346', '1F954', '1F955', '1F33D', '1F336', '1F952',
				'1F96C', '1F966', '1F9C4', '1F9C5', '1F95C', '1F330'
			],
			'food-prepared' =>
			[
				'1F35E', '1F950', '1F956', '1F968', '1F96F', '1F95E', '1F9C7',
				'1F9C0', '1F356', '1F357', '1F969', '1F953', '1F354', '1F35F',
				'1F355', '1F32D', '1F96A', '1F32E', '1F32F', '1F959', '1F9C6',
				'1F95A', '1F373', '1F958', '1F372', '1F963', '1F957', '1F37F',
				'1F9C8', '1F9C2', '1F96B'
			],
			'food-asian' =>
			[
				'1F371', '1F358', '1F359', '1F35A', '1F35B', '1F35C', '1F35D',
				'1F360', '1F362', '1F363', '1F364', '1F365', '1F96E', '1F361',
				'1F95F', '1F960', '1F961'
			],
			'food-marine' =>
			[
				'1F980', '1F99E', '1F990', '1F991', '1F9AA'
			],
			'food-sweet' =>
			[
				'1F366', '1F367', '1F368', '1F369', '1F36A', '1F382', '1F370',
				'1F9C1', '1F967', '1F36B', '1F36C', '1F36D', '1F36E', '1F36F'
			],
			'drink' =>
			[
				'1F37C', '1F95B', '2615 FE0F', '1F375', '1F376', '1F37E',
				'1F377', '1F378', '1F379', '1F37A', '1F37B', '1F942', '1F943',
				'1F964', '1F9C3', '1F9C9', '1F9CA'
			],
			'dishware' =>
			[
				'1F962', '1F37D', '1F374', '1F944', '1F52A', '1F3FA'
			]
		],
		'travel-and-places' =>
		[
			'place-map' =>
			[
				'1F30D', '1F30E', '1F30F', '1F310', '1F5FA', '1F5FE', '1F9ED'
			],
			'place-geographic' =>
			[
				'1F3D4', '26F0 FE0F', '1F30B', '1F5FB', '1F3D5', '1F3D6',
				'1F3DC', '1F3DD', '1F3DE'
			],
			'place-building' =>
			[
				'1F3DF', '1F3DB', '1F3D7', '1F9F1', '1F3D8', '1F3DA', '1F3E0',
				'1F3E1', '1F3E2', '1F3E3', '1F3E4', '1F3E5', '1F3E6', '1F3E8',
				'1F3E9', '1F3EA', '1F3EB', '1F3EC', '1F3ED', '1F3EF', '1F3F0',
				'1F492', '1F5FC', '1F5FD'
			],
			'place-religious' =>
			[
				'26EA FE0F', '1F54C', '1F6D5', '1F54D', '26E9 FE0F', '1F54B'
			],
			'place-other' =>
			[
				'26F2 FE0F', '26FA FE0F', '1F301', '1F303', '1F3D9', '1F304',
				'1F305', '1F306', '1F307', '1F309', '2668 FE0F', '1F3A0',
				'1F3A1', '1F3A2', '1F488', '1F3AA'
			],
			'transport-ground' =>
			[
				'1F682', '1F683', '1F684', '1F685', '1F686', '1F687', '1F688',
				'1F689', '1F68A', '1F69D', '1F69E', '1F68B', '1F68C', '1F68D',
				'1F68E', '1F690', '1F691', '1F692', '1F693', '1F694', '1F695',
				'1F696', '1F697', '1F698', '1F699', '1F69A', '1F69B', '1F69C',
				'1F3CE', '1F3CD', '1F6F5', '1F9BD', '1F9BC', '1F6FA', '1F6B2',
				'1F6F4', '1F6F9', '1F68F', '1F6E3', '1F6E4', '1F6E2',
				'26FD FE0F', '1F6A8', '1F6A5', '1F6A6', '1F6D1', '1F6A7'
			],
			'transport-water' =>
			[
				'2693 FE0F', '26F5 FE0F', '1F6F6', '1F6A4', '1F6F3',
				'26F4 FE0F', '1F6E5', '1F6A2'
			],
			'transport-air' =>
			[
				'2708 FE0F', '1F6E9', '1F6EB', '1F6EC', '1FA82', '1F4BA',
				'1F681', '1F69F', '1F6A0', '1F6A1', '1F6F0', '1F680', '1F6F8'
			],
			'hotel' =>
			[
				'1F9F3'
			],
			'time' =>
			[
				'231B FE0F', '23F3 FE0F', '231A FE0F', '23F0 FE0F',
				'23F1 FE0F', '23F2 FE0F', '1F570', '1F553'
			],
			'sky-&-weather' =>
			[
				'1F311', '1F312', '1F313', '1F314', '1F315', '1F316', '1F317',
				'1F318', '1F319', '1F31A', '1F31B', '1F31C', '1F321',
				'2600 FE0F', '1F31D', '1F31E', '1FA90', '2B50 FE0F', '1F31F',
				'1F320', '1F30C', '2601 FE0F', '26C5 FE0F', '26C8 FE0F',
				'1F324', '1F325', '1F326', '1F327', '1F328', '1F329', '1F32A',
				'1F32B', '1F32C', '1F300', '1F308', '1F302', '2602 FE0F',
				'2614 FE0F', '26F1 FE0F', '26A1 FE0F', '2744 FE0F',
				'2603 FE0F', '26C4 FE0F', '2604 FE0F', '1F525', '1F4A7',
				'1F30A'
			]
		],
		'activities' =>
		[
			'event' =>
			[
				'1F383', '1F384', '1F386', '1F387', '1F9E8', '2728 FE0F',
				'1F388', '1F389', '1F38A', '1F38B', '1F38D', '1F38E', '1F38F',
				'1F390', '1F391', '1F9E7', '1F380', '1F381', '1F397', '1F39F',
				'1F3AB'
			],
			'award-medal' =>
			[
				'1F396', '1F3C6', '1F3C5', '1F947', '1F948', '1F949'
			],
			'sport' =>
			[
				'26BD FE0F', '26BE FE0F', '1F94E', '1F3C0', '1F3D0', '1F3C8',
				'1F3C9', '1F3BE', '1F94F', '1F3B3', '1F3CF', '1F3D1', '1F3D2',
				'1F94D', '1F3D3', '1F3F8', '1F94A', '1F94B', '1F945',
				'26F3 FE0F', '26F8 FE0F', '1F3A3', '1F93F', '1F3BD', '1F3BF',
				'1F6F7', '1F94C'
			],
			'game' =>
			[
				'1F3AF', '1FA80', '1FA81', '1F52B', '1F3B1', '1F52E', '1F3AE',
				'1F579', '1F3B0', '1F3B2', '1F9E9', '1F9F8', '1F0CF', '1F004',
				'1F3B4'
			],
			'arts-&-crafts' =>
			[
				'1F3AD', '1F5BC', '1F3A8', '1F9F5', '1F9F6'
			]
		],
		'objects' =>
		[
			'clothing' =>
			[
				'1F453', '1F576', '1F97D', '1F97C', '1F9BA', '1F454', '1F455',
				'1F456', '1F9E3', '1F9E4', '1F9E5', '1F9E6', '1F457', '1F458',
				'1F97B', '1FA71', '1FA72', '1FA73', '1F459', '1F45A', '1F45B',
				'1F45C', '1F45D', '1F6CD', '1F392', '1F45E', '1F45F', '1F97E',
				'1F97F', '1F460', '1F461', '1FA70', '1F462', '1F451', '1F452',
				'1F3A9', '1F393', '1F9E2', '26D1 FE0F', '1F4FF', '1F484',
				'1F48D', '1F48E'
			],
			'sound' =>
			[
				'1F507', '1F508', '1F509', '1F50A', '1F4E2', '1F4E3', '1F4EF',
				'1F514', '1F515'
			],
			'music' =>
			[
				'1F3BC', '1F3B5', '1F3B6', '1F399', '1F39A', '1F39B', '1F3A4',
				'1F3A7', '1F4FB'
			],
			'musical-instrument' =>
			[
				'1F3B7', '1F3B8', '1F3B9', '1F3BA', '1F3BB', '1FA95', '1F941'
			],
			'phone' =>
			[
				'1F4F1', '260E FE0F', '1F4DE', '1F4DF', '1F4E0'
			],
			'computer' =>
			[
				'1F50B', '1F50C', '1F4BB', '1F5A5', '1F5A8', '2328 FE0F',
				'1F5B1', '1F5B2', '1F4BD', '1F4BE', '1F4BF', '1F4C0', '1F9EE'
			],
			'light-&-video' =>
			[
				'1F3A5', '1F39E', '1F4FD', '1F3AC', '1F4FA', '1F4F7', '1F4F8',
				'1F4F9', '1F4FC', '1F50E', '1F56F', '1F4A1', '1F526', '1F3EE',
				'1FA94'
			],
			'book-paper' =>
			[
				'1F4D4', '1F4D5', '1F4D6', '1F4D7', '1F4D8', '1F4D9', '1F4DA',
				'1F4D3', '1F4D2', '1F4C3', '1F4DC', '1F4C4', '1F4F0', '1F5DE',
				'1F4D1', '1F516', '1F3F7'
			],
			'money' =>
			[
				'1F4B0', '1F4B4', '1F4B5', '1F4B6', '1F4B7', '1F4B8', '1F4B3',
				'1F4B9'
			],
			'mail' =>
			[
				'2709 FE0F', '1F4E7', '1F4E4', '1F4E5', '1F4E6', '1F4EB',
				'1F4EE', '1F5F3'
			],
			'writing' =>
			[
				'270F FE0F', '2712 FE0F', '1F58B', '1F58A', '1F58C', '1F58D',
				'1F4DD'
			],
			'office' =>
			[
				'1F4BC', '1F4C1', '1F4C2', '1F5C2', '1F4C6', '1F5D2', '1F5D3',
				'1F4C7', '1F4C8', '1F4C9', '1F4CA', '1F4CB', '1F4CC', '1F4CD',
				'1F4CE', '1F587', '1F4CF', '1F4D0', '2702 FE0F', '1F5C3',
				'1F5C4', '1F5D1'
			],
			'lock' =>
			[
				'1F512', '1F511', '1F5DD'
			],
			'tool' =>
			[
				'1F528', '1FA93', '26CF FE0F', '2692 FE0F', '1F6E0', '1F5E1',
				'2694 FE0F', '1F4A3', '1F3F9', '1F6E1', '1F527', '1F529',
				'2699 FE0F', '1F5DC', '2696 FE0F', '1F9AF', '1F517',
				'26D3 FE0F', '1F9F0', '1F9F2'
			],
			'science' =>
			[
				'2697 FE0F', '1F9EA', '1F9EB', '1F9EC', '1F52C', '1F52D',
				'1F4E1'
			],
			'medical' =>
			[
				'1F489', '1FA78', '1F48A', '1FA79', '1FA7A'
			],
			'household' =>
			[
				'1F6AA', '1F6CF', '1F6CB', '1FA91', '1F6BD', '1F6BF', '1F6C1',
				'1FA92', '1F9F4', '1F9F7', '1F9F9', '1F9FA', '1F9FB', '1F9FC',
				'1F9FD', '1F9EF', '1F6D2'
			],
			'other-object' =>
			[
				'1F6AC', '1F5FF'
			]
		],
		'symbols' =>
		[
			'transport-sign' =>
			[
				'1F3E7', '1F6AE', '1F6B0', '267F FE0F', '1F6B9', '1F6BA',
				'1F6BB', '1F6BC', '1F6BE', '1F6C2', '1F6C3', '1F6C4', '1F6C5'
			],
			'warning' =>
			[
				'26A0 FE0F', '1F6B8', '26D4 FE0F', '1F6AB', '1F6B3', '1F6AD',
				'1F6AF', '1F6B1', '1F6B7', '1F4F5', '1F51E', '2622 FE0F',
				'2623 FE0F'
			],
			'arrow' =>
			[
				'2B06 FE0F', '2197 FE0F', '27A1 FE0F', '2198 FE0F',
				'2B07 FE0F', '2199 FE0F', '2B05 FE0F', '2196 FE0F',
				'2195 FE0F', '2194 FE0F', '21A9 FE0F', '21AA FE0F',
				'2934 FE0F', '2935 FE0F', '1F503', '1F504', '1F519', '1F51A',
				'1F51B', '1F51C', '1F51D'
			],
			'religion' =>
			[
				'1F6D0', '269B FE0F', '1F549', '2721 FE0F', '2638 FE0F',
				'262F FE0F', '271D FE0F', '2626 FE0F', '262A FE0F',
				'262E FE0F', '1F54E', '1F52F'
			],
			'zodiac' =>
			[
				'2648 FE0F', '2649 FE0F', '264A FE0F', '264B FE0F',
				'264C FE0F', '264D FE0F', '264E FE0F', '264F FE0F',
				'2650 FE0F', '2651 FE0F', '2652 FE0F', '2653 FE0F',
				'26CE FE0F'
			],
			'av-symbol' =>
			[
				'1F500', '1F501', '1F502', '25B6 FE0F', '23E9 FE0F',
				'23ED FE0F', '23EF FE0F', '25C0 FE0F', '23EA FE0F',
				'23EE FE0F', '1F53C', '23EB FE0F', '1F53D', '23EC FE0F',
				'23F8 FE0F', '23F9 FE0F', '23FA FE0F', '23CF FE0F', '1F3A6',
				'1F505', '1F506', '1F4F6', '1F4F3', '1F4F4'
			],
			'gender' =>
			[
				'2640 FE0F', '2642 FE0F'
			],
			'math' =>
			[
				'2716 FE0F', '2795 FE0F', '2796 FE0F', '2797 FE0F', '267E FE0F'
			],
			'punctuation' =>
			[
				'2049 FE0F', '2753 FE0F', '2754 FE0F', '2755 FE0F',
				'2757 FE0F', '3030 FE0F'
			],
			'currency' =>
			[
				'1F4B1', '1F4B2'
			],
			'other-symbol' =>
			[
				'267B FE0F', '269C FE0F', '1F531', '1F4DB', '1F530',
				'2B55 FE0F', '2705 FE0F', '2611 FE0F', '2714 FE0F',
				'274C FE0F', '274E FE0F', '27B0 FE0F', '27BF FE0F',
				'303D FE0F', '2733 FE0F', '2734 FE0F'
			],
			'geometric' =>
			[
				'1F534', '1F535', '26AB FE0F', '26AA FE0F', '2B1B FE0F',
				'2B1C FE0F', '25FC FE0F', '25FB FE0F', '25FE FE0F',
				'25FD FE0F', '25AA FE0F', '25AB FE0F', '1F536', '1F537',
				'1F538', '1F539', '1F53A', '1F53B', '1F4A0', '1F518', '1F533',
				'1F532'
			],
			'flags' =>
			[
				'1F3C1', '1F6A9', '1F38C', '1F3F4', '1F3F3'
			]
		]
	];

	/**
	 * Expression régulière pour matcher les émojis de Unicode 15.
	 * Créée à partir du fichier suivant :
	 * http://www.unicode.org/Public/UCD/latest/ucd/emoji/emoji-data.txt
	 *
	 * @var string
	 */
	const REGEXP = '(?:(?:\x{261D}|\x{26F9}|[\x{270A}-\x{270C}]|\x{270D}|\x{1F385}|[\x{1F3C2}-\x{'
		. '1F3C4}]|\x{1F3C7}|\x{1F3CA}|[\x{1F3CB}-\x{1F3CC}]|[\x{1F442}-\x{1F443}]|[\x{1F446}-\x{'
		. '1F450}]|[\x{1F466}-\x{1F46B}]|[\x{1F46C}-\x{1F46D}]|[\x{1F46E}-\x{1F478}]|\x{1F47C}|['
		. '\x{1F481}-\x{1F483}]|[\x{1F485}-\x{1F487}]|\x{1F48F}|\x{1F491}|\x{1F4AA}|[\x{1F574}-\x'
		. '{1F575}]|\x{1F57A}|\x{1F590}|[\x{1F595}-\x{1F596}]|[\x{1F645}-\x{1F647}]|[\x{1F64B}-\x'
		. '{1F64F}]|\x{1F6A3}|[\x{1F6B4}-\x{1F6B5}]|\x{1F6B6}|\x{1F6C0}|\x{1F6CC}|\x{1F90C}|\x{1F'
		. '90F}|\x{1F918}|[\x{1F919}-\x{1F91E}]|\x{1F91F}|\x{1F926}|\x{1F930}|[\x{1F931}-\x{1F932'
		. '}]|[\x{1F933}-\x{1F939}]|[\x{1F93C}-\x{1F93E}]|\x{1F977}|[\x{1F9B5}-\x{1F9B6}]|[\x{1F9'
		. 'B8}-\x{1F9B9}]|\x{1F9BB}|[\x{1F9CD}-\x{1F9CF}]|[\x{1F9D1}-\x{1F9DD}]|[\x{1FAC3}-\x{1FA'
		. 'C5}]|[\x{1FAF0}-\x{1FAF6}]|[\x{1FAF7}-\x{1FAF8}])(?:[\x{1F3FB}-\x{1F3FF}])?|(?:[\x{231'
		. 'A}-\x{231B}]|[\x{23E9}-\x{23EC}]|\x{23F0}|\x{23F3}|[\x{2614}-\x{2615}]|[\x{2648}-\x{26'
		. '53}]|\x{267F}|\x{2693}|\x{26A1}|[\x{26BD}-\x{26BE}]|[\x{26C4}-\x{26C5}]|\x{26CE}|\x{26'
		. 'D4}|\x{26EA}|[\x{26F2}-\x{26F3}]|\x{26F5}|\x{26FA}|\x{26FD}|\x{2705}|[\x{270A}-\x{270B'
		. '}]|\x{2728}|\x{274C}|\x{274E}|[\x{2753}-\x{2755}]|\x{2757}|[\x{2795}-\x{2797}]|\x{27B0'
		. '}|\x{27BF}|[\x{2B1B}-\x{2B1C}]|\x{2B50}|\x{2B55}|\x{1F004}|\x{1F0CF}|\x{1F18E}|[\x{1F1'
		. '91}-\x{1F19A}]|\x{1F201}|\x{1F21A}|\x{1F22F}|[\x{1F232}-\x{1F236}]|[\x{1F238}-\x{1F23A'
		. '}]|[\x{1F250}-\x{1F251}]|[\x{1F300}-\x{1F30C}]|[\x{1F30D}-\x{1F30E}]|\x{1F30F}|\x{1F31'
		. '0}|\x{1F311}|\x{1F312}|[\x{1F313}-\x{1F315}]|[\x{1F316}-\x{1F318}]|\x{1F319}|\x{1F31A}'
		. '|\x{1F31B}|\x{1F31C}|[\x{1F31D}-\x{1F31E}]|[\x{1F31F}-\x{1F320}]|[\x{1F32D}-\x{1F32F}]'
		. '|[\x{1F330}-\x{1F331}]|[\x{1F332}-\x{1F333}]|[\x{1F334}-\x{1F335}]|[\x{1F337}-\x{1F34A'
		. '}]|\x{1F34B}|[\x{1F34C}-\x{1F34F}]|\x{1F350}|[\x{1F351}-\x{1F37B}]|\x{1F37C}|[\x{1F37E'
		. '}-\x{1F37F}]|[\x{1F380}-\x{1F393}]|[\x{1F3A0}-\x{1F3C4}]|\x{1F3C5}|\x{1F3C6}|\x{1F3C7}'
		. '|\x{1F3C8}|\x{1F3C9}|\x{1F3CA}|[\x{1F3CF}-\x{1F3D3}]|[\x{1F3E0}-\x{1F3E3}]|\x{1F3E4}|['
		. '\x{1F3E5}-\x{1F3F0}]|\x{1F3F4}|[\x{1F3F8}-\x{1F407}]|\x{1F408}|[\x{1F409}-\x{1F40B}]|['
		. '\x{1F40C}-\x{1F40E}]|[\x{1F40F}-\x{1F410}]|[\x{1F411}-\x{1F412}]|\x{1F413}|\x{1F414}|'
		. '\x{1F415}|\x{1F416}|[\x{1F417}-\x{1F429}]|\x{1F42A}|[\x{1F42B}-\x{1F43E}]|\x{1F440}|['
		. '\x{1F442}-\x{1F464}]|\x{1F465}|[\x{1F466}-\x{1F46B}]|[\x{1F46C}-\x{1F46D}]|[\x{1F46E}-'
		. '\x{1F4AC}]|\x{1F4AD}|[\x{1F4AE}-\x{1F4B5}]|[\x{1F4B6}-\x{1F4B7}]|[\x{1F4B8}-\x{1F4EB}]'
		. '|[\x{1F4EC}-\x{1F4ED}]|\x{1F4EE}|\x{1F4EF}|[\x{1F4F0}-\x{1F4F4}]|\x{1F4F5}|[\x{1F4F6}-'
		. '\x{1F4F7}]|\x{1F4F8}|[\x{1F4F9}-\x{1F4FC}]|[\x{1F4FF}-\x{1F502}]|\x{1F503}|[\x{1F504}-'
		. '\x{1F507}]|\x{1F508}|\x{1F509}|[\x{1F50A}-\x{1F514}]|\x{1F515}|[\x{1F516}-\x{1F52B}]|['
		. '\x{1F52C}-\x{1F52D}]|[\x{1F52E}-\x{1F53D}]|[\x{1F54B}-\x{1F54E}]|[\x{1F550}-\x{1F55B}]'
		. '|[\x{1F55C}-\x{1F567}]|\x{1F57A}|[\x{1F595}-\x{1F596}]|\x{1F5A4}|[\x{1F5FB}-\x{1F5FF}]'
		. '|\x{1F600}|[\x{1F601}-\x{1F606}]|[\x{1F607}-\x{1F608}]|[\x{1F609}-\x{1F60D}]|\x{1F60E}'
		. '|\x{1F60F}|\x{1F610}|\x{1F611}|[\x{1F612}-\x{1F614}]|\x{1F615}|\x{1F616}|\x{1F617}|\x{'
		. '1F618}|\x{1F619}|\x{1F61A}|\x{1F61B}|[\x{1F61C}-\x{1F61E}]|\x{1F61F}|[\x{1F620}-\x{1F6'
		. '25}]|[\x{1F626}-\x{1F627}]|[\x{1F628}-\x{1F62B}]|\x{1F62C}|\x{1F62D}|[\x{1F62E}-\x{1F6'
		. '2F}]|[\x{1F630}-\x{1F633}]|\x{1F634}|\x{1F635}|\x{1F636}|[\x{1F637}-\x{1F640}]|[\x{1F6'
		. '41}-\x{1F644}]|[\x{1F645}-\x{1F64F}]|\x{1F680}|[\x{1F681}-\x{1F682}]|[\x{1F683}-\x{1F6'
		. '85}]|\x{1F686}|\x{1F687}|\x{1F688}|\x{1F689}|[\x{1F68A}-\x{1F68B}]|\x{1F68C}|\x{1F68D}'
		. '|\x{1F68E}|\x{1F68F}|\x{1F690}|[\x{1F691}-\x{1F693}]|\x{1F694}|\x{1F695}|\x{1F696}|\x{'
		. '1F697}|\x{1F698}|[\x{1F699}-\x{1F69A}]|[\x{1F69B}-\x{1F6A1}]|\x{1F6A2}|\x{1F6A3}|[\x{1'
		. 'F6A4}-\x{1F6A5}]|\x{1F6A6}|[\x{1F6A7}-\x{1F6AD}]|[\x{1F6AE}-\x{1F6B1}]|\x{1F6B2}|[\x{1'
		. 'F6B3}-\x{1F6B5}]|\x{1F6B6}|[\x{1F6B7}-\x{1F6B8}]|[\x{1F6B9}-\x{1F6BE}]|\x{1F6BF}|\x{1F'
		. '6C0}|[\x{1F6C1}-\x{1F6C5}]|\x{1F6CC}|\x{1F6D0}|[\x{1F6D1}-\x{1F6D2}]|\x{1F6D5}|[\x{1F6'
		. 'D6}-\x{1F6D7}]|\x{1F6DC}|[\x{1F6DD}-\x{1F6DF}]|[\x{1F6EB}-\x{1F6EC}]|[\x{1F6F4}-\x{1F6'
		. 'F6}]|[\x{1F6F7}-\x{1F6F8}]|\x{1F6F9}|\x{1F6FA}|[\x{1F6FB}-\x{1F6FC}]|[\x{1F7E0}-\x{1F7'
		. 'EB}]|\x{1F7F0}|\x{1F90C}|[\x{1F90D}-\x{1F90F}]|[\x{1F910}-\x{1F918}]|[\x{1F919}-\x{1F9'
		. '1E}]|\x{1F91F}|[\x{1F920}-\x{1F927}]|[\x{1F928}-\x{1F92F}]|\x{1F930}|[\x{1F931}-\x{1F9'
		. '32}]|[\x{1F933}-\x{1F93A}]|[\x{1F93C}-\x{1F93E}]|\x{1F93F}|[\x{1F940}-\x{1F945}]|[\x{1'
		. 'F947}-\x{1F94B}]|\x{1F94C}|[\x{1F94D}-\x{1F94F}]|[\x{1F950}-\x{1F95E}]|[\x{1F95F}-\x{1'
		. 'F96B}]|[\x{1F96C}-\x{1F970}]|\x{1F971}|\x{1F972}|[\x{1F973}-\x{1F976}]|[\x{1F977}-\x{1'
		. 'F978}]|\x{1F979}|\x{1F97A}|\x{1F97B}|[\x{1F97C}-\x{1F97F}]|[\x{1F980}-\x{1F984}]|[\x{1'
		. 'F985}-\x{1F991}]|[\x{1F992}-\x{1F997}]|[\x{1F998}-\x{1F9A2}]|[\x{1F9A3}-\x{1F9A4}]|[\x'
		. '{1F9A5}-\x{1F9AA}]|[\x{1F9AB}-\x{1F9AD}]|[\x{1F9AE}-\x{1F9AF}]|[\x{1F9B0}-\x{1F9B9}]|['
		. '\x{1F9BA}-\x{1F9BF}]|\x{1F9C0}|[\x{1F9C1}-\x{1F9C2}]|[\x{1F9C3}-\x{1F9CA}]|\x{1F9CB}|'
		. '\x{1F9CC}|[\x{1F9CD}-\x{1F9CF}]|[\x{1F9D0}-\x{1F9E6}]|[\x{1F9E7}-\x{1F9FF}]|[\x{1FA70}'
		. '-\x{1FA73}]|\x{1FA74}|[\x{1FA75}-\x{1FA77}]|[\x{1FA78}-\x{1FA7A}]|[\x{1FA7B}-\x{1FA7C}'
		. ']|[\x{1FA80}-\x{1FA82}]|[\x{1FA83}-\x{1FA86}]|[\x{1FA87}-\x{1FA88}]|\x{1FA89}|\x{1FA8F'
		. '}|[\x{1FA90}-\x{1FA95}]|[\x{1FA96}-\x{1FAA8}]|[\x{1FAA9}-\x{1FAAC}]|[\x{1FAAD}-\x{1FAA'
		. 'F}]|[\x{1FAB0}-\x{1FAB6}]|[\x{1FAB7}-\x{1FABA}]|[\x{1FABB}-\x{1FABD}]|\x{1FABE}|\x{1FA'
		. 'BF}|[\x{1FAC0}-\x{1FAC2}]|[\x{1FAC3}-\x{1FAC5}]|\x{1FAC6}|[\x{1FACE}-\x{1FACF}]|[\x{1F'
		. 'AD0}-\x{1FAD6}]|[\x{1FAD7}-\x{1FAD9}]|[\x{1FADA}-\x{1FADB}]|\x{1FADC}|\x{1FADF}|[\x{1F'
		. 'AE0}-\x{1FAE7}]|\x{1FAE8}|\x{1FAE9}|[\x{1FAF0}-\x{1FAF6}]|[\x{1FAF7}-\x{1FAF8}])|(?:\x'
		. '{2049}|\x{2139}|[\x{2194}-\x{2199}]|[\x{21A9}-\x{21AA}]|[\x{231A}-\x{231B}]|\x{2328}|'
		. '\x{23CF}|[\x{23E9}-\x{23EC}]|[\x{23ED}-\x{23EE}]|\x{23EF}|\x{23F0}|[\x{23F1}-\x{23F2}]'
		. '|\x{23F3}|[\x{23F8}-\x{23FA}]|\x{24C2}|\x{25B6}|\x{25C0}|[\x{2600}-\x{2601}]|[\x{2602}'
		. '-\x{2603}]|\x{2604}|\x{260E}|\x{2611}|[\x{2614}-\x{2615}]|\x{2618}|\x{261D}|\x{2620}|['
		. '\x{2622}-\x{2623}]|\x{2626}|\x{262A}|\x{262E}|\x{262F}|[\x{2638}-\x{2639}]|\x{263A}|['
		. '\x{2648}-\x{2653}]|\x{265F}|\x{2668}|\x{267B}|\x{267E}|\x{267F}|\x{2692}|\x{2693}|\x{2'
		. '694}|\x{2695}|[\x{2696}-\x{2697}]|\x{2699}|[\x{269B}-\x{269C}]|[\x{26A0}-\x{26A1}]|\x{'
		. '26A7}|[\x{26B0}-\x{26B1}]|[\x{26BD}-\x{26BE}]|[\x{26C4}-\x{26C5}]|\x{26C8}|\x{26CE}|\x'
		. '{26CF}|\x{26D1}|\x{26D3}|\x{26D4}|\x{26E9}|\x{26EA}|[\x{26F0}-\x{26F1}]|[\x{26F2}-\x{2'
		. '6F3}]|\x{26F4}|\x{26F5}|[\x{26F7}-\x{26F9}]|\x{26FA}|\x{26FD}|\x{2702}|\x{2705}|[\x{27'
		. '08}-\x{270C}]|\x{270D}|\x{270F}|\x{2712}|\x{2714}|\x{2716}|\x{271D}|\x{2721}|\x{2728}|'
		. '[\x{2733}-\x{2734}]|\x{2744}|\x{2747}|\x{274C}|\x{274E}|[\x{2753}-\x{2755}]|\x{2757}|'
		. '\x{2763}|\x{2764}|[\x{2795}-\x{2797}]|\x{27A1}|\x{27B0}|\x{27BF}|[\x{2934}-\x{2935}]|['
		. '\x{2B05}-\x{2B07}]|[\x{2B1B}-\x{2B1C}]|\x{2B50}|\x{2B55}|\x{3030}|\x{303D}|\x{3297}|\x'
		. '{3299}|\x{1F004}|\x{1F0CF}|[\x{1F170}-\x{1F171}]|[\x{1F17E}-\x{1F17F}]|\x{1F18E}|[\x{1'
		. 'F191}-\x{1F19A}]|[\x{1F201}-\x{1F202}]|\x{1F21A}|\x{1F22F}|[\x{1F232}-\x{1F23A}]|[\x{1'
		. 'F250}-\x{1F251}]|[\x{1F300}-\x{1F30C}]|[\x{1F30D}-\x{1F30E}]|\x{1F30F}|\x{1F310}|\x{1F'
		. '311}|\x{1F312}|[\x{1F313}-\x{1F315}]|[\x{1F316}-\x{1F318}]|\x{1F319}|\x{1F31A}|\x{1F31'
		. 'B}|\x{1F31C}|[\x{1F31D}-\x{1F31E}]|[\x{1F31F}-\x{1F320}]|\x{1F321}|[\x{1F324}-\x{1F32C'
		. '}]|[\x{1F32D}-\x{1F32F}]|[\x{1F330}-\x{1F331}]|[\x{1F332}-\x{1F333}]|[\x{1F334}-\x{1F3'
		. '35}]|\x{1F336}|[\x{1F337}-\x{1F34A}]|\x{1F34B}|[\x{1F34C}-\x{1F34F}]|\x{1F350}|[\x{1F3'
		. '51}-\x{1F37B}]|\x{1F37C}|\x{1F37D}|[\x{1F37E}-\x{1F37F}]|[\x{1F380}-\x{1F393}]|[\x{1F3'
		. '96}-\x{1F397}]|[\x{1F399}-\x{1F39B}]|[\x{1F39E}-\x{1F39F}]|[\x{1F3A0}-\x{1F3C4}]|\x{1F'
		. '3C5}|\x{1F3C6}|\x{1F3C7}|\x{1F3C8}|\x{1F3C9}|\x{1F3CA}|[\x{1F3CB}-\x{1F3CE}]|[\x{1F3CF'
		. '}-\x{1F3D3}]|[\x{1F3D4}-\x{1F3DF}]|[\x{1F3E0}-\x{1F3E3}]|\x{1F3E4}|[\x{1F3E5}-\x{1F3F0'
		. '}]|\x{1F3F3}|\x{1F3F4}|\x{1F3F5}|\x{1F3F7}|[\x{1F3F8}-\x{1F407}]|\x{1F408}|[\x{1F409}-'
		. '\x{1F40B}]|[\x{1F40C}-\x{1F40E}]|[\x{1F40F}-\x{1F410}]|[\x{1F411}-\x{1F412}]|\x{1F413}'
		. '|\x{1F414}|\x{1F415}|\x{1F416}|[\x{1F417}-\x{1F429}]|\x{1F42A}|[\x{1F42B}-\x{1F43E}]|'
		. '\x{1F43F}|\x{1F440}|\x{1F441}|[\x{1F442}-\x{1F464}]|\x{1F465}|[\x{1F466}-\x{1F46B}]|['
		. '\x{1F46C}-\x{1F46D}]|[\x{1F46E}-\x{1F4AC}]|\x{1F4AD}|[\x{1F4AE}-\x{1F4B5}]|[\x{1F4B6}-'
		. '\x{1F4B7}]|[\x{1F4B8}-\x{1F4EB}]|[\x{1F4EC}-\x{1F4ED}]|\x{1F4EE}|\x{1F4EF}|[\x{1F4F0}-'
		. '\x{1F4F4}]|\x{1F4F5}|[\x{1F4F6}-\x{1F4F7}]|\x{1F4F8}|[\x{1F4F9}-\x{1F4FC}]|\x{1F4FD}|['
		. '\x{1F4FF}-\x{1F502}]|\x{1F503}|[\x{1F504}-\x{1F507}]|\x{1F508}|\x{1F509}|[\x{1F50A}-\x'
		. '{1F514}]|\x{1F515}|[\x{1F516}-\x{1F52B}]|[\x{1F52C}-\x{1F52D}]|[\x{1F52E}-\x{1F53D}]|['
		. '\x{1F549}-\x{1F54A}]|[\x{1F54B}-\x{1F54E}]|[\x{1F550}-\x{1F55B}]|[\x{1F55C}-\x{1F567}]'
		. '|[\x{1F56F}-\x{1F570}]|[\x{1F573}-\x{1F579}]|\x{1F57A}|\x{1F587}|[\x{1F58A}-\x{1F58D}]'
		. '|\x{1F590}|[\x{1F595}-\x{1F596}]|\x{1F5A4}|\x{1F5A5}|\x{1F5A8}|[\x{1F5B1}-\x{1F5B2}]|'
		. '\x{1F5BC}|[\x{1F5C2}-\x{1F5C4}]|[\x{1F5D1}-\x{1F5D3}]|[\x{1F5DC}-\x{1F5DE}]|\x{1F5E1}|'
		. '\x{1F5E3}|\x{1F5E8}|\x{1F5EF}|\x{1F5F3}|\x{1F5FA}|[\x{1F5FB}-\x{1F5FF}]|\x{1F600}|[\x{'
		. '1F601}-\x{1F606}]|[\x{1F607}-\x{1F608}]|[\x{1F609}-\x{1F60D}]|\x{1F60E}|\x{1F60F}|\x{1'
		. 'F610}|\x{1F611}|[\x{1F612}-\x{1F614}]|\x{1F615}|\x{1F616}|\x{1F617}|\x{1F618}|\x{1F619'
		. '}|\x{1F61A}|\x{1F61B}|[\x{1F61C}-\x{1F61E}]|\x{1F61F}|[\x{1F620}-\x{1F625}]|[\x{1F626}'
		. '-\x{1F627}]|[\x{1F628}-\x{1F62B}]|\x{1F62C}|\x{1F62D}|[\x{1F62E}-\x{1F62F}]|[\x{1F630}'
		. '-\x{1F633}]|\x{1F634}|\x{1F635}|\x{1F636}|[\x{1F637}-\x{1F640}]|[\x{1F641}-\x{1F644}]|'
		. '[\x{1F645}-\x{1F64F}]|\x{1F680}|[\x{1F681}-\x{1F682}]|[\x{1F683}-\x{1F685}]|\x{1F686}|'
		. '\x{1F687}|\x{1F688}|\x{1F689}|[\x{1F68A}-\x{1F68B}]|\x{1F68C}|\x{1F68D}|\x{1F68E}|\x{1'
		. 'F68F}|\x{1F690}|[\x{1F691}-\x{1F693}]|\x{1F694}|\x{1F695}|\x{1F696}|\x{1F697}|\x{1F698'
		. '}|[\x{1F699}-\x{1F69A}]|[\x{1F69B}-\x{1F6A1}]|\x{1F6A2}|\x{1F6A3}|[\x{1F6A4}-\x{1F6A5}'
		. ']|\x{1F6A6}|[\x{1F6A7}-\x{1F6AD}]|[\x{1F6AE}-\x{1F6B1}]|\x{1F6B2}|[\x{1F6B3}-\x{1F6B5}'
		. ']|\x{1F6B6}|[\x{1F6B7}-\x{1F6B8}]|[\x{1F6B9}-\x{1F6BE}]|\x{1F6BF}|\x{1F6C0}|[\x{1F6C1}'
		. '-\x{1F6C5}]|\x{1F6CB}|\x{1F6CC}|[\x{1F6CD}-\x{1F6CF}]|\x{1F6D0}|[\x{1F6D1}-\x{1F6D2}]|'
		. '\x{1F6D5}|[\x{1F6D6}-\x{1F6D7}]|\x{1F6DC}|[\x{1F6DD}-\x{1F6DF}]|[\x{1F6E0}-\x{1F6E5}]|'
		. '\x{1F6E9}|[\x{1F6EB}-\x{1F6EC}]|\x{1F6F0}|\x{1F6F3}|[\x{1F6F4}-\x{1F6F6}]|[\x{1F6F7}-'
		. '\x{1F6F8}]|\x{1F6F9}|\x{1F6FA}|[\x{1F6FB}-\x{1F6FC}]|[\x{1F7E0}-\x{1F7EB}]|\x{1F7F0}|'
		. '\x{1F90C}|[\x{1F90D}-\x{1F90F}]|[\x{1F910}-\x{1F918}]|[\x{1F919}-\x{1F91E}]|\x{1F91F}|'
		. '[\x{1F920}-\x{1F927}]|[\x{1F928}-\x{1F92F}]|\x{1F930}|[\x{1F931}-\x{1F932}]|[\x{1F933}'
		. '-\x{1F93A}]|[\x{1F93C}-\x{1F93E}]|\x{1F93F}|[\x{1F940}-\x{1F945}]|[\x{1F947}-\x{1F94B}'
		. ']|\x{1F94C}|[\x{1F94D}-\x{1F94F}]|[\x{1F950}-\x{1F95E}]|[\x{1F95F}-\x{1F96B}]|[\x{1F96'
		. 'C}-\x{1F970}]|\x{1F971}|\x{1F972}|[\x{1F973}-\x{1F976}]|[\x{1F977}-\x{1F978}]|\x{1F979'
		. '}|\x{1F97A}|\x{1F97B}|[\x{1F97C}-\x{1F97F}]|[\x{1F980}-\x{1F984}]|[\x{1F985}-\x{1F991}'
		. ']|[\x{1F992}-\x{1F997}]|[\x{1F998}-\x{1F9A2}]|[\x{1F9A3}-\x{1F9A4}]|[\x{1F9A5}-\x{1F9A'
		. 'A}]|[\x{1F9AB}-\x{1F9AD}]|[\x{1F9AE}-\x{1F9AF}]|[\x{1F9B0}-\x{1F9B9}]|[\x{1F9BA}-\x{1F'
		. '9BF}]|\x{1F9C0}|[\x{1F9C1}-\x{1F9C2}]|[\x{1F9C3}-\x{1F9CA}]|\x{1F9CB}|\x{1F9CC}|[\x{1F'
		. '9CD}-\x{1F9CF}]|[\x{1F9D0}-\x{1F9E6}]|[\x{1F9E7}-\x{1F9FF}]|[\x{1FA70}-\x{1FA73}]|\x{1'
		. 'FA74}|[\x{1FA75}-\x{1FA77}]|[\x{1FA78}-\x{1FA7A}]|[\x{1FA7B}-\x{1FA7C}]|[\x{1FA80}-\x{'
		. '1FA82}]|[\x{1FA83}-\x{1FA86}]|[\x{1FA87}-\x{1FA88}]|\x{1FA89}|\x{1FA8F}|[\x{1FA90}-\x{'
		. '1FA95}]|[\x{1FA96}-\x{1FAA8}]|[\x{1FAA9}-\x{1FAAC}]|[\x{1FAAD}-\x{1FAAF}]|[\x{1FAB0}-'
		. '\x{1FAB6}]|[\x{1FAB7}-\x{1FABA}]|[\x{1FABB}-\x{1FABD}]|\x{1FABE}|\x{1FABF}|[\x{1FAC0}-'
		. '\x{1FAC2}]|[\x{1FAC3}-\x{1FAC5}]|\x{1FAC6}|[\x{1FACE}-\x{1FACF}]|[\x{1FAD0}-\x{1FAD6}]'
		. '|[\x{1FAD7}-\x{1FAD9}]|[\x{1FADA}-\x{1FADB}]|\x{1FADC}|\x{1FADF}|[\x{1FAE0}-\x{1FAE7}]'
		. '|\x{1FAE8}|\x{1FAE9}|[\x{1FAF0}-\x{1FAF6}]|[\x{1FAF7}-\x{1FAF8}])\x{FE0F}?)';
}
?>