'use strict';

/**
 * Gestion du thème.
 *
 * @license http://www.gnu.org/licenses/gpl.html
 * @link http://www.igalerie.org/
 */
function Theme()
{
	// Modification de la valeur d'une propriété CSS.
	const _cssProp = (prop, val) => document.documentElement.style.setProperty(prop, val);

	// Fonction abrégée pour document.querySelector().
	const _q = selector => { return document.querySelector(selector); };

	// Fonction abrégée pour document.querySelectorAll().
	const _qAll = selector => { return document.querySelectorAll(selector); };



	// Sélecteur de couleur.
	let _colorPicker;

	// Indique si le panneau des options est ouvert.
	let _optionsOpen = false;

	// Paramètres du thème durant l'édition des options.
	let _themeParams;



	// Initialisation.
	_init();



	/**
	 * Effectue une requête Ajax.
	 *
	 * @param object data
	 * @param object loadend
	 *
	 * @return void
	 */
	function _ajax(data, loadend)
	{
		const xhr = new XMLHttpRequest();

		// Chargement terminé.
		xhr.addEventListener('loadend', () =>
		{
			const r = xhr.response;

			if (!r || typeof r != 'object')
			{
				return;
			}

			switch (r.status)
			{
				case 'error' :
				case 'info' :
				case 'no_result' :
				case 'success' :
					if (loadend && loadend[r.status])
					{
						loadend[r.status](r);
					}
					break;

				default :
					if (loadend && loadend.dflt)
					{
						loadend.dflt(r);
					}
			}
		});

		// Requête.
		xhr.open('POST', gallery_path + '/ajax.php');
		xhr.setRequestHeader('Content-type', 'application/json');
		xhr.responseType = 'json';

		// On envoie les données.
		data.anticsrf = anticsrf;
		xhr.send(JSON.stringify(data));
	}

	/**
	 * Gestion du fond.
	 *
	 * @return void
	 */
	function _background()
	{
		_backgroundChangeType();

		_qAll('#theme_background_type input').forEach(elem =>
		{
			elem.addEventListener('change', function()
			{
				_themeParams['background'][_themeParams['style']]['type'] = this.value;
				_backgroundChangeType();
			});
		});

		_q('#theme_background_filename').addEventListener('change', function()
		{
			_themeParams['background'][_themeParams['style']]['file']
				= this.selectedOptions[0].value;
			_backgroundChangeFile();
		});

		_q('#theme_background_fixed').addEventListener('change', function()
		{
			_themeParams['background'][_themeParams['style']]['fixed'] = this.checked ? 1 : 0;
			_texturesCSS('background',
				_themeParams['background'][_themeParams['style']]['texture']);

			_cssProp('--background-attachment', this.checked ? 'fixed' : 'scroll');
		});

		_q('#theme_background_internal_hatches').addEventListener('change', function()
		{
			if (_themeParams['style'] != 'dark')
			{
				return;
			}
			_themeParams['background'][_themeParams['style']]['internal_hatches']
				= this.checked ? 1 : 0;
			_q('body').classList.toggle('hatches', this.checked);
		});
	}

	/**
	 * Modification du code CSS du fond.
	 *
	 * @return void
	 */
	function _backgroundChangeCSS()
	{
		const style = _themeParams['style'];

		_cssProp('--background-color', style == 'dark' ? '#111' : 'none');
		_cssProp('--background-image', 'none');

		_q('body').classList.add('bg');

		switch (_themeParams['background'][style]['type'])
		{
			case 'none' :
				_q('body').classList.remove('bg');
				break;

			case 'file' :
				_backgroundChangeFile();
				break;

			case 'texture' :
				_texturesCSS('background',
					_themeParams['background'][_themeParams['style']]['texture']);
				break;
		}

		if (style == 'dark')
		{
			_q('body').classList.toggle(
				'hatches',
				_themeParams['background'][style]['internal_hatches'] == 1
			);
		}
		else
		{
			_q('body').classList.toggle('hatches', false);
		}
	}

	/**
	 * Modification du fichier utilisé comme fond.
	 *
	 * @return void
	 */
	function _backgroundChangeFile()
	{
		const background = _themeParams['background'][_themeParams['style']];
		const file = background['file'];

		_cssProp('--background-color', _themeParams['style'] == 'dark' ? '#111' : 'none');
		_cssProp('--background-image', file ? `url(${theme_background_dir}${file})` : 'none');
		_cssProp('--background-attachment', background['fixed'] == 1 ? 'fixed' : 'scroll');

		_qAll('#theme_background_filename option').forEach(
			elem => elem.removeAttribute('selected')
		);

		let selector = `#theme_background_filename option[value="${file}"]`;
		if (!_q(selector))
		{
			selector = '#theme_background_filename option:first-child';
			if (!_q(selector))
			{
				return;
			}
			_themeParams['background'][_themeParams['style']]['file'] = _q(selector).value;
			_cssProp('--background-image', `url(${theme_background_dir}${_q(selector).value})`);
		}

		_q(selector).setAttribute('selected', '');
		_q(selector).selected = true;
	}

	/**
	 * Modification du type de fond.
	 *
	 * @return void
	 */
	function _backgroundChangeType()
	{
		_backgroundChangeCSS();

		const style = _themeParams['style'];

		_q('#theme_background_type_none').checked = false;
		_q('#theme_background_type_border').checked = false;
		_q('#theme_background_type_texture').checked = false;
		_q('#theme_background_fixed').checked = _themeParams['background'][style]['fixed'] == 1;

		_q('#theme_background_file').style.display = 'none';
		_q('#theme_background_texture').style.display = 'none';
		_q('#theme_background_fixed').closest('p').style.display = 'none';
		_q('#theme_background_internal_hatches').closest('p').style.display
			= style == 'dark' ? 'block' : 'none';

		switch (_themeParams['background'][style]['type'])
		{
			case 'none' :
				_q('#theme_background_type_none').checked = true;
				_q('#theme_background_internal_hatches').closest('p').style.display = 'none';
				break;

			case 'border' :
				_q('#theme_background_type_border').checked = true;
				break;

			case 'file' :
				_q('#theme_background_type_file').checked = true;
				_q('#theme_background_file').style.display = 'block';
				_q('#theme_background_fixed').closest('p').style.display = 'block';
				break;

			case 'texture' :
				_q('#theme_background_type_texture').checked = true;
				_q('#theme_background_texture').style.display = 'flex';
				_q('#theme_background_fixed').closest('p').style.display = 'block';
				break;
		}

		if (style == 'dark')
		{
			_q('#theme_background_internal_hatches').checked
				= _themeParams['background'][style]['internal_hatches'] == 1;
		}
	}

	/**
	 * Gestion de la bannière.
	 *
	 * @return void
	 */
	function _banner()
	{
		_bannerChangeType();
		_bannerChangeHeight();

		_qAll('#theme_banner_type input').forEach(elem =>
		{
			elem.addEventListener('change', function()
			{
				_themeParams['banner'][_themeParams['style']]['type'] = this.value;
				_bannerChangeType();
			});
		});

		_q('#theme_banner_filename').addEventListener('change', function()
		{
			_themeParams['banner'][_themeParams['style']]['file'] = this.selectedOptions[0].value;
			_bannerChangeFile();
		});

		_q('#theme_banner_height').addEventListener('change', function()
		{
			this.querySelectorAll('option').forEach(elem => elem.removeAttribute('selected'));
			this.selectedOptions[0].setAttribute('selected', '');
			_themeParams['banner'][_themeParams['style']]['height']
				= this.selectedOptions[0].value + 'px';
			_bannerChangeHeight();
		});
	}

	/**
	 * Modification du code CSS pour la bannière.
	 *
	 * @return void
	 */
	function _bannerChangeCSS()
	{
		_q('#header_top').closest('header').classList.remove('banner', 'color', 'none');

		switch (_themeParams['banner'][_themeParams['style']]['type'])
		{
			case 'color' :
				_q('#header_top').closest('header').classList.add('color');
				break;

			case 'file' :
				_q('#header_top').closest('header').classList.add('banner');
				_bannerChangeFile();
				break;

			case 'none' :
				_q('#header_top').closest('header').classList.add('none');
				break;

			case 'texture' :
				_q('#header_top').closest('header').classList.add('banner');
				_texturesCSS('banner', _themeParams['banner'][_themeParams['style']]['texture']);
				break;
		}
	}

	/**
	 * Modification du fichier utilisé comme bannière.
	 *
	 * @return void
	 */
	function _bannerChangeFile()
	{
		const file = _themeParams['banner'][_themeParams['style']]['file'];

		_cssProp('--banner-color', 'transparent');
		_cssProp('--banner-image', file ? `url(${theme_banners_dir}${file})` : 'none');
		_cssProp('--banner-size', 'cover');
		_cssProp('--banner-position', 'center');
		_cssProp('--banner-repeat', 'no-repeat');

		_qAll('#theme_banner_filename option').forEach(elem => elem.removeAttribute('selected'));

		let selector = `#theme_banner_filename option[value="${file}"]`;
		if (!_q(selector))
		{
			selector = '#theme_banner_filename option:first-child';
			if (!_q(selector))
			{
				return;
			}
			_themeParams['banner'][_themeParams['style']]['file'] = _q(selector).value;
			_cssProp('--banner-image', `url(${theme_banners_dir}${_q(selector).value})`);
		}

		_q(selector).setAttribute('selected', '');
		_q(selector).selected = true;
	}

	/**
	 * Modification de la hauteur de la bannière.
	 *
	 * @return void
	 */
	function _bannerChangeHeight()
	{
		_bannerChangeHeightCSS();

		const val = _themeParams['banner'][_themeParams['style']]['height'];
		const current = _q(`#theme_banner_height option[value="${parseInt(val)}"]`);

		_qAll('#theme_banner_height option').forEach(elem => elem.removeAttribute('selected'));

		if (current)
		{
			current.setAttribute('selected', '');
			current.selected = true;
		}
	}

	/**
	 * Modification du code CSS de la hauteur de la bannière.
	 *
	 * @return void
	 */
	function _bannerChangeHeightCSS()
	{
		_cssProp('--header-height', _themeParams['banner'][_themeParams['style']]['height']);
	}

	/**
	 * Modification de la couleur de fond des textures utilisées pour la bannière.
	 *
	 * @param string color
	 *
	 * @return void
	 */
	function _bannerChangeTexturesColor(color)
	{
		if (_themeParams['style'] == 'clear')
		{
			// Changement de la couleur des textures de la liste des textures.
			_qAll('div[data-type="banner"] .theme_textures_images div').forEach(elem =>
			{
				elem.style.backgroundColor = color;
			});

			// Changement de la couleur de la bannière.
			_cssProp('--banner-color', color);
		}
	}

	/**
	 * Modification du type de bannière.
	 *
	 * @return void
	 */
	function _bannerChangeType()
	{
		_bannerChangeCSS();

		_qAll('#theme_banner_type input').forEach(elem => elem.checked = false);
		_q('#theme_banner_file').style.display = 'none';
		_q('#theme_banner_texture').style.display = 'none';
		_q('#theme_banner_height').closest('p').style.display = 'none';

		switch (_themeParams['banner'][_themeParams['style']]['type'])
		{
			case 'color' :
				_q('#theme_banner_type_color').checked = true;
				break;

			case 'file' :
				_q('#theme_banner_type_file').checked = true;
				_q('#theme_banner_file').style.display = 'block';
				_q('#theme_banner_height').closest('p').style.display = 'block';
				break;

			case 'none' :
				_q('#theme_banner_type_none').checked = true;
				break;

			case 'texture' :
				_q('#theme_banner_type_texture').checked = true;
				_q('#theme_banner_texture').style.display = 'flex';
				_q('#theme_banner_height').closest('p').style.display = 'block';
				break;
		}
	}

	/**
	 * Gestion de la couleur du thème.
	 *
	 * @return void
	 */
	function _color()
	{
		// Type de couleur : prédéfinie ou personnalisée.
		_qAll('input[name="theme_color_type"]').forEach(elem =>
		{
			elem.addEventListener('change', function()
			{
				if (this.value == 'palette')
				{
					_q('#theme_palette').style.display = 'block';
					_q('#theme_custom_color').style.display = 'none';

					_colorChangePalette(_themeParams['color'][_themeParams['style']]);
				}
				else
				{
					_q('#theme_palette').style.display = 'none';
					_q('#theme_custom_color').style.display = 'block';
				}
			});
		});

		// Couleurs prédéfinies.
		_qAll('#theme_palette a').forEach(elem =>
		{
			const color_code = elem.getAttribute('data-code');

			elem.style.background = color_code;
			elem.addEventListener('click', evt =>
			{
				evt.preventDefault();
				_colorChangePalette(color_code);
			});
		});

		// Sélecteur de couleur.
		const color_init = _themeParams['color'][_themeParams['style']];
		_colorPicker = new iro.ColorPicker('#theme_color_picker',
		{
			color: color_init,
			width: 300
		});
		function color_picker_change()
		{
			const color = _colorPicker.color.hexString;

			_q('#theme_color_code').value = color;
			_cssProp('--color', color);
			_themeParams['color'][_themeParams['style']] = color;
			_bannerChangeTexturesColor(color);
		}
		_colorPicker.on('color:change', color_picker_change);

		// Code couleur.
		_q('#theme_color_code').value = color_init;
		_q('#theme_color_code').addEventListener('input', function()
		{
			if (RegExp('^#[a-f0-9]{6}$', 'i').test(this.value))
			{
				_colorPicker.color.hexString = this.value;
				color_picker_change();
			}
		});

		// Couleur initiale.
		_colorChange(color_init);
	}

	/**
	 * Modification de la couleur du thème.
	 *
	 * @param string color
	 *
	 * @return void
	 */
	function _colorChange(color)
	{
		if (_q(`#theme_palette a[data-code="${color}"]`))
		{
			_q('#theme_color_type_palette').checked = true;
			_q('#theme_color_type_custom').checked = false;
			_q('#theme_palette').style.display = 'block';
			_q('#theme_custom_color').style.display = 'none';
			_colorChangePalette(color);
		}
		else
		{
			_q('#theme_color_type_palette').checked = false;
			_q('#theme_color_type_custom').checked = true;
			_q('#theme_palette').style.display = 'none';
			_q('#theme_custom_color').style.display = 'block';
			_colorPicker.color.hexString = color;
		}
	}

	/**
	 * Couleur prédéfinie.
	 *
	 * @param string color
	 *
	 * @return void
	 */
	function _colorChangePalette(color)
	{
		_colorPicker.color.hexString = color;
		_cssProp('--color', color);

		_qAll('#theme_palette a').forEach(elem => elem.classList.remove('current'));

		const elem = _q(`#theme_palette a[data-code="${color}"]`);
		if (elem)
		{
			_q(`#theme_palette a[data-code="${color}"]`).classList.add('current');
		}

		_bannerChangeTexturesColor(color);
		_themeParams['color'][_themeParams['style']] = color;
	}

	/**
	 * Gestion du code CSS personnalisé.
	 *
	 * @return void
	 */
	function _css()
	{
		_cssChange(true);

		_q('#theme_css_code').addEventListener('input', function()
		{
			_themeParams['css'][_themeParams['style']] = this.value;
			_cssChange();
		});
	}

	/**
	 * Modification du code CSS personnalisé.
	 *
	 * @param bool init
	 *
	 * @return void
	 */
	function _cssChange(init = false)
	{
		const css = _themeParams['css'][_themeParams['style']];
		_q('#theme_css_code').value = css;

		if (!init)
		{
			_q('#style_custom').textContent = css.replace(
				/(\{(?:[\s\S](?![{}]))*?)(#color|#gallery_path)((?:[\s\S](?![{]))*?\})/gi,
				m =>
				{
					return m.replace(/#color/gi, _themeParams['color'][_themeParams['style']])
							.replace(/#gallery_path/, gallery_path);
				}
			);
		}
	}

	/**
	 * Retourne du code HTML avec entités décodées.
	 *
	 * @param string html
	 *
	 * @return string
	 */
	function _decodeHTML(html)
	{
		if (html === null)
		{
			return '';
		}
		const input = document.createElement('input');
		input.insertAdjacentHTML('afterbegin', html);

		return input.textContent;
	}

	/**
	 * Gestion des options d'affichage.
	 *
	 * @return void
	 */
	function _display()
	{
		for (const name in _themeParams['display'])
		{
			const display = _q('#theme_display_' + name);
			if (display)
			{
				display.checked = _themeParams['display'][name] == '1';
				display.addEventListener('click', function()
				{
					const name = this.getAttribute('id').replace('theme_display_', '');
					_themeParams['display'][name] = this.checked ? '1' : '0';
					_qAll(`*[data-option-display="${name}"]`)
						.forEach(elem => elem.classList.toggle('show', this.checked));
				});
			}
		}
	}

	/**
	 * Gestion des fichiers.
	 *
	 * @return void
	 */
	function _files()
	{
		// Bouton de rechargement de la liste des fichiers.
		_qAll('.theme_file_icons a[data-dir]').forEach(elem =>
		{
			elem.addEventListener('click', evt =>
			{
				evt.preventDefault();
				if (elem.classList.contains('load'))
				{
					return;
				}

				const dir = elem.getAttribute('data-dir');

				elem.classList.remove('rotate');
				elem.classList.add('load');

				_ajax(
				{
					section: 'theme-files',
					dir: dir
				},
				{
					success: r =>
					{
						elem.classList.remove('load');
						elem.classList.add('rotate');

						const select = _q(
							'#' + elem.closest('p').querySelector('select').getAttribute('id')
						);
						select.replaceChildren();

						for (let i = 0; i < r.files.length; i++)
						{
							const option = document.createElement('option');
							option.value = r.files[i];
							option.textContent = r.files[i];
							select.append(option);
						}

						switch (dir)
						{
							case 'background' : _backgroundChangeFile(); break;
							case 'banners' : _bannerChangeFile(); break;
						}
					}
				});
			});
		});
	}

	/**
	 * Gestion des options de la galerie.
	 *
	 * @return void
	 */
	function _gallery()
	{
		// Largeur de la galerie.
		_galleryChangeSize();

		_qAll('input[name="theme_gallery_size"]').forEach(elem =>
		{
			elem.addEventListener('change', function()
			{
				_themeParams['gallery']['size'] = this.value;
				_galleryChangeSize();
			});
		});

		// Taille du texte.
		_galleryChangeTextSize();

		_q('#theme_text_size').addEventListener('change', function()
		{
			this.querySelectorAll('option').forEach(elem => elem.removeAttribute('selected'));
			this.selectedOptions[0].setAttribute('selected', '');
			_themeParams['text']['size'] = this.selectedOptions[0].value;
			_galleryChangeTextSize();
		});

		// Largeur des descriptions.
		_galleryChangeDescWidthLimit();

		_q('#theme_text_desc_width_limit').checked
			= _themeParams['text']['desc_width_limit'] == 1;

		_q('#theme_text_desc_width_limit').addEventListener('change', function()
		{
			_themeParams['text']['desc_width_limit'] = this.checked ? 1 : 0;
			_galleryChangeDescWidthLimit();
		});

		// Largeur du menu.
		_galleryChangeMenuWidth();

		_q('#theme_menu_width').addEventListener('change', function()
		{
			this.querySelectorAll('option').forEach(elem => elem.removeAttribute('selected'));
			this.selectedOptions[0].setAttribute('selected', '');
			_themeParams['menu']['width'] = this.selectedOptions[0].value;
			_galleryChangeMenuWidth();
		});
	}

	/**
	 * Modification de la largeur des descriptions.
	 *
	 * @return void
	 */
	function _galleryChangeDescWidthLimit()
	{
		_cssProp('--desc-max-width', _themeParams['text']['desc_width_limit'] == 1
			? 'min(80%, 900px)'
			: 'none');
	}

	/**
	 * Modification de la largeur du menu.
	 *
	 * @return void
	 */
	function _galleryChangeMenuWidth()
	{
		const val = _themeParams['menu']['width'];
		_cssProp('--menu-max-width', val + '%');
		_q('#theme_menu_width option').selected = false;

		const current = _q(`#theme_menu_width option[value="${parseInt(val)}"]`);
		current.selected = true;
		current.setAttribute('selected', '');
	}

	/**
	 * Modification de la largeur de la galerie.
	 *
	 * @return void
	 */
	function _galleryChangeSize()
	{
		const gallery_size = _themeParams['gallery']['size'];
		_q('#theme_gallery_size_' + gallery_size).checked = true;
		_q('#gallery').className = gallery_size;
	}

	/**
	 * Modification de la taille du texte.
	 *
	 * @return void
	 */
	function _galleryChangeTextSize()
	{
		const val = _themeParams['text']['size'];
		_cssProp('--text-size', val + '%');

		const current = _q(`#theme_text_size option[value="${parseInt(val)}"]`);
		current.selected = true;
		current.setAttribute('selected', '');
	}

	/**
	 * Initialisation.
	 *
	 * @return void
	 */
	function _init()
	{
		if (typeof theme_params_json == 'object')
		{
			for (const style of ['clear', 'dark'])
			{
				theme_params_json['css'][style]
					= _decodeHTML(theme_params_json['css'][style]);
			}
			_themeParams = JSON.parse(JSON.stringify(theme_params_json));
		}

		if (typeof user_style_params_json['style'] == 'string')
		{
			for (const style of ['clear', 'dark'])
			{
				user_style_params_json['css'][style]
					= _decodeHTML(user_style_params_json['css'][style]);
			}
		}

		// Changement du style par l'utilisateur.
		if (user_style_change && _q('#header_top'))
		{
			_styleButtonChange(user_style, true);
		}
		_q('#user_style_large a').addEventListener('click', _styleButtonClick);

		// Si les options du thème ne sont pas disponibles, on arrête là.
		if (!_q('#theme_template'))
		{
			return;
		}

		// Insertion du code HTML des options du thème.
		_q('#content').append(_q('#theme_template').content.cloneNode(true));

		// Panneau des options.
		_options();

		// Modification du style.
		_style();

		// Modification de la couleur.
		_color();

		// Gestion des textures.
		_textures();

		// Gestion des fichiers.
		_files();

		// Modification de la bannière.
		_banner();

		// Modification du fond.
		_background();

		// Code CSS personnalisé.
		_css();

		// Options de la galerie.
		_gallery();

		// Options des vignettes.
		_thumbs();

		// Options d'affichage.
		_display();
	}

	/**
	 * Gestion du panneau des options.
	 *
	 * @return void
	 */
	function _options()
	{
		// Bouton pour accéder aux options.
		_q('#theme_options_icon a').addEventListener('click', evt =>
		{
			evt.preventDefault();
			_q('#theme_options form').style.display = 'block';
			_q('#theme_options').classList.toggle('open');
		});
		_q('#theme_options').addEventListener('transitionend', function(evt)
		{
			if (evt.target != this)
			{
				return;
			}
			if (_optionsOpen)
			{
				_optionsOpen = false;
				theme_params_json = JSON.parse(JSON.stringify(_themeParams));
				if (user_style_change == 1)
				{
					const is_style = typeof user_style_params_json['style'] != 'undefined';
					if (is_style)
					{
						_themeParams = JSON.parse(JSON.stringify(user_style_params_json));
					}
					_styleChange(is_style ? user_style : theme_style);
				}
				_q('#theme_options_icon a').focus();

				// Quand les options sont cachées, on désactive l'affichage du
				// formulaire afin d'éviter que les éléments du formulaire soient
				// atteignables lors d'une navigation au clavier.
				_q('#theme_options form').style.display = 'none';
			}
			else
			{
				_optionsOpen = true;
				_themeParams = JSON.parse(JSON.stringify(theme_params_json));
				_styleChange(theme_style);
			}
		});

		// Menu.
		_qAll('#theme_options_menu a').forEach(elem =>
		{
			elem.addEventListener('click', function(evt)
			{
				evt.preventDefault();
				_qAll('#theme_options fieldset').forEach(elem => elem.style.display = 'none');
				_q('#theme_options_menu a.current').classList.remove('current');
				_q('#' + this.getAttribute('data-id')).style.display = 'block';
				this.classList.add('current');
			});
		});

		// Aide contextuelle.
		_qAll('#theme_options .infos_icon').forEach(elem =>
		{
			elem.addEventListener('click', evt =>
			{
				evt.preventDefault();
				const i = _q('#' + elem.getAttribute('data-id'));
				i.style.display = i.checkVisibility() ? 'none' : 'block';
			});
		});

		// Enregistrement des paramètres.
		const submit = _q('#theme_options input[type="submit"]');
		function theme_report_hide()
		{
			_qAll('div[id^="theme_report"]').forEach(elem => elem.style.display = 'none');
			submit.removeAttribute('disabled');
			submit.blur();
		}
		_q('#theme_options form').addEventListener('submit', evt =>
		{
			evt.preventDefault();

			theme_params_json = JSON.parse(JSON.stringify(_themeParams));
			if (typeof user_style_params_json['style'] != 'undefined')
			{
				user_style_params_json = JSON.parse(JSON.stringify(_themeParams));
			}
			function message(type, message)
			{
				const options = {duration: 250, fill: 'forwards'};
				const report = _q('#theme_report_' + type);

				_q(`#theme_report_${type} span`).textContent = message;
				_q('#theme_loading').style.display = 'none';
				report.style.opacity = 0;
				report.style.display = 'flex';

				report.animate({opacity: 1}, options);
				options.delay = 2000;
				report.animate({opacity: 0}, options).finished.then(theme_report_hide);
			}

			submit.setAttribute('disabled', '');
			_q('#theme_loading').style.display = 'block';

			_ajax(
			{
				section: 'theme-params',
				template: template_dir,
				params: _themeParams
			},
			{
				error: r => message(r.status, r.message),
				info: r => message(r.status, r.message),
				success: r => message(r.status, r.message),
			});
		});

		// Bouton "annuler".
		_q('#theme_options form input[name="cancel"]')
			.addEventListener('click', () => _q('#theme_options_icon a').click());

		// Message de rapport.
		_qAll('div[id^="theme_report"]')
			.forEach(elem => elem.addEventListener('click', theme_report_hide));
	}

	/**
	 * Gestion du style.
	 *
	 * @return void
	 */
	function _style()
	{
		// Style clair ou sombre.
		_q('#theme_style_clear').checked = theme_style == 'clear';
		_q('#theme_style_dark').checked = theme_style == 'dark';

		_qAll('input[name="theme_style"]').forEach(elem =>
		{
			elem.addEventListener('change', function()
			{
				theme_style = this.value;
				_styleChange(theme_style);
				_styleButtonChange(theme_style, _q('#user_style_large.active') ? true : false);
			});
		});

		// Autoriser les visiteurs à changer le style.
		_q('#theme_style_change').checked = _themeParams['style_change'] == 1;

		_q('#theme_style_change').addEventListener('change', function()
		{
			_themeParams['style_change'] = user_style_change = this.checked ? 1 : 0;
			_styleButtonChange(_themeParams['style'], this.checked);
		});

		// Boutons de formulaires.
		_styleChangeFormButtons();

		_qAll('input[name="theme_form_buttons_style"]').forEach(elem =>
		{
			elem.addEventListener('change', function()
			{
				_themeParams['buttons'][_themeParams['style']]['style'] = this.value;
				_styleChangeFormButtons();
			});
		});
	}

	/**
	 * Modification du bouton de changement de style.
	 *
	 * @param string style
	 * @param bool active
	 *
	 * @return void
	 */
	function _styleButtonChange(style, active)
	{
		const large = _q('#user_style_large');
		const large_a = _q('#user_style_large a');
		large.classList.remove('active');

		if (_q('#user_style_small'))
		{
			_q('#user_style_small').remove();
		}

		if (active)
		{
			const cursor = _optionsOpen ? 'not-allowed' : 'pointer';
			const data_style = style == 'dark' ? 'clear' : 'dark';
			const icon = style == 'dark' ? '&#xe902;' : '&#xe908;';
			const tag = _q('#user_menu') ? 'li' : 'div';

			large.classList.add('active');
			large.setAttribute('data-style', data_style);
			large_a.replaceChildren();
			large_a.insertAdjacentHTML('afterbegin', icon);
			large_a.style.cursor = cursor;

			_q(_q('#user_menu') ? '#user_menu' : '#title').insertAdjacentHTML(
				_q('#user_menu') ? 'afterbegin' : 'afterend',
				`<${tag} data-style="${data_style}" id="user_style_small">
					<a href="javascript:;">${icon}</a>
				</${tag}>`
			);

			_q('#user_style_small a').style.cursor = cursor;
			_q('#user_style_small a').addEventListener('click', _styleButtonClick);
		}
	}

	/**
	 * Gestion de l'événement "click" sur le bouton de changement de style.
	 *
	 * @param object evt
	 *
	 * @return void
	 */
	function _styleButtonClick(evt)
	{
		evt.preventDefault();

		if (_q('#theme_options') && _optionsOpen)
		{
			return;
		}

		user_style = this.parentElement.getAttribute('data-style');
		if (typeof user_style_params_json['style'] != 'undefined')
		{
			_themeParams = JSON.parse(JSON.stringify(user_style_params_json));
		}
		else
		{
			user_style_params_json = JSON.parse(JSON.stringify(theme_params_json));
		}
		_styleButtonChange(user_style, true);
		_styleChange(user_style);
		_cssProp('--color', _themeParams['color'][user_style]);

		_ajax(
		{
			section: 'cookie-prefs-write',
			param: 'theme_default_user_style',
			value: user_style
		});
	}

	/**
	 * Effectue toutes les modifications liées au changement de style.
	 *
	 * @param string style
	 *
	 * @return void
	 */
	function _styleChange(style)
	{
		if (_q('#style_dark'))
		{
			_q('#style_dark').remove();
		}
		if (style == 'dark')
		{
			_q('#style_clear').insertAdjacentHTML('afterend', style_dark);
		}

		_themeParams['style'] = style;

		_texturesCSS('banner', _themeParams['banner'][style]['texture']);
		_texturesCSS('background', _themeParams['background'][style]['texture']);

		_styleButtonChange(style, user_style_change == 1);

		if (_q('#theme_options'))
		{
			_colorChange(_themeParams['color'][style]);

			_texturesList('banner');
			_texturesList('background');

			_bannerChangeType();
			_bannerChangeHeight();
			_backgroundChangeType();
			_styleChangeFormButtons();
			_cssChange();
		}
		else
		{
			_backgroundChangeCSS();
			_bannerChangeCSS();
			_bannerChangeHeightCSS();
			_styleChangeFormButtonsCSS();
		}
	}

	/**
	 * Style des boutons de formulaires.
	 *
	 * @return void
	 */
	function _styleChangeFormButtons()
	{
		_styleChangeFormButtonsCSS();

		_q('#theme_form_buttons_style_1').checked = false;
		_q('#theme_form_buttons_style_2').checked = false;
		_q('#theme_form_buttons_style_'
			+ _themeParams['buttons'][_themeParams['style']]['style']).checked = true;
	}

	/**
	 * Modification du code CSS des boutons de formulaires.
	 *
	 * @return void
	 */
	function _styleChangeFormButtonsCSS()
	{
		_q('body').classList.toggle(
			'btn_2_' + _themeParams['style'],
			_themeParams['buttons'][_themeParams['style']]['style'] == 2
		);
		_q('body').classList.toggle(
			'btn_2_' + (_themeParams['style'] == 'dark' ? 'clear' : 'dark'),
			false
		);
	}

	/**
	 * Gestion des textures.
	 *
	 * @return void
	 */
	function _textures()
	{
		// Création des listes de textures.
		_texturesList('banner');
		_texturesList('background');

		// Boutons de navigation au sein des textures.
		_qAll('.theme_textures_next,.theme_textures_prev').forEach(elem =>
		{
			elem.addEventListener('click', function()
			{
				if (!elem.classList.contains('disabled'))
				{
					_texturesList(
						this.parentElement.getAttribute('data-type'),
						elem.classList.contains('theme_textures_next') ? 1 : -1
					);
				}
			});
		});
	}

	/**
	 * Modifie les paramètres CSS des textures.
	 *
	 * @param string name
	 * @param string file
	 *
	 * @return void
	 */
	function _texturesCSS(name, file)
	{
		const style = _themeParams['style'];
		if (_themeParams[name][style]['type'] == 'texture')
		{
			const color = theme_textures_json[name][style][file] == ''
				? _themeParams['color'][_themeParams['style']]
				: theme_textures_json[name][style][file];
			const attachment = name == 'background' && _themeParams[name][style]['fixed'] == 1
				? 'fixed'
				: 'scroll';

			_cssProp(`--${name}-attachment`, attachment);
			_cssProp(`--${name}-color`, color);
			_cssProp(`--${name}-image`, `url(${theme_textures_dir}${file})`);
			_cssProp(`--${name}-size`, 'auto');
			_cssProp(`--${name}-position`, '0 0');
			_cssProp(`--${name}-repeat`, 'repeat');
		}
	}

	/**
	 * Fabrication des listes de textures.
	 *
	 * @param string type
	 * @param int next_page
	 *
	 * @return void
	 */
	function _texturesList(type, next_page = 0)
	{
		const list = _q(`#theme_options div[data-type="${type}"]`);
		const style = _themeParams['style'];
		const items_count = Object.keys(theme_textures_json[type][style]).length;
		const nb_per_page = 3;
		const pages_count = Math.ceil(items_count / nb_per_page);

		// Image et page courantes.
		let current_page = 1;
		let current_item = 0;
		let i = 1;
		for (const file in theme_textures_json[type][style])
		{
			if (file == _themeParams[type][style]['texture'])
			{
				current_item = i;
				current_page = Math.ceil(i / nb_per_page);
				break;
			}
			i++;
		}
		if (next_page)
		{
			current_page = parseInt(list.getAttribute('data-page')) + next_page;
		}
		list.setAttribute('data-page', current_page);

		// Création de chaque image.
		const end = current_page * nb_per_page;
		const start = 1 + end - nb_per_page;
		const images = list.querySelector('.theme_textures_images');
		images.replaceChildren();
		i = 1;
		for (const file in theme_textures_json[type][style])
		{
			if (i >= start && i <= end)
			{
				const div = document.createElement('div');
				const color = theme_textures_json[type][style][file] == ''
					? _themeParams['color'][_themeParams['style']]
					: theme_textures_json[type][style][file];
				div.setAttribute('data-texture', file);
				if (i == current_item)
				{
					div.classList.add('selected');
				}
				div.style.background = `${color} url(${theme_textures_dir}${file})`;
				images.append(div);
			}
			i++;
		}

		// Boutons de navigation entre les pages de textures.
		list.querySelectorAll('div:has(span)').forEach(elem => elem.classList.add('disabled'));
		if (current_page > 1)
		{
			list.querySelector('.theme_textures_prev').classList.remove('disabled');
		}
		if (current_page != pages_count)
		{
			list.querySelector('.theme_textures_next').classList.remove('disabled');
		}

		// Sélection d'une texture.
		images.querySelectorAll('div').forEach(elem =>
		{
			elem.addEventListener('click', function()
			{
				const file = this.getAttribute('data-texture');
				const type = this.closest('.theme_textures').getAttribute('data-type');
				const current = _themeParams[type][_themeParams['style']]['texture'];

				if (file != current)
				{
					_texturesCSS(type, file);
					_themeParams[type][_themeParams['style']]['texture'] = file;
					this.parentElement.querySelectorAll('div')
						.forEach(elem => elem.classList.remove('selected'));
					this.classList.add('selected');
				}
			});
		});
	}

	/**
	 * Gestion des vignettes.
	 *
	 * @return void
	 */
	function _thumbs()
	{
		_q('#theme_thumbs_cat_size_' + _themeParams['thumbs_cat']['size'].replace(' ', '_'))
			.checked = true;
		_q('#theme_thumbs_cat_type_' + _themeParams['thumbs_cat']['layout']).checked = true;
		_q('#theme_thumbs_items_size_' + _themeParams['thumbs_items']['size']).checked = true;
		_q('#theme_thumbs_items_type_' + _themeParams['thumbs_items']['layout']).checked = true;

		_qAll('input[name="theme_thumbs_cat_size"]').forEach(elem =>
		{
			elem.addEventListener('change', function()
			{
				_themeParams['thumbs_cat']['size'] = this.value;
				_galleryChangeSize();
				_thumbsCatChange();
			});
		});

		_qAll('input[name="theme_thumbs_cat_type"]').forEach(elem =>
		{
			elem.addEventListener('change', function()
			{
				_themeParams['thumbs_cat']['layout'] = this.value;
				_thumbsCatChange();
			});
		});

		_qAll('input[name="theme_thumbs_items_size"]').forEach(elem =>
		{
			elem.addEventListener('change', function()
			{
				_themeParams['thumbs_items']['size'] = this.value;
				_galleryChangeSize();
				_thumbsItemsChange();
			});
		});

		_qAll('input[name="theme_thumbs_items_type"]').forEach(elem =>
		{
			elem.addEventListener('change', function()
			{
				_themeParams['thumbs_items']['layout'] = this.value;
				_thumbsItemsChange();
			});
		});
	}

	/**
	 * Modification des vignettes de catégories.
	 *
	 * @return void
	 */
	function _thumbsCatChange()
	{
		if (!_q('#thumbs_cat'))
		{
			return;
		}

		const cat_id = [];
		const layout = _themeParams['thumbs_cat']['layout'];
		const size = _themeParams['thumbs_cat']['size'];
		const thumb_size = () =>
		{
			switch (layout)
			{
				case 'portrait' :
					return size == 'large' ? '363x544' : '232x348';
				case 'square' :
					return size == 'large' ? '363x363' : '232x232';
				default :
					return size == 'large' ? '363x262' : '232x167';
			}
		};

		_qAll('#thumbs_cat dt a').forEach(elem =>
		{
			cat_id.push(parseInt(elem.getAttribute('data-id')));
		});
		if (!cat_id.length)
		{
			return;
		}

		_ajax(
		{
			section: 'thumbs-cat-source',
			cat_id: cat_id,
			size: thumb_size()
		},
		{
			error: r => alert('error: ' + r.message),
			success: r =>
			{
				_q('#thumbs_cat').classList.remove('small', 'large',
					'landscape', 'portrait', 'square', 'standard');
				_q('#thumbs_cat').classList.add(layout);
				_q('#thumbs_cat').classList.add(size);
				for (const id in r.thumbs_source)
				{
					_q(`#thumbs_cat dt a[data-id="${id}"] img`)
						.setAttribute('src', r.thumbs_source[id]);
				}
			}
		});
	}

	/**
	 * Modification des vignettes de fichiers.
	 *
	 * @return void
	 */
	function _thumbsItemsChange()
	{
		if (!_qAll('.thumbs_items'))
		{
			return;
		}

		const items_id = [];
		const layout = _themeParams['thumbs_items']['layout'];
		const size = _themeParams['thumbs_items']['size'];
		const thumb_size = () =>
		{
			switch (layout)
			{
				case 'landscape' :
					return size == 'superlarge'
						? '378x252'
						: (size == 'large' ? '248x165' : '161x107');
				case 'portrait' :
					return size == 'superlarge'
						? '378x566'
						: (size == 'large' ? '248x371' : '161x241');
				case 'square' :
					return size == 'superlarge'
						? '378x378'
						: (size == 'large' ? '248x248' : '161x161');
				default :
					return size == 'superlarge'
						? '368'
						: (size == 'large' ? '238' : '151');
			}
		};

		_qAll('.thumbs_items *[data-id]').forEach(elem =>
		{
			items_id.push(parseInt(elem.getAttribute('data-id')));
		});
		if (!items_id.length)
		{
			return;
		}

		_ajax(
		{
			section: 'thumbs-items-source',
			items_id: items_id,
			size: thumb_size()
		},
		{
			error: r => alert('error: ' + r.message),
			success: r =>
			{
				_qAll('.thumbs_items').forEach(elem =>
				{
					elem.classList.remove('small', 'large', 'superlarge',
						'landscape', 'portrait', 'square', 'standard');
				});
				_qAll('.thumbs_items').forEach(elem => elem.classList.add(layout));
				_qAll('.thumbs_items').forEach(elem => elem.classList.add(size));
				for (const id in r.thumbs_source)
				{
					_qAll(`.thumbs_items *[data-id="${id}"] img`)
						.forEach(elem => elem.setAttribute('src', r.thumbs_source[id]));
				}
			}
		});
	}
}