<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

/**
 * This class represents a text property in a property form.
 *
 * @author Jörg Lützenkirchen <luetzenkirchen@leifos.com>
 */
class ilScheduleInputGUI extends ilFormPropertyGUI
{
    protected \ILIAS\DI\UIServices $ui;
    protected ilGlobalTemplateInterface $tpl;
    protected array $value = [];
    protected string $validationFailureMessage;

    public function __construct(
        string $a_title = "",
        string $a_postvar = ""
    ) {
        global $DIC;

        $this->lng = $DIC->language();
        $this->tpl = $DIC["tpl"];
        parent::__construct($a_title, $a_postvar);
        $this->ui = $DIC->ui();
    }

    public function setValue(array $a_value): void
    {
        $this->value = $a_value;
    }

    public function getValue(): array
    {
        return $this->value;
    }

    public function setValidationFailureMessage(string $a_msg): void
    {
        $this->validationFailureMessage = $a_msg;
    }

    public function getValidationFailureMessage(): string
    {
        return $this->validationFailureMessage;
    }

    /**
     * Set value by array
     *
     * @param	array	$a_values	value array
     */
    public function setValueByArray(array $a_values): void
    {
        $this->setValue($this->getPostData($this->getPostVar(), false));
    }

    public function checkInput(): bool
    {
        $lng = $this->lng;
        $data = $this->getPostData($this->getPostVar(), false);
        if (count($data)) {
            // slots may not overlap
            foreach ($data as $slot => $days) {
                if (!$days) {
                    $this->setAlert($lng->txt("msg_input_does_not_match_regexp"));
                    return false;
                }

                $parts = explode("-", $slot);
                $from = str_replace(":", "", $parts[0]);
                $to = str_replace(":", "", $parts[1]);
                if ($from >= $to) {
                    $this->setAlert($lng->txt("msg_input_does_not_match_regexp"));
                    return false;
                }

                foreach ($data as $rslot => $rdays) {
                    if ($slot != $rslot && $rdays && array_intersect($days, $rdays)) {
                        $rparts = explode("-", $rslot);
                        $rfrom = str_replace(":", "", $rparts[0]);
                        $rto = str_replace(":", "", $rparts[1]);

                        if (($rfrom > $from && $rfrom < $to) ||
                            ($rto > $from && $rto < $to) ||
                            ($rfrom < $from && $rto > $to)) {
                            $this->setAlert($lng->txt("msg_input_does_not_match_regexp"));
                            return false;
                        }
                    }
                }
            }

            return true;
        }

        if ($this->getRequired()) {
            $this->setAlert($lng->txt("msg_input_is_required"));
            return false;
        }

        return true;
    }

    public function getInput(): array
    {
        return $this->getPostData($this->getPostVar());
    }

    public function getPostData(
        string $a_post_var,
        $a_remove_invalid = true
    ): array {
        $res = array();
        for ($loop = 0; $loop < 240; $loop++) {
            $days = $this->strArray($a_post_var . "_days~" . $loop);
            $from = self::parseTime(
                $this->str($a_post_var . "_from_hh~" . $loop),
                $this->str($a_post_var . "_from_mm~" . $loop)
            );
            $to = self::parseTime(
                $this->str($a_post_var . "_to_hh~" . $loop),
                $this->str($a_post_var . "_to_mm~" . $loop)
            );

            // only if any part was edited (js based gui)
            if ($days || $from !== "00:00" || $to !== "00:00") {
                $slot = $from . "-" . $to;
                if ($days) {
                    if (isset($res[$slot])) {
                        $res[$slot] = array_unique(array_merge($res[$slot], $days));
                    } else {
                        $res[$slot] = $days;
                    }
                } else {
                    $res[$slot] = array();
                }

                if ($a_remove_invalid && !($days && $from && $to && $from !== $to)) {
                    unset($res[$slot]);
                }
            }
        }
        return $res;
    }

    protected function render(
        string $a_mode = ""
    ): string {
        $lng = $this->lng;

        $tpl = new ilTemplate("tpl.schedule_input.html", true, true, "Modules/BookingManager");

        $lng->loadLanguageModule("dateplaner");

        $def = $this->getValue();
        if (!$def) {
            $def = array(null => null);
        }

        $days = array("Mo", "Tu", "We", "Th", "Fr", "Sa", "Su");
        $row = 0;
        foreach ($def as $slot => $days_select) {
            $tpl->setCurrentBlock("days");
            foreach ($days as $day) {
                $day_value = strtolower($day);

                $tpl->setVariable("ROW", $row);
                $tpl->setVariable("ID", $this->getFieldId());
                $tpl->setVariable("POST_VAR", $this->getPostVar());
                $tpl->setVariable("DAY", $day_value);
                $tpl->setVariable("TXT_DAY", $lng->txt($day . "_short"));

                if ($days_select && in_array($day_value, $days_select, true)) {
                    $tpl->setVariable("DAY_STATUS", " checked=\"checked\"");
                }

                $tpl->parseCurrentBlock();
            }

            $add_gl = $this->ui->factory()->symbol()->glyph()->add();
            $rem_gl = $this->ui->factory()->symbol()->glyph()->remove();
            $r = $this->ui->renderer();
            $tpl->setCurrentBlock("row");
            $tpl->setVariable("ROW", $row);
            $tpl->setVariable("ID", $this->getFieldId());
            $tpl->setVariable("POST_VAR", $this->getPostVar());
            $tpl->setVariable("TXT_FROM", $lng->txt("cal_from"));
            $tpl->setVariable("TXT_TO", $lng->txt("cal_until"));
            $tpl->setVariable("IMG_MULTI_ADD", $r->render($add_gl));
            $tpl->setVariable("IMG_MULTI_REMOVE", $r->render($rem_gl));
            $tpl->setVariable("TXT_MULTI_ADD", $lng->txt("add"));
            $tpl->setVariable("TXT_MULTI_REMOVE", $lng->txt("remove"));

            if ($slot) {
                $parts = explode("-", $slot);
                $from = explode(":", $parts[0]);
                $to = explode(":", $parts[1]);

                $tpl->setVariable("FROM_HH_VALUE", $from[0]);
                $tpl->setVariable("FROM_MM_VALUE", $from[1]);
                $tpl->setVariable("TO_HH_VALUE", $to[0]);
                $tpl->setVariable("TO_MM_VALUE", $to[1]);
            }

            // manage hidden buttons
            if ($row > 0) {
                // inline needed because of JS
                $tpl->setVariable("ADD_STYLE", " style=\"display:none\"");
                // $tpl->setVariable("ADD_CLASS", "ilNoDisplay");
            } else {
                // inline needed because of JS
                $tpl->setVariable("RMV_STYLE", " style=\"display:none\"");
                // $tpl->setVariable("RMV_CLASS", "ilNoDisplay");
            }

            $tpl->parseCurrentBlock();

            $row++;
        }

        return $tpl->get();
    }

    public function insert(ilTemplate $a_tpl): void
    {
        $tpl = $this->tpl;

        $tpl->addJavascript("Modules/BookingManager/js/ScheduleInput.js");

        $html = $this->render();

        $a_tpl->setCurrentBlock("prop_generic");
        $a_tpl->setVariable("PROP_GENERIC", $html);
        $a_tpl->parseCurrentBlock();
    }

    /**
     * Parse/normalize incoming time values
     */
    protected static function parseTime(
        string $a_hours,
        string $a_minutes
    ): string {
        $hours = (int) $a_hours;
        $min = (int) $a_minutes;
        if ($hours > 23) {
            $hours = 23;
            $min = 59;
        }
        if ($min > 59) {
            $min = 59;
        }
        return str_pad($hours, 2, "0", STR_PAD_LEFT) . ":" .
            str_pad($min, 2, "0", STR_PAD_LEFT);
    }
}
