<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

use ILIAS\TestQuestionPool\ManipulateThumbnailsInChoiceQuestionsTrait;

/**
 * @author		Björn Heyser <bheyser@databay.de>
 * @version		$Id$
 *
 * @package     Modules/TestQuestionPool
 */
class assKprimChoice extends assQuestion implements ilObjQuestionScoringAdjustable, ilObjAnswerScoringAdjustable, ilAssSpecificFeedbackOptionLabelProvider, ilAssQuestionLMExportable, ilAssQuestionAutosaveable
{
    use ManipulateThumbnailsInChoiceQuestionsTrait;

    public const NUM_REQUIRED_ANSWERS = 4;

    public const PARTIAL_SCORING_NUM_CORRECT_ANSWERS = 3;

    public const ANSWER_TYPE_SINGLE_LINE = 'singleLine';
    public const ANSWER_TYPE_MULTI_LINE = 'multiLine';

    public const OPTION_LABEL_RIGHT_WRONG = 'right_wrong';
    public const OPTION_LABEL_PLUS_MINUS = 'plus_minus';
    public const OPTION_LABEL_APPLICABLE_OR_NOT = 'applicable_or_not';
    public const OPTION_LABEL_ADEQUATE_OR_NOT = 'adequate_or_not';
    public const OPTION_LABEL_CUSTOM = 'customlabel';

    public const DEFAULT_THUMB_SIZE = 150;
    public const THUMB_PREFIX = 'thumb.';

    private $shuffleAnswersEnabled;

    private $answerType;

    private $thumbSize;

    private $scorePartialSolutionEnabled;

    private $optionLabel;

    private $customTrueOptionLabel;

    private $customFalseOptionLabel;

    private $specificFeedbackSetting;

    private $answers;

    public function __construct($title = '', $comment = '', $author = '', $owner = -1, $question = '')
    {
        parent::__construct($title, $comment, $author, $owner, $question);

        $this->shuffleAnswersEnabled = true;
        $this->answerType = self::ANSWER_TYPE_SINGLE_LINE;
        $this->thumbSize = self::DEFAULT_THUMB_SIZE;
        $this->scorePartialSolutionEnabled = true;
        $this->optionLabel = self::OPTION_LABEL_RIGHT_WRONG;
        $this->customTrueOptionLabel = '';
        $this->customFalseOptionLabel = '';

        $this->specificFeedbackSetting = ilAssConfigurableMultiOptionQuestionFeedback::FEEDBACK_SETTING_ALL;

        $this->answers = [];
    }

    public function getQuestionType(): string
    {
        return 'assKprimChoice';
    }

    public function getAdditionalTableName(): string
    {
        return "qpl_qst_kprim";
    }

    public function getAnswerTableName(): string
    {
        return "qpl_a_kprim";
    }

    public function setShuffleAnswersEnabled($shuffleAnswersEnabled): void
    {
        $this->shuffleAnswersEnabled = $shuffleAnswersEnabled;
    }

    public function isShuffleAnswersEnabled(): bool
    {
        return $this->shuffleAnswersEnabled;
    }

    public function setAnswerType($answerType): void
    {
        $this->answerType = $answerType;
    }

    public function getAnswerType(): string
    {
        return $this->answerType;
    }

    public function setThumbSize(int $thumbSize): void
    {
        $this->thumbSize = $thumbSize;
    }

    public function getThumbSize(): int
    {
        return $this->thumbSize;
    }

    public function setScorePartialSolutionEnabled($scorePartialSolutionEnabled): void
    {
        $this->scorePartialSolutionEnabled = $scorePartialSolutionEnabled;
    }

    public function isScorePartialSolutionEnabled(): bool
    {
        return $this->scorePartialSolutionEnabled;
    }

    public function setOptionLabel($optionLabel): void
    {
        $this->optionLabel = $optionLabel;
    }

    public function getOptionLabel(): string
    {
        return $this->optionLabel;
    }

    public function setCustomTrueOptionLabel($customTrueOptionLabel): void
    {
        $this->customTrueOptionLabel = $customTrueOptionLabel;
    }

    public function getCustomTrueOptionLabel()
    {
        return $this->customTrueOptionLabel;
    }

    public function setCustomFalseOptionLabel($customFalseOptionLabel): void
    {
        $this->customFalseOptionLabel = $customFalseOptionLabel;
    }

    public function getCustomFalseOptionLabel()
    {
        return $this->customFalseOptionLabel;
    }

    public function setSpecificFeedbackSetting($specificFeedbackSetting): void
    {
        $this->specificFeedbackSetting = $specificFeedbackSetting;
    }

    public function getSpecificFeedbackSetting(): int
    {
        return $this->specificFeedbackSetting;
    }

    public function setAnswers($answers): void
    {
        if (is_null($answers)) {
            return;
        }
        $clean_answer_text = function (ilAssKprimChoiceAnswer $answer) {
            $answer->setAnswertext(
                $this->getHtmlQuestionContentPurifier()->purify($answer->getAnswertext())
            );
            return $answer;
        };
        $this->answers = array_map($clean_answer_text, $answers);
    }

    public function getAnswers(): array
    {
        return $this->answers;
    }

    public function getAnswer($position)
    {
        foreach ($this->getAnswers() as $answer) {
            if ($answer->getPosition() == $position) {
                return $answer;
            }
        }

        return null;
    }

    public function addAnswer(ilAssKprimChoiceAnswer $answer): void
    {
        $answer->setAnswertext(
            $this->getHtmlQuestionContentPurifier()->purify($answer->getAnswertext())
        );
        $this->answers[] = $answer;
    }

    public function loadFromDb($questionId): void
    {
        $res = $this->db->queryF($this->buildQuestionDataQuery(), ['integer'], [$questionId]);

        while ($data = $this->db->fetchAssoc($res)) {
            $this->setId($questionId);

            $this->setOriginalId($data['original_id']);

            $this->setObjId($data['obj_fi']);

            $this->setTitle($data['title'] ?? '');
            $this->setNrOfTries($data['nr_of_tries']);
            $this->setComment($data['description'] ?? '');
            $this->setAuthor($data['author']);
            $this->setPoints($data['points']);
            $this->setOwner($data['owner']);
            $this->setLastChange($data['tstamp']);
            $this->setQuestion(ilRTE::_replaceMediaObjectImageSrc($data['question_text'] ?? '', 1));

            $this->setShuffleAnswersEnabled((bool) $data['shuffle_answers']);

            if ($this->isValidAnswerType($data['answer_type'])) {
                $this->setAnswerType($data['answer_type']);
            }

            if (is_numeric($data['thumb_size'])) {
                $this->setThumbSize((int) $data['thumb_size']);
            }

            if ($this->isValidOptionLabel($data['opt_label'])) {
                $this->setOptionLabel($data['opt_label']);
            }

            if ($data['custom_true'] !== null) {
                $this->setCustomTrueOptionLabel($data['custom_true']);
            }

            if ($data['custom_false'] !== null) {
                $this->setCustomFalseOptionLabel($data['custom_false']);
            }

            if ($data['score_partsol'] !== null) {
                $this->setScorePartialSolutionEnabled((bool) $data['score_partsol']);
            }

            if (isset($data['feedback_setting'])) {
                $this->setSpecificFeedbackSetting((int) $data['feedback_setting']);
            }

            try {
                $this->setLifecycle(ilAssQuestionLifecycle::getInstance($data['lifecycle']));
            } catch (ilTestQuestionPoolInvalidArgumentException $e) {
                $this->setLifecycle(ilAssQuestionLifecycle::getDraftInstance());
            }

            try {
                $this->setAdditionalContentEditingMode($data['add_cont_edit_mode']);
            } catch (ilTestQuestionPoolException $e) {
            }
        }

        $this->loadAnswerData($questionId);

        parent::loadFromDb($questionId);
    }

    private function loadAnswerData($questionId): void
    {
        global $DIC;
        $ilDB = $DIC['ilDB'];

        $res = $this->db->queryF(
            "SELECT * FROM {$this->getAnswerTableName()} WHERE question_fi = %s ORDER BY position ASC",
            ['integer'],
            [$questionId]
        );

        while ($data = $ilDB->fetchAssoc($res)) {
            $answer = new ilAssKprimChoiceAnswer();

            $answer->setPosition($data['position']);

            $answer->setAnswertext(ilRTE::_replaceMediaObjectImageSrc($data['answertext'] ?? '', 1));

            $answer->setImageFile($data['imagefile']);
            $answer->setThumbPrefix($this->getThumbPrefix());
            $answer->setImageFsDir($this->getImagePath());
            $answer->setImageWebDir($this->getImagePathWeb());

            $answer->setCorrectness($data['correctness']);

            $this->answers[$answer->getPosition()] = $answer;
        }

        for ($i = count($this->answers); $i < self::NUM_REQUIRED_ANSWERS; $i++) {
            $answer = new ilAssKprimChoiceAnswer();

            $answer->setPosition($i);

            $this->answers[$answer->getPosition()] = $answer;
        }
    }

    public function saveToDb($originalId = ''): void
    {
        if ($originalId == '') {
            $this->saveQuestionDataToDb();
        } else {
            $this->saveQuestionDataToDb($originalId);
        }

        $this->saveAdditionalQuestionDataToDb();
        $this->saveAnswerSpecificDataToDb();

        parent::saveToDb($originalId);
    }

    public function saveAdditionalQuestionDataToDb()
    {
        $this->db->replace(
            $this->getAdditionalTableName(),
            [
                'question_fi' => ['integer', $this->getId()]
            ],
            [
                'shuffle_answers' => ['integer', (int) $this->isShuffleAnswersEnabled()],
                'answer_type' => ['text', $this->getAnswerType()],
                'thumb_size' => ['integer', $this->getThumbSize()],
                'opt_label' => ['text', $this->getOptionLabel()],
                'custom_true' => ['text', $this->getCustomTrueOptionLabel()],
                'custom_false' => ['text', $this->getCustomFalseOptionLabel()],
                'score_partsol' => ['integer', (int) $this->isScorePartialSolutionEnabled()],
                'feedback_setting' => ['integer', $this->getSpecificFeedbackSetting()]
            ]
        );
    }

    public function saveAnswerSpecificDataToDb()
    {
        foreach ($this->getAnswers() as $answer) {
            $this->db->replace(
                $this->getAnswerTableName(),
                [
                    'question_fi' => ['integer', $this->getId()],
                    'position' => ['integer', (int) $answer->getPosition()]
                ],
                [
                    'answertext' => ['text', $answer->getAnswertext()],
                    'imagefile' => ['text', $answer->getImageFile()],
                    'correctness' => ['integer', (int) $answer->getCorrectness()]
                ]
            );
        }
    }

    public function isComplete(): bool
    {
        foreach ([$this->title, $this->author, $this->question] as $text) {
            if (!strlen($text)) {
                return false;
            }
        }

        if (!isset($this->points)) {
            return false;
        }

        foreach ($this->getAnswers() as $answer) {
            /* @var ilAssKprimChoiceAnswer $answer */

            if (is_null($answer->getCorrectness())) {
                return false;
            }

            if (
                (!is_string($answer->getAnswertext()) || $answer->getAnswertext() === '') &&
                (!is_string($answer->getImageFile()) || $answer->getImageFile() === '')
            ) {
                return false;
            }
        }

        return true;
    }

    /**
     * Saves the learners input of the question to the database.
     *
     * @access public
     * @param integer $active_id Active id of the user
     * @param integer $pass Test pass
     * @return boolean $status
     */
    public function saveWorkingData($active_id, $pass = null, $authorized = true): bool
    {
        /** @var ilDBInterface $ilDB */
        $ilDB = $GLOBALS['DIC']['ilDB'];

        if ($pass === null) {
            $pass = ilObjTest::_getPass($active_id);
        }

        $entered_values = 0;

        $this->getProcessLocker()->executeUserSolutionUpdateLockOperation(function () use (&$entered_values, $active_id, $pass, $authorized) {
            $this->removeCurrentSolution($active_id, $pass, $authorized);

            $solutionSubmit = $this->getSolutionSubmit();

            foreach ($solutionSubmit as $answerIndex => $answerValue) {
                if ($answerValue !== null) {
                    $this->saveCurrentSolution($active_id, $pass, (int) $answerIndex, (int) $answerValue, $authorized);
                    $entered_values++;
                }
            }
        });

        if ($entered_values) {
            if (ilObjAssessmentFolder::_enabledAssessmentLogging()) {
                assQuestion::logAction($this->lng->txtlng(
                    "assessment",
                    "log_user_entered_values",
                    ilObjAssessmentFolder::_getLogLanguage()
                ), $active_id, $this->getId());
            }
        } else {
            if (ilObjAssessmentFolder::_enabledAssessmentLogging()) {
                assQuestion::logAction($this->lng->txtlng(
                    "assessment",
                    "log_user_not_entered_values",
                    ilObjAssessmentFolder::_getLogLanguage()
                ), $active_id, $this->getId());
            }
        }

        return true;
    }

    /**
     * Returns the points, a learner has reached answering the question.
     * The points are calculated from the given answers.
     *
     * @access public
     * @param integer $active_id
     * @param integer $pass
     * @param boolean $returndetails (deprecated !!)
     * @return float/array $points/$details (array $details is deprecated !!)
     */
    public function calculateReachedPoints($active_id, $pass = null, $authorizedSolution = true, $returndetails = false): float
    {
        if ($returndetails) {
            throw new ilTestException('return details not implemented for ' . __METHOD__);
        }

        global $DIC;
        $ilDB = $DIC['ilDB'];

        $found_values = [];
        if (is_null($pass)) {
            $pass = $this->getSolutionMaxPass($active_id);
        }

        $result = $this->getCurrentSolutionResultSet($active_id, $pass, $authorizedSolution);

        while ($data = $ilDB->fetchAssoc($result)) {
            $found_values[(int) $data['value1']] = (int) $data['value2'];
        }

        $points = $this->calculateReachedPointsForSolution($found_values, $active_id);

        return $points;
    }

    public function getValidAnswerTypes(): array
    {
        return [self::ANSWER_TYPE_SINGLE_LINE, self::ANSWER_TYPE_MULTI_LINE];
    }

    public function isValidAnswerType($answerType): bool
    {
        $validTypes = $this->getValidAnswerTypes();
        return in_array($answerType, $validTypes);
    }

    public function isSingleLineAnswerType($answerType): bool
    {
        return $answerType == assKprimChoice::ANSWER_TYPE_SINGLE_LINE;
    }

    /**
     * @param ilLanguage $lng
     * @return array
     */
    public function getAnswerTypeSelectOptions(ilLanguage $lng): array
    {
        return [
            self::ANSWER_TYPE_SINGLE_LINE => $lng->txt('answers_singleline'),
            self::ANSWER_TYPE_MULTI_LINE => $lng->txt('answers_multiline')
        ];
    }

    public function getValidOptionLabels(): array
    {
        return [
            self::OPTION_LABEL_RIGHT_WRONG,
            self::OPTION_LABEL_PLUS_MINUS,
            self::OPTION_LABEL_APPLICABLE_OR_NOT,
            self::OPTION_LABEL_ADEQUATE_OR_NOT,
            self::OPTION_LABEL_CUSTOM
        ];
    }

    public function getValidOptionLabelsTranslated(ilLanguage $lng): array
    {
        return [
            self::OPTION_LABEL_RIGHT_WRONG => $lng->txt('option_label_right_wrong'),
            self::OPTION_LABEL_PLUS_MINUS => $lng->txt('option_label_plus_minus'),
            self::OPTION_LABEL_APPLICABLE_OR_NOT => $lng->txt('option_label_applicable_or_not'),
            self::OPTION_LABEL_ADEQUATE_OR_NOT => $lng->txt('option_label_adequate_or_not'),
            self::OPTION_LABEL_CUSTOM => $lng->txt('option_label_custom')
        ];
    }

    public function isValidOptionLabel($optionLabel): bool
    {
        $validLabels = $this->getValidOptionLabels();
        return in_array($optionLabel, $validLabels);
    }

    public function getTrueOptionLabelTranslation(ilLanguage $lng, $optionLabel)
    {
        switch ($optionLabel) {
            case self::OPTION_LABEL_RIGHT_WRONG:
                return $lng->txt('option_label_right');

            case self::OPTION_LABEL_PLUS_MINUS:
                return $lng->txt('option_label_plus');

            case self::OPTION_LABEL_APPLICABLE_OR_NOT:
                return $lng->txt('option_label_applicable');

            case self::OPTION_LABEL_ADEQUATE_OR_NOT:
                return $lng->txt('option_label_adequate');

            case self::OPTION_LABEL_CUSTOM:
            default:
                return $this->getCustomTrueOptionLabel();
        }
    }

    public function getFalseOptionLabelTranslation(ilLanguage $lng, $optionLabel)
    {
        switch ($optionLabel) {
            case self::OPTION_LABEL_RIGHT_WRONG:
                return $lng->txt('option_label_wrong');

            case self::OPTION_LABEL_PLUS_MINUS:
                return $lng->txt('option_label_minus');

            case self::OPTION_LABEL_APPLICABLE_OR_NOT:
                return $lng->txt('option_label_not_applicable');

            case self::OPTION_LABEL_ADEQUATE_OR_NOT:
                return $lng->txt('option_label_not_adequate');

            case self::OPTION_LABEL_CUSTOM:
            default:
                return $this->getCustomFalseOptionLabel();
        }
    }

    public function getInstructionTextTranslation(ilLanguage $lng, $optionLabel): string
    {
        return sprintf(
            $lng->txt('kprim_instruction_text'),
            $this->getTrueOptionLabelTranslation($lng, $optionLabel),
            $this->getFalseOptionLabelTranslation($lng, $optionLabel)
        );
    }

    public function isCustomOptionLabel($labelValue): bool
    {
        return $labelValue == self::OPTION_LABEL_CUSTOM;
    }

    public function handleFileUploads($answers, $files): void
    {
        foreach ($answers as $answer) {
            /* @var ilAssKprimChoiceAnswer $answer */

            if (!isset($files[$answer->getPosition()])) {
                continue;
            }

            $this->handleFileUpload($answer, $files[$answer->getPosition()]);
        }
    }

    private function handleFileUpload(ilAssKprimChoiceAnswer $answer, $fileData): int
    {
        $imagePath = $this->getImagePath();

        if (!file_exists($imagePath)) {
            ilFileUtils::makeDirParents($imagePath);
        }

        $filename = $this->buildHashedImageFilename($fileData['name'], true);

        $answer->setImageFsDir($imagePath);
        $answer->setImageFile($filename);

        if (!ilFileUtils::moveUploadedFile($fileData['tmp_name'], $filename, $answer->getImageFsPath())) {
            return 2;
        }

        $this->generateThumbForFile($filename, $this->getImagePath(), $this->getThumbSize());

        return 0;
    }

    public function removeAnswerImage($position): void
    {
        $answer = $this->getAnswer($position);

        if (file_exists($answer->getImageFsPath())) {
            ilFileUtils::delDir($answer->getImageFsPath());
        }

        if (file_exists($answer->getThumbFsPath())) {
            ilFileUtils::delDir($answer->getThumbFsPath());
        }

        $answer->setImageFile(null);
    }

    protected function getSolutionSubmit(): array
    {
        $solutionSubmit = [];
        $post = $this->dic->http()->wrapper()->post();

        foreach ($this->getAnswers() as $index => $a) {
            if ($post->has("kprim_choice_result_$index")) {
                $value = $post->retrieve(
                    "kprim_choice_result_$index",
                    $this->dic->refinery()->kindlyTo()->string()
                );
                if (is_numeric($value)) {
                    $solutionSubmit[] = $value;
                }
            } else {
                $solutionSubmit[] = null;
            }
        }
        return $solutionSubmit;
    }

    protected function calculateReachedPointsForSolution($found_values, $active_id = 0): float
    {
        $numCorrect = 0;
        if ($found_values == null) {
            $found_values = [];
        }
        foreach ($this->getAnswers() as $key => $answer) {
            if (!isset($found_values[$answer->getPosition()])) {
                continue;
            }

            if ($found_values[$answer->getPosition()] == $answer->getCorrectness()) {
                $numCorrect++;
            }
        }

        if ($numCorrect >= self::NUM_REQUIRED_ANSWERS) {
            $points = $this->getPoints();
        } elseif ($this->isScorePartialSolutionEnabled() && $numCorrect >= self::PARTIAL_SCORING_NUM_CORRECT_ANSWERS) {
            $points = $this->getPoints() / 2;
        } else {
            $points = 0;
        }

        if ($active_id) {
            if (count($found_values) == 0) {
                $points = 0;
            }
        }
        return (float) $points;
    }

    public function duplicate(bool $for_test = true, string $title = "", string $author = "", int $owner = -1, $testObjId = null): int
    {
        if ($this->id <= 0) {
            // The question has not been saved. It cannot be duplicated
            return -1;
        }
        // duplicate the question in database
        $this_id = $this->getId();
        $thisObjId = $this->getObjId();

        $clone = $this;

        $original_id = $this->questioninfo->getOriginalId($this->id);
        $clone->id = -1;

        if ((int) $testObjId > 0) {
            $clone->setObjId($testObjId);
        }

        if ($title) {
            $clone->setTitle($title);
        }

        if ($author) {
            $clone->setAuthor($author);
        }
        if ($owner) {
            $clone->setOwner($owner);
        }

        if ($for_test) {
            $clone->saveToDb($original_id);
        } else {
            $clone->saveToDb();
        }

        // copy question page content
        $clone->copyPageOfQuestion($this_id);
        // copy XHTML media objects
        $clone->copyXHTMLMediaObjectsOfQuestion($this_id);
        // duplicate the images
        $clone->cloneAnswerImages($this_id, $thisObjId, $clone->getId(), $clone->getObjId());

        $clone->onDuplicate($thisObjId, $this_id, $clone->getObjId(), $clone->getId());

        return $clone->id;
    }

    public function createNewOriginalFromThisDuplicate($target_parent_id, $target_question_title = ""): int
    {
        if ($this->getId() <= 0) {
            throw new RuntimeException('The question has not been saved. It cannot be duplicated');
        }

        $source_question_id = $this->id;
        $source_parent_id = $this->getObjId();

        // duplicate the question in database
        $clone = $this;
        $clone->id = -1;

        $clone->setObjId($target_parent_id);

        if ($target_question_title) {
            $clone->setTitle($target_question_title);
        }

        $clone->saveToDb();
        // copy question page content
        $clone->copyPageOfQuestion($source_question_id);
        // copy XHTML media objects
        $clone->copyXHTMLMediaObjectsOfQuestion($source_question_id);
        // duplicate the image
        $clone->cloneAnswerImages($source_question_id, $source_parent_id, $clone->getId(), $clone->getObjId());

        $clone->onCopy($source_parent_id, $source_question_id, $target_parent_id, $clone->getId());

        return $clone->id;
    }

    /**
     * Copies an assMultipleChoice object
     */
    public function copyObject($target_questionpool_id, $title = ""): int
    {
        if ($this->getId() <= 0) {
            throw new RuntimeException('The question has not been saved. It cannot be duplicated');
        }
        // duplicate the question in database
        $clone = $this;

        $original_id = $this->questioninfo->getOriginalId($this->id);
        $clone->id = -1;
        $source_questionpool_id = $this->getObjId();
        $clone->setObjId($target_questionpool_id);
        if ($title) {
            $clone->setTitle($title);
        }
        $clone->saveToDb();
        // copy question page content
        $clone->copyPageOfQuestion($original_id);
        // copy XHTML media objects
        $clone->copyXHTMLMediaObjectsOfQuestion($original_id);
        // duplicate the image
        $clone->cloneAnswerImages($original_id, $source_questionpool_id, $clone->getId(), $clone->getObjId());

        $clone->onCopy($source_questionpool_id, $original_id, $clone->getObjId(), $clone->getId());

        return $clone->id;
    }

    protected function beforeSyncWithOriginal($origQuestionId, $dupQuestionId, $origParentObjId, $dupParentObjId): void
    {
        parent::beforeSyncWithOriginal($origQuestionId, $dupQuestionId, $origParentObjId, $dupParentObjId);

        $question = self::instantiateQuestion($origQuestionId);

        foreach ($question->getAnswers() as $answer) {
            $question->removeAnswerImage($answer->getPosition());
        }
    }

    protected function afterSyncWithOriginal($origQuestionId, $dupQuestionId, $origParentObjId, $dupParentObjId): void
    {
        parent::afterSyncWithOriginal($origQuestionId, $dupQuestionId, $origParentObjId, $dupParentObjId);

        $this->cloneAnswerImages($dupQuestionId, $dupParentObjId, $origQuestionId, $origParentObjId);
    }

    protected function cloneAnswerImages(
        $source_question_id,
        $source_parent_id,
        $target_question_id,
        $target_parent_id
    ): void {
        /** @var $ilLog ilLogger */
        global $DIC;
        $ilLog = $DIC['ilLog'];

        $source_path = $this->questionFilesService->buildImagePath($source_question_id, $source_parent_id);
        $target_path = $this->questionFilesService->buildImagePath($target_question_id, $target_parent_id);

        foreach ($this->getAnswers() as $answer) {
            $filename = $answer->getImageFile();

            if ($filename === null || $filename === '') {
                continue;
            }

            if (!file_exists($target_path)) {
                ilFileUtils::makeDirParents($target_path);
            }

            if (file_exists($source_path . $filename)) {
                if (!copy($source_path . $filename, $target_path . $filename)) {
                    $ilLog->warning(sprintf(
                        "Could not clone source image '%s' to '%s' (srcQuestionId: %s|tgtQuestionId: %s|srcParentObjId: %s|tgtParentObjId: %s)",
                        $source_path . $filename,
                        $target_path . $filename,
                        $source_question_id,
                        $target_question_id,
                        $source_parent_id,
                        $target_parent_id
                    ));
                }
            }

            if (file_exists($source_path . $this->getThumbPrefix() . $filename)) {
                if (!copy($source_path . $this->getThumbPrefix() . $filename, $target_path . $this->getThumbPrefix() . $filename)) {
                    $ilLog->warning(sprintf(
                        "Could not clone thumbnail source image '%s' to '%s' (srcQuestionId: %s|tgtQuestionId: %s|srcParentObjId: %s|tgtParentObjId: %s)",
                        $source_path . $this->getThumbPrefix() . $filename,
                        $target_path . $this->getThumbPrefix() . $filename,
                        $source_question_id,
                        $target_question_id,
                        $source_parent_id,
                        $target_parent_id
                    ));
                }
            }
        }
    }

    protected function getRTETextWithMediaObjects(): string
    {
        $combinedText = parent::getRTETextWithMediaObjects();

        foreach ($this->getAnswers() as $answer) {
            $combinedText .= $answer->getAnswertext();
        }

        return $combinedText;
    }

    /**
     * @param ilAssSelfAssessmentMigrator $migrator
     */
    protected function lmMigrateQuestionTypeSpecificContent(ilAssSelfAssessmentMigrator $migrator): void
    {
        foreach ($this->getAnswers() as $answer) {
            /* @var ilAssKprimChoiceAnswer $answer */
            $answer->setAnswertext($migrator->migrateToLmContent($answer->getAnswertext()));
        }
    }

    /**
     * Returns a JSON representation of the question
     */
    public function toJSON(): string
    {
        $this->lng->loadLanguageModule('assessment');

        $result = [];
        $result['id'] = $this->getId();
        $result['type'] = $this->getQuestionType();
        $result['title'] = $this->getTitleForHTMLOutput();
        $result['question'] = $this->formatSAQuestion($this->getQuestion());
        $result['instruction'] = $this->getInstructionTextTranslation(
            $this->lng,
            $this->getOptionLabel()
        );
        $result['nr_of_tries'] = $this->getNrOfTries();
        $result['shuffle'] = $this->isShuffleAnswersEnabled();
        $result['feedback'] = [
            'onenotcorrect' => $this->formatSAQuestion($this->feedbackOBJ->getGenericFeedbackTestPresentation($this->getId(), false)),
            'allcorrect' => $this->formatSAQuestion($this->feedbackOBJ->getGenericFeedbackTestPresentation($this->getId(), true))
        ];

        $result['trueOptionLabel'] = $this->getTrueOptionLabelTranslation($this->lng, $this->getOptionLabel());
        $result['falseOptionLabel'] = $this->getFalseOptionLabelTranslation($this->lng, $this->getOptionLabel());

        $result['num_allowed_failures'] = $this->getNumAllowedFailures();

        $answers = [];
        $has_image = false;

        foreach ($this->getAnswers() as $key => $answer) {
            if (strlen((string) $answer->getImageFile())) {
                $has_image = true;
            }

            $answers[] = [
                'answertext' => $this->formatSAQuestion($answer->getAnswertext() ?? ''),
                'correctness' => (bool) $answer->getCorrectness(),
                'order' => (int) $answer->getPosition(),
                'image' => (string) $answer->getImageFile(),
                'feedback' => $this->formatSAQuestion(
                    $this->feedbackOBJ->getSpecificAnswerFeedbackExportPresentation($this->getId(), 0, $key)
                )
            ];
        }

        $result['answers'] = $answers;

        if ($has_image) {
            $result['path'] = $this->getImagePathWeb();
            $result['thumb'] = $this->getThumbSize();
        }

        $mobs = ilObjMediaObject::_getMobsOfObject("qpl:html", $this->getId());
        $result['mobs'] = $mobs;

        return json_encode($result);
    }

    private function getNumAllowedFailures(): int
    {
        if ($this->isScorePartialSolutionEnabled()) {
            return self::NUM_REQUIRED_ANSWERS - self::PARTIAL_SCORING_NUM_CORRECT_ANSWERS;
        }

        return 0;
    }

    public function getSpecificFeedbackAllCorrectOptionLabel(): string
    {
        return 'feedback_correct_kprim';
    }

    public static function isObligationPossible(int $questionId): bool
    {
        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function setExportDetailsXLSX(ilAssExcelFormatHelper $worksheet, int $startrow, int $col, int $active_id, int $pass): int
    {
        parent::setExportDetailsXLSX($worksheet, $startrow, $col, $active_id, $pass);

        $solution = $this->getSolutionValues($active_id, $pass);

        $i = 1;
        foreach ($this->getAnswers() as $id => $answer) {
            $worksheet->setCell($startrow + $i, $col, $answer->getAnswertext());
            $worksheet->setBold($worksheet->getColumnCoord($col) . ($startrow + $i));
            $correctness = false;
            foreach ($solution as $solutionvalue) {
                if ($id == $solutionvalue['value1']) {
                    $correctness = $solutionvalue['value2'];
                    break;
                }
            }
            $worksheet->setCell($startrow + $i, $col + 2, $correctness);
            $i++;
        }

        return $startrow + $i + 1;
    }

    public function moveAnswerDown($position): bool
    {
        if ($position < 0 || $position >= (self::NUM_REQUIRED_ANSWERS - 1)) {
            return false;
        }

        for ($i = 0, $max = count($this->answers); $i < $max; $i++) {
            if ($i == $position) {
                $movingAnswer = $this->answers[$i];
                $targetAnswer = $this->answers[ $i + 1 ];

                $movingAnswer->setPosition($position + 1);
                $targetAnswer->setPosition($position);

                $this->answers[ $i + 1 ] = $movingAnswer;
                $this->answers[$i] = $targetAnswer;
            }
        }
        return true;
    }

    public function moveAnswerUp($position): bool
    {
        if ($position <= 0 || $position > (self::NUM_REQUIRED_ANSWERS - 1)) {
            return false;
        }

        for ($i = 0, $max = count($this->answers); $i < $max; $i++) {
            if ($i == $position) {
                $movingAnswer = $this->answers[$i];
                $targetAnswer = $this->answers[ $i - 1 ];

                $movingAnswer->setPosition($position - 1);
                $targetAnswer->setPosition($position);

                $this->answers[ $i - 1 ] = $movingAnswer;
                $this->answers[$i] = $targetAnswer;
            }
        }

        return true;
    }
}
