<?php

declare(strict_types=1);

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/


/**
 * @author            Jesús López Reyes <lopez@leifos.com>
 * @ilCtrl_IsCalledBy ilAppointmentPresentationCourseGUI: ilCalendarAppointmentPresentationGUI
 * @ingroup           ServicesCalendar
 */
class ilAppointmentPresentationCourseGUI extends ilAppointmentPresentationGUI implements ilCalendarAppointmentPresentation
{
    public function collectPropertiesAndActions(): void
    {
        $settings = ilCalendarSettings::_getInstance();
        $this->lng->loadLanguageModule("crs");

        $app = $this->appointment;

        $cat_info = $this->getCatInfo();

        $crs = new ilObjCourse($cat_info['obj_id'], false);
        $files = ilCourseFile::_readFilesByCourse($cat_info['obj_id']);

        // get course ref id (this is possible, since courses only have one ref id)
        $refs = ilObject::_getAllReferences($cat_info['obj_id']);
        $crs_ref_id = current($refs);

        // add common section (title, description, object/calendar, location)
        $this->addCommonSection($app, $cat_info['obj_id']);

        //count number of files
        if (count($files)) {
            $this->has_files = true;
        }
        // add specific section only if the event is autogenerated.
        if ($app['event']->isAutoGenerated()) {
            $this->addInfoSection($this->lng->txt("cal_crs_info"));

            if ($crs->getImportantInformation()) {
                $this->addInfoProperty(
                    $this->lng->txt("crs_important_info"),
                    ilUtil::makeClickable(nl2br($crs->getImportantInformation()))
                );
            }

            if ($crs->getSyllabus()) {
                $this->addInfoProperty(
                    $this->lng->txt("crs_syllabus"),
                    ilUtil::makeClickable(nl2br($crs->getSyllabus()))
                );
            }

            if ($this->has_files) {
                $links = array();
                foreach ($files as $file) {
                    $this->ctrl->setParameter($this, 'file_id', $file->getFileId());
                    $this->ctrl->setParameterByClass('ilobjcoursegui', 'file_id', $file->getFileId());
                    $this->ctrl->setParameterByClass('ilobjcoursegui', 'ref_id', $crs_ref_id);

                    $file_name = $file->getFileName();
                    $links[$file_name] = $this->ui->renderer()->render(($this->ui->factory()->button()->shy(
                        $file_name,
                        $this->ctrl->getLinkTargetByClass(array("ilRepositoryGUI", "ilobjcoursegui"), 'sendfile')
                    )));

                    $this->ctrl->setParameterByClass('ilobjcoursegui', 'ref_id', '');
                }
                ksort($links, SORT_NATURAL | SORT_FLAG_CASE);

                $this->addInfoProperty($this->lng->txt("files"), implode("<br>", $links));
                $this->addListItemProperty($this->lng->txt("files"), implode(", ", $links));
            }

            // tutorial support members
            $parts = ilParticipants::getInstanceByObjId($cat_info['obj_id']);
            //contacts is an array of user ids.
            $contacts = $parts->getContacts();
            $sorted_ids = ilUtil::_sortIds($contacts, 'usr_data', 'lastname', 'usr_id');

            $names = [];
            foreach ($sorted_ids as $usr_id) {
                $name_presentation = $this->getUserName((int) $usr_id, true);
                if (strlen($name_presentation)) {
                    $names[] = $name_presentation;
                }
            }
            if (count($names)) {
                $this->addInfoProperty($this->lng->txt('crs_mem_contacts'), implode('<br/>', $names));
                $this->addListItemProperty($this->lng->txt('crs_mem_contacts'), implode('<br />', $names));
            }

            //course contact
            $contact_info = [];
            if ($crs->getContactName()) {
                $contact_info[] = [
                    'name' => $this->lng->txt('crs_contact_name'),
                    'value' => $crs->getContactName()
                ];
            }

            if ($crs->getContactEmail()) {
                //TODO: optimize this
                //$courseGUI = new ilObjCourseGUI("", $crs_ref_id);
                $emails = explode(",", $crs->getContactEmail());
                foreach ($emails as $email) {
                    $email = trim($email);
                    $etpl = new ilTemplate("tpl.crs_contact_email.html", true, true, 'Modules/Course');
                    $etpl->setVariable(
                        "EMAIL_LINK",
                        ilMailFormCall::getLinkTarget(
                            $this->getInfoScreen(),
                            'showSummary',
                            array(),
                            array(
                                'type' => 'new',
                                'rcp_to' => $email,
                                //'sig' => $courseGUI->createMailSignature()
                            ),
                            array(
                                ilMailFormCall::CONTEXT_KEY => ilCourseMailTemplateMemberContext::ID,
                                'ref_id' => $crs->getRefId(),
                                'ts' => time()
                            )
                        )
                    );
                    $etpl->setVariable("CONTACT_EMAIL", $email);
                    $contact_info[] = [
                        'name' => $this->lng->txt('crs_contact_email'),
                        'value' => $etpl->get()
                    ];
                }
            }

            if ($crs->getContactPhone()) {
                $contact_info[] = [
                    'name' => $this->lng->txt('crs_contact_phone'),
                    'value' => $crs->getContactPhone()
                ];
            }
            if ($crs->getContactResponsibility()) {
                $contact_info[] = [
                    'name' => $this->lng->txt('crs_contact_responsibility'),
                    'value' => $crs->getContactResponsibility()
                ];
            }
            if ($crs->getContactConsultation()) {
                $contact_info[] = [
                    'name' => $this->lng->txt('crs_contact_consultation'),
                    'value' => nl2br($crs->getContactConsultation())
                ];
            }
            if ($contact_info !== []) {
                $this->addInfoSection($this->lng->txt('crs_contact'));
                foreach ($contact_info as $info) {
                    $this->addInfoProperty(
                        $info['name'],
                        $info['value']
                    );
                }
            }
            $this->addMetaData('crs', $cat_info['obj_id']);

            // last edited
            $this->addLastUpdate($app);
        }

        $this->addAction($this->lng->txt("crs_open"), ilLink::_getStaticLink($crs_ref_id, "crs"));

        // register/unregister to appointment
        if ($settings->isCGRegistrationEnabled()) {
            if (!$app['event']->isAutoGenerated()) {
                $reg = new ilCalendarRegistration($app['event']->getEntryId());

                if ($reg->isRegistered(
                    $this->user->getId(),
                    new ilDateTime($app['dstart'], IL_CAL_UNIX),
                    new ilDateTime($app['dend'], IL_CAL_UNIX)
                )) {
                    //$this->ctrl->setParameterByClass('ilcalendarappointmentgui','seed',$this->getSeed()->get(IL_CAL_DATE));
                    $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'app_id', $app['event']->getEntryId());
                    $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'dstart', $app['dstart']);
                    $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'dend', $app['dend']);

                    $this->addAction(
                        $this->lng->txt('cal_reg_unregister'),
                        $this->ctrl->getLinkTargetByClass('ilcalendarappointmentgui', 'confirmUnregister')
                    );
                } else {
                    //$this->ctrl->setParameterByClass('ilcalendarappointmentgui','seed',$this->getSeed()->get(IL_CAL_DATE));
                    $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'app_id', $app['event']->getEntryId());
                    $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'dstart', $app['dstart']);
                    $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'dend', $app['dend']);

                    $this->addAction(
                        $this->lng->txt('cal_reg_register'),
                        $this->ctrl->getLinkTargetByClass('ilcalendarappointmentgui', 'confirmRegister')
                    );
                }

                $registered = $reg->getRegisteredUsers(
                    new \ilDateTime($app['dstart'], IL_CAL_UNIX),
                    new \ilDateTime($app['dend'], IL_CAL_UNIX)
                );

                $users = "";
                foreach ($registered as $user) {
                    $users .= $this->getUserName($user) . '<br />';
                }
                if ($users != "") {
                    $this->addInfoProperty($this->lng->txt("cal_reg_registered_users"), $users);
                }
            }
        }
    }
}
