<?php

declare(strict_types=1);

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

/**
 * @author       Stefan Meyer <smeyer.ilias@gmx.de>
 * @ilCtrl_Calls ilCalendarWeekGUI: ilCalendarAppointmentGUI
 * @ilCtrl_Calls ilCalendarWeekGUI: ilCalendarAppointmentPresentationGUI
 * @ingroup      ServicesCalendar
 */
class ilCalendarWeekGUI extends ilCalendarViewGUI
{
    protected int $num_appointments = 1;
    protected ?ilCalendarUserSettings $user_settings;
    protected array $weekdays = array();

    protected ilCalendarAppointmentColors $app_colors;
    protected array $seed_info = [];
    protected string $timezone = 'UTC';
    protected ?ilCalendarSettings $cal_settings = null;
    protected array $colspans = [];

    // config
    protected int $raster = 15;
    //setup_calendar
    protected int $user_id = 0;
    protected bool $disable_empty;
    protected bool $no_add;

    public function __construct(ilDate $seed_date)
    {
        parent::__construct($seed_date, ilCalendarViewGUI::CAL_PRESENTATION_WEEK);
    }

    public function initialize(int $a_calendar_presentation_type): void
    {
        parent::initialize($a_calendar_presentation_type); // TODO: Change the autogenerated stub
        $this->seed_info = (array) $this->seed->get(IL_CAL_FKT_GETDATE, '', 'UTC');
        $this->user_settings = ilCalendarUserSettings::_getInstanceByUserId($this->user->getId());
        $this->app_colors = new ilCalendarAppointmentColors($this->user->getId());
        if ($this->user->getTimeZone()) {
            $this->timezone = $this->user->getTimeZone();
        }
    }

    public function executeCommand(): void
    {
        $this->ctrl->saveParameter($this, 'seed');
        $next_class = $this->ctrl->getNextClass();
        switch ($next_class) {
            case "ilcalendarappointmentpresentationgui":
                $this->ctrl->setReturn($this, "");
                $gui = ilCalendarAppointmentPresentationGUI::_getInstance($this->seed, (array) $this->getCurrentApp());
                $this->ctrl->forwardCommand($gui);
                break;
            case 'ilcalendarappointmentgui':
                $this->ctrl->setReturn($this, '');
                $this->tabs_gui->setSubTabActive((string) ilSession::get('cal_last_tab'));

                // initial date for new calendar appointments
                $idate = new ilDate($this->initInitialDateFromQuery(), IL_CAL_DATE);
                $app = new ilCalendarAppointmentGUI($this->seed, $idate, $this->initAppointmentIdFromQuery());
                $this->ctrl->forwardCommand($app);
                break;

            default:
                $time = microtime(true);
                $cmd = $this->ctrl->getCmd("show");
                $this->$cmd();
                $this->main_tpl->setContent($this->tpl->get());
                break;
        }
    }

    public function show(): void
    {
        $num_apps = [];
        $morning_aggr = $this->getMorningAggr();
        $evening_aggr = $this->user_settings->getDayEnd() * 60;

        $this->tpl = new ilTemplate('tpl.week_view.html', true, true, 'Services/Calendar');

        ilYuiUtil::initDragDrop();

        $navigation = new ilCalendarHeaderNavigationGUI($this, $this->seed, ilDateTime::WEEK);
        $this->tpl->setVariable('NAVIGATION', $navigation->getHTML());
        $this->setUpCalendar();

        $scheduler = new ilCalendarSchedule(
            $this->seed,
            ilCalendarSchedule::TYPE_WEEK,
            $this->user->getId(),
            $this->disable_empty
        );
        $scheduler->addSubitemCalendars(true);
        $scheduler->calculate();

        $counter = 0;
        $hours = null;
        $all_fullday = array();
        foreach (ilCalendarUtil::_buildWeekDayList($this->seed, $this->user_settings->getWeekStart())->get() as $date) {
            $daily_apps = $scheduler->getByDay($date, $this->timezone);
            if (!$this->view_with_appointments && count($daily_apps)) {
                $this->view_with_appointments = true;
            }
            $hours = $this->parseHourInfo(
                $daily_apps,
                $date,
                $counter,
                $hours,
                $morning_aggr,
                $evening_aggr
            );
            $this->weekdays[] = $date;

            $num_apps[$date->get(IL_CAL_DATE)] = count($daily_apps);

            $all_fullday[] = $daily_apps;
            $counter++;
        }

        $this->calculateColspans($hours);

        $this->cal_settings = ilCalendarSettings::_getInstance();

        // Table header
        $counter = 0;
        foreach (ilCalendarUtil::_buildWeekDayList($this->seed, $this->user_settings->getWeekStart())->get() as $date) {
            $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'seed', $date->get(IL_CAL_DATE));
            $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'idate', $date->get(IL_CAL_DATE));
            $this->ctrl->setParameterByClass('ilcalendardaygui', 'seed', $date->get(IL_CAL_DATE));

            if (!$this->no_add) {
                $this->addAppointmentLink($date);
            }

            $this->addHeaderDate($date, $num_apps);

            $this->tpl->setCurrentBlock('day_header_row');
            $this->tpl->setVariable('DAY_COLSPAN', max($this->colspans[$counter], 1));
            $this->tpl->parseCurrentBlock();

            $counter++;
        }

        // show fullday events
        $this->addFullDayEvents($all_fullday);

        //show timed events
        $this->addTimedEvents($hours, $morning_aggr, $evening_aggr);

        $this->tpl->setVariable("TXT_TIME", $this->lng->txt("time"));
    }

    protected function showFulldayAppointment(array $a_app): void
    {
        $event_tpl = new ilTemplate('tpl.day_event_view.html', true, true, 'Services/Calendar');
        $event_tpl->setCurrentBlock('fullday_app');

        $title = $this->getAppointmentShyButton($a_app['event'], (string) $a_app['dstart'], "");

        $event_tpl->setVariable('EVENT_CONTENT', $title);

        $color = $this->app_colors->getColorByAppointment($a_app['event']->getEntryId());
        $font_color = ilCalendarUtil::calculateFontColor($color);

        $event_tpl->setVariable('F_APP_BGCOLOR', $color);
        $event_tpl->setVariable('F_APP_FONTCOLOR', $font_color);

        $this->ctrl->clearParametersByClass('ilcalendarappointmentgui');
        $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'app_id', $a_app['event']->getEntryId());
        $event_tpl->setVariable(
            'F_APP_EDIT_LINK',
            $this->ctrl->getLinkTargetByClass('ilcalendarappointmentgui', 'edit')
        );

        if ($event_html_by_plugin = $this->getContentByPlugins($a_app['event'], $a_app['dstart'], $title, $event_tpl)) {
            $event_html = $event_html_by_plugin;
        } else {
            $event_tpl->parseCurrentBlock();
            $event_html = $event_tpl->get();
        }

        $this->tpl->setCurrentBlock("content_fd");
        $this->tpl->setVariable("CONTENT_EVENT_FD", $event_html);
        $this->tpl->parseCurrentBlock();

        $this->num_appointments++;
    }

    protected function showAppointment(array $a_app): void
    {
        $time = '';
        $event_tpl = new ilTemplate('tpl.week_event_view.html', true, true, 'Services/Calendar');

        $ilUser = $this->user;

        $this->tpl->setCurrentBlock('not_empty');

        $this->ctrl->clearParametersByClass('ilcalendarappointmentgui');
        $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'app_id', $a_app['event']->getEntryId());

        $color = $this->app_colors->getColorByAppointment($a_app['event']->getEntryId());
        $style = 'background-color: ' . $color . ';';
        $style .= ('color:' . ilCalendarUtil::calculateFontColor($color));
        $td_style = $style;

        if (!$a_app['event']->isFullDay()) {
            $time = $this->getAppointmentTimeString($a_app['event']);

            $td_style .= $a_app['event']->getPresentationStyle();
        }

        $shy = $this->getAppointmentShyButton($a_app['event'], (string) $a_app['dstart'], "");

        $title = ($time != "") ? $time . " " . $shy : $shy;

        $event_tpl->setCurrentBlock('event_cell_content');
        $event_tpl->setVariable("STYLE", $style);
        $event_tpl->setVariable('EVENT_CONTENT', $title);

        if ($event_html_by_plugin = $this->getContentByPlugins($a_app['event'], $a_app['dstart'], $title, $event_tpl)) {
            $event_html = $event_html_by_plugin;
        } else {
            $event_tpl->parseCurrentBlock();
            $event_html = $event_tpl->get();
        }

        $this->tpl->setVariable('GRID_CONTENT', $event_html);

        // provide table cell attributes
        $this->tpl->parseCurrentBlock();

        $this->tpl->setCurrentBlock('day_cell');

        $this->tpl->setVariable('DAY_ID', 'a' . $this->num_appointments);
        $this->tpl->setVariable('TD_ROWSPAN', $a_app['rowspan']);
        $this->tpl->setVariable('TD_STYLE', $a_app['event']->getPresentationStyle());
        $this->tpl->setVariable('TD_CLASS', 'calevent il_calevent');

        $this->tpl->parseCurrentBlock();

        $this->num_appointments++;
    }

    /**
     * calculate overlapping hours
     * @access protected
     * @return array hours
     */
    protected function parseHourInfo(
        array $daily_apps,
        ilDateTime $date,
        int $num_day,
        array $hours = null,
        int $morning_aggr = 0,
        int $evening_aggr = 0
    ): array {
        for ($i = $morning_aggr; $i <= $evening_aggr; $i += $this->raster) {
            $hours[$i][$num_day]['apps_start'] = array();
            $hours[$i][$num_day]['apps_num'] = 0;
            switch ($this->user_settings->getTimeFormat()) {
                case ilCalendarSettings::TIME_FORMAT_24:
                    if ($morning_aggr > 0 && $i == $morning_aggr) {
                        $hours[$i][$num_day]['txt'] = sprintf('%02d:00', 0) . ' - ' .
                            sprintf('%02d:00', ceil(($i + 1) / 60));
                    } else {
                        if (!isset($hours[$i][$num_day]['txt'])) {
                            $hours[$i][$num_day]['txt'] = sprintf('%02d:%02d', floor($i / 60), $i % 60);
                        } else {
                            $hours[$i][$num_day]['txt'] .= sprintf('%02d:%02d', floor($i / 60), $i % 60);
                        }
                    }
                    if ($evening_aggr < 23 * 60 && $i == $evening_aggr) {
                        if (!isset($hours[$i][$num_day]['txt'])) {
                            $hours[$i][$num_day]['txt'] = ' - ' . sprintf('%02d:00', 0);
                        } else {
                            $hours[$i][$num_day]['txt'] .= ' - ' . sprintf('%02d:00', 0);
                        }
                    }
                    break;

                case ilCalendarSettings::TIME_FORMAT_12:
                    if ($morning_aggr > 0 && $i == $morning_aggr) {
                        $hours[$i][$num_day]['txt'] =
                            date('h a', mktime(0, 0, 0, 1, 1, 2000)) . ' - ' .
                            date('h a', mktime($this->user_settings->getDayStart(), 0, 0, 1, 1, 2000));
                    } else {
                        if (!isset($hours[$i][$num_day]['txt'])) {
                            $hours[$i][$num_day]['txt'] = date('h a', mktime((int) floor($i / 60), $i % 60, 0, 1, 1, 2000));
                        } else {
                            $hours[$i][$num_day]['txt'] .= date('h a', mktime((int) floor($i / 60), $i % 60, 0, 1, 1, 2000));
                        }
                    }
                    if ($evening_aggr < 23 * 60 && $i == $evening_aggr) {
                        $hours[$i][$num_day]['txt'] =
                            date('h a', mktime($this->user_settings->getDayEnd(), 0, 0, 1, 1, 2000)) . ' - ' .
                            date('h a', mktime(0, 0, 0, 1, 1, 2000));
                    }
                    break;
            }
        }

        $date_info = $date->get(IL_CAL_FKT_GETDATE, '', 'UTC');

        foreach ($daily_apps as $app) {
            // fullday appointment are not relavant
            if ($app['fullday']) {
                continue;
            }
            // start hour for this day
            #21636
            if ($app['start_info']['mday'] != $date_info['mday']) {
                $start = 0;
            } else {
                $start = $app['start_info']['hours'] * 60 + $app['start_info']['minutes'];
            }
            #21132 #21636
            //$start = $app['start_info']['hours']*60+$app['start_info']['minutes'];

            // end hour for this day
            if ($app['end_info']['mday'] != $date_info['mday']) {
                $end = 23 * 60;
            } elseif ($app['start_info']['hours'] == $app['end_info']['hours']) {
                $end = $start + $this->raster;
            } else {
                $end = $app['end_info']['hours'] * 60 + $app['end_info']['minutes'];
            }
            #21132 #21636
            //$end = $app['end_info']['hours']*60+$app['end_info']['minutes'];

            if ($start < $morning_aggr) {
                $start = $morning_aggr;
            }
            if ($end <= $morning_aggr) {
                $end = $morning_aggr + $this->raster;
            }
            if ($start > $evening_aggr) {
                $start = $evening_aggr;
            }
            if ($end > $evening_aggr + $this->raster) {
                $end = $evening_aggr + $this->raster;
            }
            if ($end <= $start) {
                $end = $start + $this->raster;
            }

            // map start and end to raster
            $start = floor($start / $this->raster) * $this->raster;
            $end = ceil($end / $this->raster) * $this->raster;

            $first = true;
            for ($i = $start; $i < $end; $i += $this->raster) {
                if ($first) {
                    $app['rowspan'] = ceil(($end - $start) / $this->raster);
                    $hours[$i][$num_day]['apps_start'][] = $app;
                    $first = false;
                }
                $hours[$i][$num_day]['apps_num']++;
            }
        }
        return $hours;
    }

    protected function calculateColspans(array $hours): void
    {
        foreach ($hours as $hour_num => $hours_per_day) {
            foreach ($hours_per_day as $num_day => $hour) {
                $this->colspans[$num_day] = max($this->colspans[$num_day] ?? 0, $hour['apps_num'] ?? 0);
            }
        }
    }

    protected function getMorningAggr(): int
    {
        if ($this->user_settings->getDayStart()) {
            // push starting point to last "slot" of hour BEFORE morning aggregation
            $morning_aggr = ($this->user_settings->getDayStart() - 1) * 60 + (60 - $this->raster);
        } else {
            $morning_aggr = 0;
        }

        return $morning_aggr;
    }

    protected function addAppointmentLink(ilDateTime $date): void
    {
        $new_app_url = $this->ctrl->getLinkTargetByClass('ilcalendarappointmentgui', 'add');

        $this->tpl->setCurrentBlock("new_app");
        //$this->tpl->setVariable('NEW_APP_LINK',$new_app_url);
        $this->tpl->setVariable('NEW_APP_GLYPH', $this->ui_renderer->render(
            $this->ui_factory->symbol()->glyph()->add($new_app_url)
        ));
        // $this->tpl->setVariable('NEW_APP_ALT',$this->lng->txt('cal_new_app'));
        $this->tpl->parseCurrentBlock();

        // }

        $this->ctrl->clearParametersByClass('ilcalendarappointmentgui');
    }

    protected function setUpCalendar(): void
    {
        $bkid = $this->initBookingUserFromQuery();
        if ($bkid) {
            $this->user_id = $bkid;
            $this->disable_empty = true;
            $this->no_add = true;
        } elseif ($this->user->getId() == ANONYMOUS_USER_ID) {
            $this->disable_empty = false;
            $this->no_add = true;
        } else {
            $this->disable_empty = false;
            $this->no_add = false;
        }
    }

    protected function addHeaderDate($date, $num_apps): void
    {
        $date_info = $date->get(IL_CAL_FKT_GETDATE, '', 'UTC');
        $dayname = ilCalendarUtil::_numericDayToString((int) $date->get(IL_CAL_FKT_DATE, 'w'), false);
        $daydate = $dayname . ' ' . $date_info['mday'] . '.';

        if (!$this->disable_empty || $num_apps[$date->get(IL_CAL_DATE)] > 0) {
            $link = $this->ctrl->getLinkTargetByClass('ilcalendardaygui', '');
            $this->ctrl->clearParametersByClass('ilcalendardaygui');

            $this->tpl->setCurrentBlock("day_view_link");
            $this->tpl->setVariable('HEADER_DATE', $daydate);
            $this->tpl->setVariable('DAY_VIEW_LINK', $link);
            $this->tpl->parseCurrentBlock();
        } else {
            $this->tpl->setCurrentBlock("day_view_no_link");
            $this->tpl->setVariable('HEADER_DATE', $daydate);
            $this->tpl->parseCurrentBlock();
        }
    }

    protected function addFullDayEvents($all_fullday): void
    {
        $counter = 0;
        foreach ($all_fullday as $daily_apps) {
            foreach ($daily_apps as $event) {
                if ($event['fullday']) {
                    $this->showFulldayAppointment($event);
                }
            }
            $this->tpl->setCurrentBlock('f_day_row');
            $this->tpl->setVariable('COLSPAN', max($this->colspans[$counter], 1));
            $this->tpl->parseCurrentBlock();
            $counter++;
        }
        $this->tpl->setCurrentBlock('fullday_apps');
        $this->tpl->setVariable('TXT_F_DAY', $this->lng->txt("cal_all_day"));
        $this->tpl->parseCurrentBlock();
    }

    protected function addTimedEvents(array $hours, int $morning_aggr, int $evening_aggr): void
    {
        $new_link_counter = 0;
        $day_id_counter = 0;
        foreach ($hours as $num_hour => $hours_per_day) {
            $first = true;
            foreach ($hours_per_day as $num_day => $hour) {
                #ADD the hours in the left side of the grid.
                if ($first) {
                    if (!($num_hour % 60) || ($num_hour == $morning_aggr && $morning_aggr) ||
                        ($num_hour == $evening_aggr && $evening_aggr)) {
                        $first = false;

                        // aggregation rows
                        if (($num_hour == $morning_aggr && $morning_aggr) ||
                            ($num_hour == $evening_aggr && $evening_aggr)) {
                            $this->tpl->setVariable('TIME_ROWSPAN', 1);
                        } // rastered hour
                        else {
                            $this->tpl->setVariable('TIME_ROWSPAN', 60 / $this->raster);
                        }

                        $this->tpl->setCurrentBlock('time_txt');

                        $this->tpl->setVariable('TIME', $hour['txt']);
                        $this->tpl->parseCurrentBlock();
                    }
                }

                foreach ($hour['apps_start'] as $app) {
                    $this->showAppointment($app);
                }
                $num_apps = $hour['apps_num'];
                $colspan = max($this->colspans[$num_day], 1);

                // Show new apointment link
                if (!$hour['apps_num'] && !$this->no_add) {
                    $this->tpl->setCurrentBlock('new_app_link');

                    $this->ctrl->clearParameterByClass('ilcalendarappointmentgui', 'app_id');

                    $this->ctrl->setParameterByClass(
                        'ilcalendarappointmentgui',
                        'idate',
                        $this->weekdays[$num_day]->get(IL_CAL_DATE)
                    );
                    $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'seed', $this->seed->get(IL_CAL_DATE));
                    $this->ctrl->setParameterByClass('ilcalendarappointmentgui', 'hour', floor($num_hour / 60));

                    //todo:it could be nice use also ranges of 15 min to create events.
                    $new_app_url = $this->ctrl->getLinkTargetByClass('ilcalendarappointmentgui', 'add');
                    $this->tpl->setVariable(
                        "DAY_NEW_APP_LINK",
                        $this->ui_renderer->render($this->ui_factory->symbol()->glyph()->add($new_app_url))
                    );
                    $this->tpl->setVariable('DAY_NEW_ID', ++$new_link_counter);
                    $this->tpl->parseCurrentBlock();
                }

                for ($i = $colspan; $i > $hour['apps_num']; $i--) {
                    $this->tpl->setCurrentBlock('day_cell');

                    // last "slot" of hour needs border
                    $empty_border = '';
                    if ($num_hour % 60 == 60 - $this->raster ||
                        ($num_hour == $morning_aggr && $morning_aggr) ||
                        ($num_hour == $evening_aggr && $evening_aggr)) {
                        $empty_border = ' calempty_border';
                    }

                    $this->tpl->setVariable('TD_CLASS', 'calempty createhover' . $empty_border);

                    $this->tpl->setVariable('DAY_ID', ++$day_id_counter);
                    $this->tpl->setVariable('TD_ROWSPAN', 1);
                    $this->tpl->parseCurrentBlock();
                }
            }
            $this->tpl->touchBlock('time_row');
        }
    }

    /**
     * @param ilCalendarEntry $a_event
     * @return string
     */
    protected function getAppointmentTimeString(ilCalendarEntry $a_event): string
    {
        $time = "";
        switch ($this->user_settings->getTimeFormat()) {
            case ilCalendarSettings::TIME_FORMAT_24:
                $time = $a_event->getStart()->get(IL_CAL_FKT_DATE, 'H:i', $this->timezone);
                break;

            case ilCalendarSettings::TIME_FORMAT_12:
                $time = $a_event->getStart()->get(IL_CAL_FKT_DATE, 'h:ia', $this->timezone);
                break;
        }

        return $time;
    }
}
