<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

class ilCronJobResult
{
    final public const STATUS_INVALID_CONFIGURATION = 1;
    final public const STATUS_NO_ACTION = 2;
    final public const STATUS_OK = 3;
    final public const STATUS_CRASHED = 4;
    final public const STATUS_RESET = 5;
    final public const STATUS_FAIL = 6;

    final public const CODE_NO_RESULT = 'job_no_result';
    final public const CODE_MANUAL_RESET = 'job_manual_reset';
    final public const CODE_SUPPOSED_CRASH = 'job_auto_deactivation_time_limit';

    protected int $status = self::STATUS_NO_ACTION;
    protected string $message = '';
    protected ?string $code = null;
    protected string $duration = '0';

    /**
     * @return string[]
     */
    public static function getCoreCodes(): array
    {
        return [
            self::CODE_NO_RESULT,
            self::CODE_MANUAL_RESET,
            self::CODE_SUPPOSED_CRASH,
        ];
    }

    public function getStatus(): int
    {
        return $this->status;
    }

    public function setStatus(int $a_value): void
    {
        if (!in_array($a_value, $this->getValidStatus(), true)) {
            throw new InvalidArgumentException(sprintf(
                'The passed status "%s" is not valid, must be of one of: %s',
                $a_value,
                implode(', ', $this->getValidStatus())
            ));
        }

        $this->status = $a_value;
    }

    /**
     * @return int[]
     */
    protected function getValidStatus(): array
    {
        return [
            self::STATUS_INVALID_CONFIGURATION,
            self::STATUS_NO_ACTION,
            self::STATUS_OK,
            self::STATUS_CRASHED,
            self::STATUS_FAIL,
            self::STATUS_RESET,
        ];
    }

    public function getMessage(): string
    {
        return $this->message;
    }

    public function setMessage(string $value): void
    {
        $value = trim($value);
        if (ilStr::strLen($value) > 400) {
            throw new InvalidArgumentException("The message must not exceed 400 characters");
        }

        $this->message = $value;
    }

    public function getCode(): ?string
    {
        return $this->code;
    }

    public function setCode(string $a_value): void
    {
        if (!in_array($a_value, self::getCoreCodes(), true)) {
            throw new InvalidArgumentException(sprintf(
                'The passed code "%s" is not valid, must be of one of: %s',
                $a_value,
                implode(', ', self::getCoreCodes())
            ));
        }

        $this->code = $a_value;
    }

    public function getDuration(): float
    {
        return (float) $this->duration;
    }

    public function setDuration(float $a_value): void
    {
        $this->duration = number_format($a_value, 3, '.', '');
    }
}
